-- CONTENTS
-- TARDIS exterior functions
-- drwho_tardis:register_tardis_exterior() API
-- TARDIS Node Definitions (using API)

local S = core.get_translator("drwho_tardis")
local data = core.get_mod_storage()

-- IMPORTANT:
-- Old Tardises will not be affected by the changes in values.
-- They will need to be manually moved by the player, after they have removed valuable
-- things from it. They can choose what interior design to place in it as well. Use /rebuild_tardis

-- TARDIS EXTERIOR FUNCTIONS
local function tardis_on_place(pos, placer, itemstack, pointed_thing)
	if placer:get_player_name() == "" then return else
		local meta = core.get_meta(pos)
		local name = placer:get_player_name()
		local user = _drwho_tardis.get_user(name) -- This will be first run, and will create init all user data automatically
		local timer = core.get_node_timer(pos)
		if user.has_a_tardis == false then
			user.has_a_tardis = true
			user.out_pos = pos -- set exterior position
			_drwho_tardis.save_user(name, user) -- save user data
			local room = core.settings:get('drwho_tardis.default_interior')
			if _drwho_tardis.rooms[room] then room = room else room = "default" end
			local _, _, exit_door_pos = _drwho_tardis:place_tardis_interior(name, room, pos) 

			_drwho_tardis:init_interior_door(name, exit_door_pos)

			meta:set_string("id", name) -- set exterior id
			timer:start(0.2)
		else -- they already have a tardis 
			core.set_node(pos, {name = "air"}) 
			core.add_item(pos, "drwho_tardis:tardis_blue")
			core.chat_send_player(name, S("You can only have one Tardis!"))
		end
	end
end

-- Check if there are any players or mobs entering a TARDIS and teleport them
local function tardis_timer(pos)
	local objs = core.get_objects_inside_radius(pos, 0.9)
	--for idx,obj in pairs(objs) do -- loop incase there are multiple objs there at once
	local obj = objs[1]
		if obj == nil then return true else --only do code if there is something there
			local meta = core.get_meta(pos)
			local id = meta:get_string("id")
			local user = _drwho_tardis.get_user(id) -- get user data
			local go_pos = table.copy(user.in_pos)
			local node = core.get_node_or_nil(go_pos)
			
			-- load area using VoxelManip if necessary
			local vm_found_true
			if node == nil then
				_drwho_tardis.log("Loading using VoxelManip...")
				local c_in_door = core.get_content_id("drwho_tardis:in_door") -- content ID for the door
				local vm = core.get_voxel_manip()
				local pos1 = vector.subtract(go_pos, vector.new(3,3,3))
				local pos2 = vector.add(go_pos, vector.new(3,3,3))
				local emin, emax = vm:read_from_map(pos1, pos2)
				local vm_data = vm:get_data()
				local a = VoxelArea:new{
					MinEdge = emin,
					MaxEdge = emax
				}

				-- Get ref to node and check if it's an in_door
				local vi = a:index(go_pos.x, go_pos.y, go_pos.z)
				if vm_data[vi] == c_in_door then
					vm_found_true = true
					_drwho_tardis.log('vm_found true')
				else
					vm_found_true = false
					_drwho_tardis.log('vm_found false')
				end
				_drwho_tardis.log('found: '..dump(vm_data[vi]))
				_drwho_tardis.log('in_door: '..c_in_door)
			end

			-- If Interior Exit Door does not exist, do not let player enter
			if vm_found_true == false or node ~= nil and node.name ~= "drwho_tardis:in_door" then
				if obj:is_player() then
					core.chat_send_player(obj:get_player_name(), S("Your Tardis interior exit door is missing from your console room, so you cannot be teleported into it. Try linking another door or rebuilding your Tardis ( /rebuild_tardis )."))
				end
				return true
			end

			-- Teleport player (or mob) into the Tardis
			node = core.get_node(go_pos) -- get node again, in case we used VM and didn't have local data on it.
			if obj:is_player() then -- if obj is a player
				local pmeta = obj:get_meta()
				local name = obj:get_player_name()
				local been_inside = user.been_inside_tardis
				go_pos = drwho_tardis.shift_1_node_from_4dir(go_pos, node.param2)
				obj:set_look_horizontal( math.rad( drwho_tardis.fourDir_to_dir_flipped(node.param2) ))
				obj:set_look_vertical( math.rad( 0 ))
				obj:set_pos(go_pos)
				_drwho_tardis.log('player '..name..' entered a Tardis from '..drwho_tardis.serialize_pos(pos))
				_drwho_tardis.unlock_award(name, "drwho_tardis:go_inside_tardis") -- unlock award
				if not been_inside then
					user.been_inside_tardis = true
				end
				pmeta:set_string("id", id)
				_drwho_tardis.save_user(id, user) -- Save user data
			else -- if obj is not a player (probably a mob)
				go_pos = drwho_tardis.shift_1_node_from_4dir(go_pos, node.param2)
				obj:set_look_horizontal( math.rad( drwho_tardis.fourDir_to_dir_flipped(node.param2) ))
				obj:set_look_vertical( math.rad( 0 ))
				obj:set_pos(go_pos)
				_drwho_tardis.log('A non-player entity entered a Tardis from '..drwho_tardis.serialize_pos(pos))
			end
		end
		return true
	--end
end

-- See API.md for more information about this function
function drwho_tardis:register_tardis_exterior(def) 
	-- REQUIRED: {name, texture}
	-- name MUST be unique
	local name = "_"..def.name -- e.g. "_blue" if def.name == "blue"
	
	if def.name == "" then 
		name = ""  -- ONLY FOR THE DEFAULT BLUE TARDIS
		def.name = "blue"
	end 
	local texture = def.texture
	-- OPTIONAL: {light, description, drawtype, title, groups}
	local light = def.light or 10 -- In case one doesn't want it to glow
	local description = def.description or S("Tardis") -- allow for custom description titles
	local drawtype = def.drawtype or "mesh" -- default to mesh
	local title = def.title or def.name -- Title to be shown on exterior selector
	local groups = def.groups or {not_in_creative_inventory = 1, tardis = 1}

	-- Register normal TARDIS
	core.register_node("drwho_tardis:tardis"..name, {
		description = description,
		tiles = {texture},
		use_texture_alpha = "clip",
		drawtype = drawtype,
		mesh = "tardis.obj", -- Only used when drawtype == "mesh"
		node_box = {type = "fixed", fixed = { { -0.5, -0.5, -0.5, 0.5, 1.5, 0.5 } }}, -- Only used when drawtype == "nodebox"
		selection_box = {type = "fixed", fixed = { { -0.5, -0.5, -0.5, 0.5, 1.5, 0.5 } }},
		collision_box = {type = "fixed", fixed = { { 0.48, -0.5,-0.5,  0.5,  1.5, 0.5}, {-0.5 , -0.5, 0.48, 0.48, 1.5, 0.5}, {-0.5,  -0.5,-0.5 ,-0.48, 1.5, 0.5}, { -0.8,-0.6,-0.8,0.8,-0.48, 0.8} }},
		light_source = light,
		groups = groups,
		after_place_node = tardis_on_place,
		diggable = false,
		on_blast = function() end,
		on_timer = tardis_timer
	})
	-- Register locked TARDIS
	core.register_node("drwho_tardis:tardis"..name.."_locked", {
			description = description.." (locked)",
			tiles = {texture},
			use_texture_alpha = "clip",
			drawtype = drawtype,
			mesh = "tardis.obj", -- Only used when drawtype == "mesh"
			node_box = {type = "fixed", fixed = { { -0.5, -0.5, -0.5, 0.5, 1.5, 0.5 } }}, -- Only used when drawtype == "nodebox"
			selection_box = {type = "fixed", fixed = { { -0.5, -0.5, -0.5, 0.5, 1.5, 0.5 } }},
			collision_box = {type = "fixed", fixed = { { -0.5, -0.5, -0.5, 0.5, 1.5, 0.5 } }},
			light_source = 10,
			groups = {not_in_creative_inventory = 1, tardis_locked = 1},
			diggable = false,
			on_blast = function() end,
	})

	-- Add it to the exterior list
	local ext_list = core.deserialize(data:get_string("ext_list")) or {}
	ext_list[#ext_list+1] = {
		itemstring = "drwho_tardis:tardis"..name,
		name = def.name,
		title = title,
		author = def.author,
		license = def.license or "CC BY-SA 4.0", -- defaults so we don't need to specify each time
		texture = texture
	}
	data:set_string("ext_list", core.serialize(ext_list))
end