-- CONTENTS
-- Achievements using `awards` for MTG and VL; `rp_achievements` for RP.
--      register travel_tardis trigger for mtg
-- Functions:
--      _drwho_tardis.register_award(name, def)
--      _drwho_tardis.unlock_award(player_name, award_name)
--      _drwho_tardis.migrate_times_travelled(name, user)
--      _drwho_tardis.increment_award_travel_tardis(name, config) 
--      _drwho_tardis.check_drwho_awards(name, user)
-- All award definitions

local S = core.get_translator("drwho_tardis")

local awards_enabled = core.get_modpath("awards")
local is_mtg = _drwho_tardis.GAMETYPE == "mtg"
local is_mcl = _drwho_tardis.GAMETYPE == "mcl"
local is_rp = _drwho_tardis.GAMETYPE == "rp"

-- All the functions meant to be used outside this file can be called without checking
-- the gametype or if `awards` is enabled, as that is done within each function.

-- Easy multi-game support: Register Award
function _drwho_tardis.register_award(name, def)
    if is_mtg and awards_enabled then
        awards.register_award(name, {
            title = def.title,
            description = def.description,
            difficulty = def.difficulty,
            trigger = def.trigger,
            requires = def.requires,
            icon = def.icon,
        })
    elseif is_mcl then
        awards.register_achievement(name, {
            title = def.title,
            description = def.description,
            trigger = def.trigger,
            --requires = def.requires, -- does not exist in VoxeLibre version
            icon = def.icon,
            type = def.type,
            group = def.group,
            secret = def.secret,
        })
    elseif is_rp then
        if def.in_rp == false then return end -- don't include some awards in rp
        if not def.trigger then def.trigger = {} end -- avoid bug
        achievements.register_achievement(name, {
            title = def.title,
            description = def.description,
            icon = def.icon,
            times = def.trigger.target or nil,
            craftitem = def.trigger.type == "craft" and def.trigger.item or nil,
            dignode = def.trigger.type == "dig" and def.trigger.node or nil,
            placenode = def.trigger.type == "place" and def.trigger.node or nil,
            difficulty = def.difficulty_rp or def.difficulty, 
        })
    end
end

-- Easy multi-game support: Unlock Award
function _drwho_tardis.unlock_award(player_name, award_name)
    if is_mtg and awards_enabled or is_mcl then
        awards.unlock(player_name, award_name)
    elseif is_rp then
        local player = core.get_player_by_name(player_name)
        achievements.trigger_achievement(player, award_name)
    end
end

-- Register Times Travelled in Tardis award trigger in mtg
if is_mtg and awards_enabled then
    awards.register_trigger("travel_tardis", {
        type = "counted",
        progress = "@1/@2 times travelled",
        auto_description = { "Travel in your TARDIS", "Travel in your TARDIS @1 times" },
    })
end

-- Switch the old userdata field into the new award data one (only used in mtg)
-- old: user.times_travelled_in_tardis
-- new: user.times_travelled_in_tardis AND data internal to awards mod in mtg
-- userdata must pass through the function - it does not get or save it itself.
function _drwho_tardis.migrate_times_travelled(name, user)
    if is_mtg and awards_enabled then
        if not user.is_times_travelled_migrated then
            local times = user.times_travelled_in_tardis
            while times >= 1 do
                _drwho_tardis.increment_award_travel_tardis(name, {migrating="yes"})
                times = times - 1
            end
            user.is_times_travelled_migrated = true -- don't let this function run again
            _drwho_tardis.log("migrated "..name.."'s times travelled in Tardis to the mtg awards data")
        end
    end
    return user
end

-- Add one to tardis travel counter and check for awards
function _drwho_tardis.increment_award_travel_tardis(name, config) 
    if not config then config = {} config.send_msg = false end

    local user = _drwho_tardis.get_user(name)
    local player = core.get_player_by_name(name)

    if not config.migrating or not config.migrating == "yes" then
        user = _drwho_tardis.migrate_times_travelled(name, user)
    end

    -- Increment counter and give award
    user.times_travelled_in_tardis = user.times_travelled_in_tardis + 1
    if is_mcl or is_rp then 
        _drwho_tardis.check_drwho_awards(name, user)
    elseif is_mtg and awards_enabled then
        awards.notify_travel_tardis(player)
    end

    if config.send_msg then
        core.chat_send_player(name, S("Times travelled in TARDIS: @1", user.times_travelled_in_tardis))
    end
    _drwho_tardis.save_user(name, user)
end

-- Unlock awards based on counter
-- pass user data into the function
function _drwho_tardis.check_drwho_awards(name, user)
    if user.times_travelled_in_tardis >= 1 then
        _drwho_tardis.unlock_award(name, "drwho_tardis:travel_in_tardis") -- unlock award (travel 1)
    end
    if user.times_travelled_in_tardis >= 10 then
        _drwho_tardis.unlock_award(name, "drwho_tardis:travel_in_tardis_10") -- unlock award (travel 10)
    end
    if user.times_travelled_in_tardis >= 50 then
        _drwho_tardis.unlock_award(name, "drwho_tardis:travel_in_tardis_50") -- unlock award (travel 50)
    end
    if user.times_travelled_in_tardis >= 500 then
        _drwho_tardis.unlock_award(name, "drwho_tardis:travel_in_tardis_500") -- unlock award (travel 500)
    end
end

-----------------------
-- AWARD DEFINITIONS --
-----------------------

_drwho_tardis.register_award("drwho_tardis:get_sonic_screwdriver", {
    title = S("It's a Sonic!"),
    description = S("Obtain a Sonic Screwdriver"),
    icon = "tardis_achievement_sonic.png",
    difficulty = 2,
    difficulty_rp = 6,
    type = "Advancement",
    group = "Doctor Who",
    -- trigger is in the Functions Console Unit line ~185
})

_drwho_tardis.register_award("drwho_tardis:craft_tardis", {
    title = S("A Gallifreyan Time Machine"),
    description = S("Craft a TARDIS"),
    icon = "tardis_achievement_craft.png",
    difficulty = 2.5,
    difficulty_rp = 7.5,
    trigger = {
        type = "craft",
        item = "drwho_tardis:tardis", -- item, alias, or group
        target = 1,
    },
    type = "Advancement",
    group = "Doctor Who",
})

_drwho_tardis.register_award("drwho_tardis:place_tardis", {
	title = S("The Journey Begins"),
	description = S("Place down a TARDIS"),
    icon = "tardis_achievement_place.png",
    difficulty = 2.5,
    difficulty_rp = 7.5,
    trigger = {
        type = "place",
        node = "drwho_tardis:tardis", -- item, alias, or group
        target = 1,
    },
    type = "Advancement",
    group = "Doctor Who",
})

_drwho_tardis.register_award("drwho_tardis:command_summon_tardis", {
    title = S("Come Here!"),
    description = S("Summon your TARDIS to your position."),
    icon = "tardis_achievement_summon.png",
    difficulty = 2.6,
    difficulty_rp = 7.6,
    type = "Advancement",
    group = "Doctor Who",
    -- trigger is awarded from chat or sonic command: init.lua~180 and sonic.lua~170. 
})

_drwho_tardis.register_award("drwho_tardis:go_inside_tardis", {
    title = S("It's Bigger on the Inside!"),
    description = S("Walk inside a TARDIS"),
    icon = "tardis_achievement_go_inside.png",
    difficulty = 2,
    difficulty_rp = 7,
    type = "Advancement",
    group = "Doctor Who",
    --trigger is in exterior.lua in function tardis_timer around line 55
})

_drwho_tardis.register_award("drwho_tardis:tardis_attack", {
    title = S("Threat Nullified."),
    description = S("Attack a nearby player using the TARDIS defense system."),
    icon = "tardis_achievement_attack.png",
    difficulty = 2,
    difficulty_rp = 8,
    type = "Challenge",
    group = "Doctor Who",
    --trigger is in console.lua in function fields.attack around line 240
})

-- Travel in your TARDIS (1, 10, 50, 500)
_drwho_tardis.register_award("drwho_tardis:travel_in_tardis", {
    title = S("Allons-y!"),
    description = S("Travel using your TARDIS"),
    icon = "tardis_achievement_travel.png",
    difficulty = 2,
    difficulty_rp = 7.6,
    type = "Advancement",
    group = "Doctor Who",
    trigger = {
        type = "travel_tardis",
        target = 1,
    },
})

_drwho_tardis.register_award("drwho_tardis:travel_in_tardis_10", {
    title = S("Geronimo!"),
    description = S("Travel using your TARDIS @1 times", 10),
    requires = {"drwho_tardis:travel_in_tardis",},
    icon = "tardis_achievement_10.png",
    difficulty = 0.5,
    difficulty_rp = 7.7,
    type = "Advancement",
    group = "Doctor Who",
    trigger = {
        type = "travel_tardis",
        target = 10,
    },})

_drwho_tardis.register_award("drwho_tardis:travel_in_tardis_50", {
    title = S("Just Passing By"),
    description = S("Travel using your TARDIS @1 times", 50),
    requires = {"drwho_tardis:travel_in_tardis_10",},
    icon = "tardis_achievement_50.png",
    difficulty = 0.5,
    difficulty_rp = 7.8,
    type = "Advancement",
    group = "Doctor Who",
    trigger = {
        type = "travel_tardis",
        target = 50,
    },
})

_drwho_tardis.register_award("drwho_tardis:travel_in_tardis_500", {
    title = S("My Way to Travel"),
    description = S("Travel using your TARDIS @1 times", 500),
    requires = {"drwho_tardis:travel_in_tardis_50",},
    icon = "tardis_achievement_500.png",
    difficulty = 0.5,
    difficulty_rp = 7.9,
    type = "Advancement",
    group = "Doctor Who",
    trigger = {
        type = "travel_tardis",
        target = 500,
    },
})

-- Defeat Drwho mobs awards
if core.get_modpath("mobs") then
    _drwho_tardis.register_award("drwho_tardis:defeat_dalek", {
        title = S("Dalek Destroyed"),
        description = S("Defeat a Dalek using Earth technology and weapons."),
        icon = "tardis_achievement_dalek.png",
        difficulty = 0.8,
        difficulty_rp = 4.5,
        type = "Challenge",
        group = "Doctor Who",
        -- Triggered in on_die
    })
    _drwho_tardis.register_award("drwho_tardis:defeat_cyberman", {
        title = S("The Age of Steel"),
        description = S("Defeat a Cyberman using Earth technology and weapons."),
        icon = "tardis_achievement_cyberman.png",
        difficulty = 0.8,
        difficulty_rp = 4.5,
        type = "Challenge",
        group = "Doctor Who",
        -- Triggered in on_die
    })
end

-- Biscuit awards (1, 50, 100)
_drwho_tardis.register_award("drwho_tardis:eat_biscuit_1", {
    title = S("Just a Normal Biscuit"),
    description = S("Eat one TARDIS Biscuit. Hint: Look at the Functions Console Unit"),
    icon = "tardis_biscuit.png",
    difficulty = 0.05,
    in_rp = false,
    trigger = {
        type   = "eat",
        item   = "drwho_tardis:biscuit", -- item, alias, or group
        target = 1,
    },
    type = "Advancement",
    group = "Overworld",
})

_drwho_tardis.register_award("drwho_tardis:eat_biscuit_50", {
    title = S("They Taste Really Good!"),
    description = S("Eat fifty TARDIS Biscuits in total."),
    requires = {"drwho_tardis:eat_biscuit_1",},
    difficulty = 0.05,
    in_rp = false,
    icon = "tardis_biscuit.png",
    trigger = {
        type   = "eat",
        item   = "drwho_tardis:biscuit", -- item, alias, or group
        target = 50,
    },
    type = "Advancement",
    group = "Overworld",
})

_drwho_tardis.register_award("drwho_tardis:eat_biscuit_100", {
    title = S("My Favourite Snack"),
    description = S("Eat one hundred TARDIS Biscuits in total."),
    requires = {"drwho_tardis:eat_biscuit_50",},
    difficulty = 0.05,
    in_rp = false,
    icon = "tardis_biscuit.png",
    trigger = {
        type   = "eat",
        item   = "drwho_tardis:biscuit", -- item, alias, or group
        target = 100,
    },
    type = "Advancement",
    group = "Overworld",
})