
local hud_ids = {}
local sound_intervals = {}
local sound_timers = {}
local update_interval = 0.1 -- Update interval in seconds
local timer = 0
local tracked_entity = {}
local hud_active = {}
local hud_update_interval = 0.1 -- HUD update interval in seconds
local max_sound_interval = 3 -- Maximum sound interval in seconds
local min_sound_interval = 0.35 -- Minimum sound interval in seconds
local max_pitch = 2 -- Maximum pitch
local min_pitch = 0.5 -- Minimum pitch
local max_volume = 1.0 -- Maximum volume
local min_volume = 0.2 -- Minimum volume

local hud_position = {x = 0.76, y = 0.63} -- Positioning the HUD on the right side of the screen
local hud_scale = 0.75 -- HUD scale factor

local function play_sounds(player)
    local player_name = player:get_player_name()
    if sound_timers[player_name] then
        if sound_intervals[player_name] then
            minetest.after(sound_intervals[player_name], function()
                play_sounds(player)
            end)
        end
        return
    end
    sound_timers[player_name] = true

    local function play_sound()
        if not hud_active[player_name] then
            sound_timers[player_name] = nil
            return
        end
        if sound_intervals[player_name] then
            local pitch = sound_intervals[player_name .. "_pitch"] or 1.0
            local volume = sound_intervals[player_name .. "_volume"] or 1.0
            minetest.sound_play("distance_sound", {to_player = player_name, pitch = pitch, gain = volume})
            minetest.after(sound_intervals[player_name], play_sound)
        end
    end
    play_sound()
end

-- Function to set player size
local function set_player_size(player, size)
    player:set_properties({
        visual_size = {x = size, y = size}
    })
end

-- Call this function to set the size for a player
minetest.register_on_joinplayer(function(player)
    set_player_size(player, 1.5) -- Example size value
end)















local function update_entity_compass(player)
    if not player or not player:is_player() then return end
    local player_name = player:get_player_name()
    local pos = player:get_pos()
    local radius = 150
    local screen_radius = 0.2

    -- Clear existing HUD elements
    for _, id in ipairs(hud_ids[player_name] or {}) do
        player:hud_remove(id)
    end
    hud_ids[player_name] = {}

    if not hud_active[player_name] then return end

    table.insert(hud_ids[player_name], player:hud_add({
        hud_elem_type = "image",
        position = hud_position,
        scale = {x = 28.8 * hud_scale, y = 28.8 * hud_scale},
        text = "visual_device.png^black_background.png"
    }))
    table.insert(hud_ids[player_name], player:hud_add({
        hud_elem_type = "image",
        position = hud_position,
        scale = {x = 5 * hud_scale, y = 5 * hud_scale},
        text = "green_dot.png"
    }))

    if tracked_entity[player_name] then
        table.insert(hud_ids[player_name], player:hud_add({
            hud_elem_type = "text",
            position = {x = hud_position.x, y = hud_position.y - 0.33},
            text = "Filter [" .. tracked_entity[player_name] .. "]",
            number = 0xFFFFFF,
        }))
    end

    local objects = minetest.get_objects_inside_radius(pos, radius)
    local object_count = 0
    local closest_distance = radius
    local closest_obj = nil

    for _, obj in ipairs(objects) do
        local is_player = obj:is_player()
        local is_item = obj:get_luaentity() and obj:get_luaentity().name == "__builtin:item"
        local obj_name
        if is_item then
            local itemstack = ItemStack(obj:get_luaentity().itemstring)
            obj_name = "Dropped " .. itemstack:get_description()
        else
            local entity_full_name = is_player and "[Player]" or (obj:get_entity_name() or "Entity")
            obj_name = entity_full_name:gsub(".*:", ""):gsub("_", " ")
        end

        if obj ~= player and tracked_entity[player_name] and (tracked_entity[player_name] == "all" or string.find(obj_name, tracked_entity[player_name])) then
            local obj_pos = obj:get_pos()
            local rel_pos = vector.subtract(obj_pos, pos)

            local yaw = player:get_look_horizontal()
            local rel_x = (-rel_pos.x * math.cos(yaw) - rel_pos.z * math.sin(yaw)) / radius * screen_radius
            local rel_z = (rel_pos.x * math.sin(yaw) - rel_pos.z * math.cos(yaw)) / radius * screen_radius

            local distance = vector.distance(pos, obj_pos)
            if distance < closest_distance then
                closest_distance = distance
                closest_obj = obj
            end
            local height_diff = rel_pos.y

            table.insert(hud_ids[player_name], player:hud_add({
                hud_elem_type = "image",
                position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z},
                scale = {x = 4 * hud_scale, y = 4 * hud_scale},
                text = is_player and "blue_dot.png" or (is_item and "yellow_dot.png" or "red_dot.png")
            }))
            if is_player then
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.08 * hud_scale},
                    text = "[Player]",
                    number = 0x00FFFF,
                }))
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.06 * hud_scale},
                    text = "(" .. obj:get_player_name() .. ")",
                    number = 0x00FFFF,
                }))
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.04 * hud_scale},
                    text = "D: " .. string.format("%.1f", distance),
                    number = 0x00FFFF,
                }))
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.02 * hud_scale},
                    text = "H: " .. string.format("%.1f", height_diff),
                    number = 0x00FFFF,
                }))
            else
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.06 * hud_scale},
                    text = obj_name,
                    number = 0x00FFFF,
                }))
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.04 * hud_scale},
                    text = "D: " .. string.format("%.1f", distance),
                    number = 0x00FFFF,
                }))
                table.insert(hud_ids[player_name], player:hud_add({
                    hud_elem_type = "text",
                    position = {x = hud_position.x + rel_x, y = hud_position.y + rel_z - 0.02 * hud_scale},
                    text = "H: " .. string.format("%.1f", height_diff),
                    number = 0x00FFFF,
                }))
            end
            object_count = object_count + 1
        end
    end

    if closest_obj then
        local closest_pos = closest_obj:get_pos()
        local distance = vector.distance(pos, closest_pos)

        local distance_interval = 3 * (distance / radius)
        distance_interval = math.max(min_sound_interval, math.min(max_sound_interval, distance_interval))
        sound_intervals[player_name] = distance_interval
        sound_intervals[player_name .. "_pitch"] = min_pitch + (max_pitch - min_pitch) * (1 - (distance / radius))
        sound_intervals[player_name .. "_volume"] = min_volume + (max_volume - min_volume) * (1 - (distance / radius))
    end

    -- Add object count display near the top corner
    table.insert(hud_ids[player_name], player:hud_add({
        hud_elem_type = "text",
        position = {x = hud_position.x - 0.0925, y = hud_position.y - 0.33},
        text = "Currently " .. object_count .. " objects",
        number = 0xFFFFFF,
    }))

    -- Add player name display near the bottom of the screen
    table.insert(hud_ids[player_name], player:hud_add({
        hud_elem_type = "text",
        position = {x = hud_position.x + 0.0925, y = hud_position.y - 0.33},
        text = "User: " .. player_name,
        number = 0xFFFFFF,
    }))
end






















minetest.register_globalstep(function(dtime)
    timer = timer + dtime
    if timer >= hud_update_interval then
        for _, player in ipairs(minetest.get_connected_players()) do
            local player_name = player:get_player_name()
            local wielded_item = player:get_wielded_item():get_name()
            if wielded_item == "tracking_tablet:entity_compass" then
                if not hud_active[player_name] then
                    hud_active[player_name] = true
                    play_sounds(player) -- Activate sounds when the item is held
                end
                update_entity_compass(player)
            else
                if hud_active[player_name] then
                    hud_active[player_name] = nil
                    -- Clear HUD elements
                    for _, id in ipairs(hud_ids[player_name] or {}) do
                        player:hud_remove(id)
                    end
                    hud_ids[player_name] = {}
                    sound_timers[player_name] = nil -- Stop the sound timer when the HUD is turned off
                end
            end
        end
        timer = 0
    end
end)

minetest.register_craftitem("tracking_tablet:entity_compass", {
    description = "Entity Tracking Tablet\n\nHow do you use it?:\n- You can use the tablet by simply holding it. to open the search filter and start tracking, simply rightclick the air!\n- This tablet allows you to watch the movments of nearby entitys, players, and even drops!",
    inventory_image = "visual_device.png^[opacity:128", -- This adds a transparent wield texture
    on_use = function(itemstack, user, pointed_thing)
        -- The HUD is now automatically handled in the globalstep function based on the wielded item
    end,
    on_secondary_use = function(itemstack, user, pointed_thing)
        local player_name = user:get_player_name()
        if not hud_active[player_name] then
            minetest.chat_send_player(player_name, "You need to be holding the tracker to identify and track your entity.")
        else
            -- Show formspec to input entity name
            minetest.show_formspec(player_name, "tracking_tablet:entity_compass_formspec", "size[4,2]" ..
                "field[1,1;3,1;entity_name;Input search filter:;]" ..
                "button_exit[1,1.5;2,1;save;Search]")
            minetest.register_on_player_receive_fields(function(player, formname, fields)
                if formname == "tracking_tablet:entity_compass_formspec" then
                    if fields.entity_name then
                        tracked_entity[player:get_player_name()] = fields.entity_name
                        update_entity_compass(player) -- Immediately update to start tracking
                        play_sounds(player) -- Immediately start sound intervals based on distance
                    end
                end
            end)
        end
    end,
})

-- Function to set HUD scale
local function set_hud_scale(player_name, scale)
    hud_scale = scale
    if hud_active[player_name] then
        update_entity_compass(minetest.get_player_by_name(player_name))
    end
end

-- Register a chat command to set HUD scale
minetest.register_chatcommand("set_hud_scale", {
    params = "<scale>",
    description = "Set the scale of the device HUD",
    func = function(name, param)
        local scale = tonumber(param)
        if scale and scale > 0 then
            set_hud_scale(name, scale)
            minetest.chat_send_player(name, "Device HUD scale set to " .. scale)
        else
            minetest.chat_send_player(name, "Invalid scale. Please enter a positive number.")
        end
    end
})
