local mod_name = minetest.get_current_modname()
local mod_path = minetest.get_modpath(mod_name)
local S = minetest.get_translator(mod_name)

local function sch(name)
    return (mod_path .. DIR_DELIM .. "schematics" .. DIR_DELIM .. name .. ".mts")
end

local function rand_range(s,m)
    return math.random() * (m-s) + s
end

local function get_natural_light(pos, time)
    if minetest.get_natural_light ~= nil then
        return minetest.get_natural_light(pos, time) or 0
    else
        return minetest.get_node_light(pos, time) or 0
    end
end
br_core.register_level({
    level = 0,
    desc = "Level 0",
    danger = 1,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 28,
    biome_roughness = 195,
    biome_pixelization = 4,
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})

br_core.register_biome({
    level = 0,
    desc = "Level 0 columns",
    danger = 1,
    on_generate = nil,
    segheight = 20,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 10
        and vector.distance(pos, vector.new(0, pos.y, 0)) < 30 then
            return false
        else
            return true
        end
    end,
    schems = {
        {name=sch("0_20_1_columns_0"), vertical_segments={1,0}},
        {name=sch("0_20_2_columns_0"), vertical_segments={0,1}, rotation="0"},
    }
})
br_core.register_biome({
    level = 0,
    desc = "Level 0 main",
    danger = 1,
    on_generate = nil,
    segheight = 20,
    prevalence = 5,
    schems = {
        {name=sch("0_20_1_columns_0"), vertical_segments={1,0}},
        --
        {name=sch("0_20_2_columns_0"), vertical_segments={0,1}, rotation="0"},
        {name=sch("0_20_2_pitfalls_0"), vertical_segments={0,1}, can_generate=function(pos, perlin)
            return ((pos.x%3==0) and pos.z%2==0)
        end},
        {name=sch("0_20_2_walls_0"), vertical_segments={0,1}},
        {name=sch("0_20_2_walls_1"), vertical_segments={0,1}},
        {name=sch("0_20_2_walls_2"), vertical_segments={0,1}},
        {name=sch("0_20_2_walls_3"), vertical_segments={0,1}},
    }
})
br_core.register_biome({
    level = 0,
    desc = "Level 0 main with doors",
    danger = 1,
    on_generate = nil,
    segheight = 20,
    prevalence = 1,
    schems = {
        {name=sch("0_20_1_columns_0"), vertical_segments={1,0}},
        --
        {name=sch("0_20_2_columns_0"), vertical_segments={0,1}, rotation="0", prevalence=12},
        {name=sch("0_20_2_walls_0"), vertical_segments={0,1}, prevalence=12},
        {name=sch("0_20_2_walls_1"), vertical_segments={0,1}, prevalence=12},
        {name=sch("0_20_2_walls_2"), vertical_segments={0,1}, prevalence=12},
        {name=sch("0_20_2_walls_3"), vertical_segments={0,1}, prevalence=12},
        {name=sch("0_20_2_doors_0"), vertical_segments={0,1}, prevalence=2},
        {name=sch("0_20_2_doors_1"), vertical_segments={0,1}, prevalence=2},
        {name=sch("0_20_2_doors_0_portal"), vertical_segments={0,1}, prevalence=2},
        {name=sch("0_20_2_doors_1_portal"), vertical_segments={0,1}, prevalence=2},
        {name=sch("0_20_2_stairs_0_portal"), vertical_segments={0,1}, prevalence=1},
    }
})
br_core.register_biome({
    level = 0,
    desc = "Level 0 concrete",
    danger = 1,
    on_generate = nil,
    segheight = 20,
    prevalence = 1,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 30 then
            return true
        else
            return false
        end
    end,
    schems = {
        {name=sch("0_20_1_columns_0"), vertical_segments={1,0}},
        --
        {name=sch("0_20_2_walls_concrete_0"), vertical_segments={0,1}, prevalence=9},
        {name=sch("0_20_2_walls_concrete_1"), vertical_segments={0,1}, prevalence=5},
        {name=sch("0_20_2_walls_concrete_1_portal"), vertical_segments={0,1}, prevalence=1},
    }
})
br_core.register_biome({
    level = 0,
    desc = "Level 0 nolight",
    danger = 1,
    on_generate = nil,
    segheight = 20,
    prevalence = 2,
    schems = {
        {name=sch("0_20_1_columns_0"), vertical_segments={1,0}},
        --
        {name=sch("0_20_2_nolight_portal_0"), vertical_segments={0,1}, prevalence=1},
        {name=sch("0_20_2_nolight_walls_0"), vertical_segments={0,1}, prevalence=3},
        {name=sch("0_20_2_nolight_walls_1"), vertical_segments={0,1}, prevalence=3},
        {name=sch("0_20_2_nolight_walls_2"), vertical_segments={0,1}, prevalence=3},
        {name=sch("0_20_2_nolight_columns_0"), vertical_segments={0,1}, prevalence=2},
    }
})
br_core.register_biome({
    level = 0,
    desc = "Level 0 dark",
    danger = 1,
    on_generate = nil,
    segheight = 20,
    prevalence = 1,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 30 then
            return true
        else
            return false
        end
    end,
    schems = {
        {name=sch("0_20_1_columns_0"), vertical_segments={1,0}},
        --
        {name=sch("0_20_2_dark_0"), vertical_segments={0,1}, prevalence=8},
        {name=sch("0_20_2_dark_walls_0"), vertical_segments={0,1}, prevalence=8},
        {name=sch("0_20_2_dark_walls_1"), vertical_segments={0,1}, prevalence=8},
        --
        {name=sch("0_20_2_dark_walls_2"), vertical_segments={0,1}, prevalence=4},
        {name=sch("0_20_2_dark_walls_2_portal"), vertical_segments={0,1}},
    }
})

br_core.register_level_sound({
    level = 0,
    time = 8,
    on_play = function(self, dtime, player)
        if player:get_pos().y < -9 then return end
        local can_play = false
        local playing = br_core.sound_get_tag("level0_hum", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level0_hum", player, {
                name = "br_fluorescent_hum",
                level = 0,
                gain = (br_sounds.master or 1) * (0.02 * math.random() + 0.005),
                fade = 0.02,
                to_player = player:get_player_name(),
                time = 8,
            })
        end
        -- shhhhhhh don't tell anyone
        local pos = player:get_pos()
        local dist = vector.distance(pos, vector.new(3000,pos.y,0)) / 5000
        if 0.2 - (dist) > 0 then
            can_play = false
            local playing2 = br_core.sound_get_tag("level0_beep", player)
            if (playing2 == nil or #playing2 < 1)
            or (playing2 and (playing2[1].time <= 2) and #playing2 == 1) then can_play = true end
            if can_play then
                br_core.sound_play("level0_beep", player, {
                    name = "br_level_0_beep",
                    level = 0,
                    gain = (br_sounds.master or 1) * math.max(0.2 - (dist), 0),
                    to_player = player:get_player_name(),
                    time = 18,
                })
            end
        end
    end
})




br_core.register_level({
    level = 1,
    desc = "Level 1",
    danger = 1,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 20,
    sun = {
        visible = false,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})

br_core.register_biome({
    level = 1,
    desc = "Level 1 columns",
    danger = 1,
    on_generate = nil,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 10 then
            return false
        else
            return true
        end
    end,
    schems = {
        {name=sch("1_open_0"), no_rotation=true},
    }
})
br_core.register_biome({
    level = 1,
    desc = "Level 1 main",
    danger = 1,
    on_generate = nil,
    prevalence = 2,
    schems = {
        {name=sch("1_open_0"), no_rotation=true},
        {name=sch("1_open_1"), no_rotation=true},
        {name=sch("1_open_0"), no_rotation=true},
        {name=sch("1_open_1"), no_rotation=true},
        {name=sch("1_open_0"), no_rotation=true},
        {name=sch("1_open_2")},
        {name=sch("1_open_2")},
        {name=sch("1_building_0")}, --PORTAL
        {name=sch("1_building_1")},
        {name=sch("1_building_2")},
        {name=sch("1_stairs_0")},
        {name=sch("1_ramp_0")},
    }
})
br_core.register_biome({
    level = 1,
    desc = "Level 1 main",
    danger = 1,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("1_stairs_0")},
        {name=sch("1_building_0")}, --PORTAL
        {name=sch("1_building_1")},
        {name=sch("1_building_2")},
        {name=sch("1_building_2")},
        {name=sch("1_building_2")},
        {name=sch("1_open_0"), no_rotation=true},
    }
})

br_core.register_level_sound({
    level = 1,
    time = 8,
    scare_timer = {},
    on_play = function(self, dtime, player)
        if self.scare_timer[player] == nil then self.scare_timer[player] = math.random(60, 80) end
        local can_play = false
        local playing = br_core.sound_get_tag("level1_hum", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level1_traffic", player, {
                name = "br_traffic_distant_cc0_klangfabrik",
                level = 1,
                gain = (br_sounds.master or 1) * (0.1 * math.random() + 0.01),
                fade = 0.02,
                to_player = player:get_player_name(),
                time = 8,
            })
            br_core.sound_play("level1_hum", player, {
                name = "br_fluorescent_hum",
                level = 1,
                gain = (br_sounds.master or 1) * (0.002 * math.random() + 0.001),
                fade = 0.01,
                to_player = player:get_player_name(),
                time = 8,
            })
        end

        self.scare_timer[player] = self.scare_timer[player] - dtime
        playing = br_core.sound_get_tag("level1_footsteps", player)
        if (self.scare_timer[player] < 0) and (playing == nil or #playing < 1) then
            self.scare_timer[player] = math.random(60, 240)
            br_core.achievements.achieve(player, "whatwasthat1", 5)
            local yaw = player:get_look_horizontal() + ((math.random(-1,1)) * 1.5) + 3.14
            local dir = vector.multiply(minetest.yaw_to_dir(yaw), 20)
            local pos = vector.add(player:get_pos(), dir)
            if not minetest.line_of_sight(pos, player:get_pos()) then
                br_core.sound_play("level1_footsteps", player, {
                    name = "br_m_trigger_footsteps_concrete",
                    level = 1,
                    gain = (br_sounds.master or 1) * 1,
                    max_hear_distance = 100,
                    to_player = player:get_player_name(),
                    time = 6,
                    pos = pos
                })
            else
                self.scare_timer[player] = 2
            end
        end
    end
})





-- pipes
br_core.register_level({
    level = 2,
    desc = "Level 2",
    danger = 3,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 20,
    sun = {
        visible = false,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})

br_core.register_biome({
    level = 2,
    desc = "Level 2 corridors",
    danger = 3,
    on_generate = nil,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 10 then
            return false
        else
            return true
        end
    end,
    schems = {
        {name=sch("2_crossroad_0"), no_rotation=true},
    }
})

br_core.register_biome({
    level = 2,
    desc = "Level 0 main",
    danger = 1,
    on_generate = nil,
    biome_roughness = 1,
    biome_pixelization = 2,
    prevalence = 1,
    schems = {
        {name=sch("2_maze_0"), rotation="0", prevalence=9},
        {name=sch("2_crossroad_0"), rotation="0", prevalence=2},
        {name=sch("2_open_0")},
        {name=sch("2_water_0")},
    }
})

br_core.register_biome({
    level = 2,
    desc = "Level 0 main",
    danger = 1,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("2_blue_1"), rotation="0", prevalence=5},
        {name=sch("2_blue_1"), rotation="90", prevalence=1},
        {name=sch("2_blue_1_portal"), no_rotation=true},
        {name=sch("2_blue_0"), rotation="0", prevalence=4},
        {name=sch("2_blue_0_portal")},
    }
})

br_core.register_biome({
    level = 2,
    desc = "Level 0 main",
    danger = 1,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("2_green_0"), rotation="0", prevalence=4},
        {name=sch("2_green_1_portal")},
        {name=sch("2_green_1"), rotation="0", prevalence=4},
        {name=sch("2_green_2"), rotation="0", prevalence=4},
        {name=sch("2_green_0_nolight")},
        {name=sch("2_green_1_nolight")},
    }
})

br_core.register_level_sound({
    level = 2,
    time = 8,
    scare_timer = {},
    on_play = function(self, dtime, player)
        if self.scare_timer[player] == nil then self.scare_timer[player] = math.random(120, 200) end
        local can_play = false
        local playing = br_core.sound_get_tag("level2_hum", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level2_hum", player, {
                name = "br_fluorescent_hum",
                level = 2,
                gain = (br_sounds.master or 1) * (0.003 * math.random() + 0.001),
                fade = 0.01,
                to_player = player:get_player_name(),
                time = 8,
            })
            br_core.sound_play("2_dripsandwater", player, {
                name = "br_dripsandwater_WTFPL_Sumianvoice",
                level = 2,
                gain = (br_sounds.master or 1) * (math.random()*0.3),
                fade = 0.2,
                max_hear_distance = 50,
                to_player = player:get_player_name(),
                time = 40,
            })
        end

        self.scare_timer[player] = self.scare_timer[player] - dtime
        playing = br_core.sound_get_tag("level2_footsteps", player)
        if (self.scare_timer[player] < 0) and (playing == nil or #playing < 1) then
            self.scare_timer[player] = math.random(60, 240)
            local yaw = player:get_look_horizontal() + ((math.random(-1,1)) * 1.5) + 3.14
            local dir = vector.multiply(minetest.yaw_to_dir(yaw), 20)
            local pos = vector.add(player:get_pos(), dir)
            br_core.sound_play("level2_footsteps", player, {
                name = "br_m_trigger_footsteps_concrete",
                level = 1,
                gain = (br_sounds.master or 1) * 1,
                max_hear_distance = 100,
                to_player = player:get_player_name(),
                time = 6,
                pos = pos
            })
        end
    end
})







-- hotel
br_core.register_level({
    level = 5,
    desc = "Level 5",
    danger = 1,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 12,
    sun = {
        visible = false,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})

br_core.register_biome({
    level = 5,
    desc = "Level 5 rust",
    danger = 1,
    on_generate = nil,
    can_generate = function(pos)
        return true
    end,
    segheight = 10,
    schems = {
        {name=sch("5_10h_rust_1_portal"), vertical_segments={1,1}},
        {name=sch("5_10h_rust_0"), rotation="0", vertical_segments={1,1}, prevalence=3},
        {name=sch("5_10h_rust_1"), rotation="0", vertical_segments={1,1}, prevalence=3},
        {name=sch("5_10h_rust_2"), rotation="0", vertical_segments={1,1}, prevalence=3},
    }
})

br_core.register_biome({
    level = 5,
    desc = "Level 0 orange",
    danger = 1,
    on_generate = nil,
    prevalence = 1,
    segheight = 10,
    schems = {
        {name=sch("5_10h_orange_2_portal"), vertical_segments={1,1}},
        {name=sch("5_10h_orange_0"), rotation="0", vertical_segments={1,1}, prevalence=3},
        {name=sch("5_10h_orange_1"), rotation="0", vertical_segments={1,1}, prevalence=3},
        {name=sch("5_10h_orange_2"), rotation="0", vertical_segments={1,1}, prevalence=3},
    }
})

br_core.register_biome({
    level = 5,
    desc = "Level 0 black",
    danger = 1,
    on_generate = nil,
    prevalence = 1,
    segheight = 10,
    schems = {
        {name=sch("5_10h_black_0"), rotation="0", vertical_segments={1,1}, prevalence=1},
        {name=sch("5_20h_black_stairs_0"), vertical_segments={1,0}, prevalence=1, skip_above=1},
        {name=sch("5_10h_black_1"), rotation="90", vertical_segments={1,1}, prevalence=1},
        {name=sch("5_10h_black_2"), rotation="90", vertical_segments={1,1}, prevalence=1},
        {name=sch("5_10h_black_3"), rotation="90", vertical_segments={1,1}, prevalence=1},
        {name=sch("5_10h_black_0"), rotation="0", vertical_segments={1,1}, prevalence=3},
        {name=sch("5_10h_black_1"), rotation="0", vertical_segments={1,1}, prevalence=6},
        {name=sch("5_10h_black_2"), rotation="0", vertical_segments={1,1}, prevalence=6},
        {name=sch("5_10h_black_3"), rotation="0", vertical_segments={1,1}, prevalence=6},
        {name=sch("5_10h_black_portal_0"), rotation="0", vertical_segments={1,1}, prevalence=2},
    }
})

br_core.register_level_sound({
    level = 5,
    time = 8,
    on_play = function(self, dtime, player)
        local can_play = false
        local playing = br_core.sound_get_tag("level5_hum", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level5_hum", player, {
                name = "br_fluorescent_hum",
                level = 5,
                gain = (br_sounds.master or 1) * (0.02 * math.random() + 0.001),
                fade = 0.01,
                to_player = player:get_player_name(),
                time = 8,
            })
        end
    end
})





-- hub
br_core.register_level({
    level = 12,
    desc = "Level 12 hub",
    danger = 0,
    grace_dist = 5,
    biome = {},
    segsize = 40,
    base_height = 23,
    sun = {
        visible = true,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})

local l12pattern = {
    1,2,2,2,2,2,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0
}
br_core.register_biome({
    level = 12,
    desc = "Level 12 tunnel",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos)
        return ((pos.z % 3 == 0) and l12pattern[pos.x%30+1] <= 1)
    end,
    schems = {
        {name=sch("12_tunnel_portal_0"), rotation="90", prevalence=4},
        {name=sch("12_tunnel_random_0"), rotation="90", prevalence=3},
        {name=sch("12_tunnel_0"), rotation="90", prevalence=10},
        {name=sch("12_tunnel_1"), rotation="90", prevalence=3},
    }
})
br_core.register_biome({
    level = 12,
    desc = "Level 12 block",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos)
        return ((pos.z % 3 ~= 0) and l12pattern[pos.x%30+1] == 0)
    end,
    schems = {
        {name=sch("12_block_0")},
    }
})
br_core.register_biome({
    level = 12,
    desc = "Level 12 empty",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos)
        return ((pos.z % 3 ~= 0) and l12pattern[pos.x%30+1] == 1)
    end,
    schems = {
        {name=sch("12_block_1")},
    }
})
br_core.register_biome({
    level = 12,
    desc = "Level 12 bridge",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos)
        return ((pos.z % 3 == 0) and l12pattern[pos.x%30+1] == 2)
    end,
    schems = {
        {name=sch("12_bridge_0"), rotation="90"},
    }
})
br_core.register_biome({
    level = 12,
    desc = "Level 12 water",
    danger = 3,
    on_generate = nil,
    can_generate = function(pos)
        return ((pos.z % 3 ~= 0) and l12pattern[pos.x%30+1] == 2)
    end,
    schems = {
        {name=sch("12_water_0"), rotation="90"},
    }
})
br_core.register_on_changed_level(function(player, from_level, to_level)
    if to_level == 12 then
        player:override_day_night_ratio(0.2)
    elseif from_level == 12 then
        player:override_day_night_ratio(1)
    end
end)





-- hub
br_core.register_level({
    level = 17,
    desc = "Level 17",
    danger = 2,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 6,
    biome_roughness = 17862,
    biome_pixelization = 4,
    sun = {
        visible = true,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})

br_core.register_biome({
    level = 17,
    desc = "Level 17 canals",
    danger = 2,
    on_generate = nil,
    segheight = 20,
    can_generate = function(pos)
        return true
    end,
    schems = {
        {name=sch("17_fill_0"), rotation="0", vertical_segments={0,1,1,1}},
        {name=sch("17_canal_0"), rotation="0", prevalence=3, vertical_segments={1,0,0,0}},
        {name=sch("17_canal_1"), rotation="0", prevalence=3, vertical_segments={1,0,0,0}},
        {name=sch("17_canal_1"), rotation="180", prevalence=3, vertical_segments={1,0,0,0}},
        {name=sch("17_canal_2"), rotation="0", prevalence=3, vertical_segments={1,0,0,0}},
        {name=sch("17_canal_2"), rotation="180", prevalence=3, vertical_segments={1,0,0,0}},
        {name=sch("17_canal_2_portal"), rotation="180", prevalence=1, vertical_segments={1,0,0,0}},
        -- {name=sch("17_canal_0"), rotation="180", prevalence=1, vertical_segments={1,0,0}},
    }
})
br_core.register_biome({
    level = 17,
    desc = "Level 17 storm drains",
    danger = 0,
    on_generate = nil,
    segheight = 20,
    can_generate = function(pos)
        return true
    end,
    schems = {
        {name=sch("17_fill_0"), rotation="0", vertical_segments={0,1,1,1}},
        {name=sch("17_storm_drains_0"), rotation="0", prevalence=1, vertical_segments={1,0,0,0}, skip_above=3},
    }
})

br_core.register_level_sound({
    level = 17,
    time = 40,
    on_play = function(self, dtime, player)
        local can_play = false
        local playing = br_core.sound_get_tag("17_dripsandwater", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("17_dripsandwater", player, {
                name = "br_dripsandwater_WTFPL_Sumianvoice",
                level = 17,
                gain = (br_sounds.master or 1) * math.random() * 0.5,
                fade = 0.2,
                -- single_sound = true,
                max_hear_distance = 50,
                to_player = player:get_player_name(),
                -- pos = get_centre_of_segment(player:get_pos()),
                time = 40,
            })
        end
    end
})





-- ???
br_core.register_level({
    level = 31,
    desc = "Level 31",
    danger = 0,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 42,
    biome_roughness = 91.798,
    sun = {
        visible = false,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})
br_core.register_biome({
    level = 31,
    desc = "Level 31",
    danger = 0,
    on_generate = nil,
    segheight = 10,
    prevalence=2,
    can_generate = function(pos) -- must be deterministic
        return ((pos.z%5 ~= 0) or (pos.x%3 == 0))
    end,
    schems = {
        {name=sch("31_10h_corridor_base"),  vertical_segments={1}, prevalence=3, rotation="0"},
        {name=sch("31_10h_corridor_roof"),  vertical_segments={0,0,0,0,0,0,0,1}, rotation="0", prevalence=7},
        {name=sch("31_10h_corridor_0"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0", prevalence=7},
        {name=sch("31_10h_corridor_1"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0", prevalence=7},
        {name=sch("31_10h_corridor_2"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0", prevalence=7},
        {name=sch("31_10h_corridor_3"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0", prevalence=7},
        {name=sch("31_10h_corridor_4"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0", prevalence=7},
        {name=sch("31_10h_corridor_5"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0", prevalence=7},
        {name=sch("31_20h_stairs_0"), vertical_segments={0,1,1,1,1,1,0,0}, skip_above=1, prevalence=3},
        {name=sch("31_20h_stairs_1"), vertical_segments={0,1,1,1,1,1,0,0}, skip_above=1, prevalence=3},
        {name=sch("31_20h_stairs_0_portal"), vertical_segments={0,1,1,1,1,1,0,0}, skip_above=1, prevalence=3},
    }
})
br_core.register_biome({
    level = 31,
    desc = "Level 31",
    danger = 0,
    on_generate = nil,
    segheight = 10,
    can_generate = function(pos) -- must be deterministic
        return (pos.z%5 ~= 0 and (pos.x%8 < 4 and pos.z%7 < 4))
    end,
    schems = {
        {name=sch("31_10h_balcony_base"),  vertical_segments={1}, prevalence=3},
        {name=sch("31_10h_balcony_roof"),  vertical_segments={0,0,0,0,0,0,0,1}},

        {name=sch("31_10h_balcony_0"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("31_10h_balcony_1"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("31_10h_balcony_2"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("31_10h_balcony_3"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("31_10h_balcony_4"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("31_10h_balcony_5"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("31_10h_balcony_6"),     vertical_segments={0,1,1,1,1,1,1,0}, prevalence=3},
        --
        {name=sch("31_10h_balcony_0_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
        {name=sch("31_10h_balcony_1_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
        {name=sch("31_10h_balcony_2_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
        {name=sch("31_10h_balcony_3_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
        {name=sch("31_10h_balcony_4_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
        {name=sch("31_10h_balcony_5_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
        {name=sch("31_10h_balcony_6_portal"),     vertical_segments={0,1,1,1,1,1,1,0}},
    }
})

br_core.register_biome({
    level = 31,
    desc = "Level 31 SHORT",
    danger = 0,
    on_generate = nil,
    segheight = 10,
    prevalence = 1,
    can_generate = function(pos) -- must be deterministic
        return pos.z%5 == 0
    end,
    schems = {
        {name=sch("31_10h_corridor_short_base"),  vertical_segments={1}, prevalence=3, rotation="0"},
        {name=sch("31_10h_corridor_short_roof"),  vertical_segments={0,0,0,0,0,0,0,1}, rotation="0"},
        {name=sch("31_10h_corridor_short_0"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0"},
        {name=sch("31_10h_corridor_short_1"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0"},
        {name=sch("31_10h_corridor_short_2"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0"},
        {name=sch("31_10h_corridor_short_3"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0"},
        {name=sch("31_10h_corridor_short_4"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0"},
        {name=sch("31_10h_corridor_short_5"),     vertical_segments={0,1,1,1,1,1,1,0}, rotation="0"},
    }
})

br_core.register_level_sound({
    level = 31,
    time = 10,
    on_play = function(self, dtime, player)
        local can_play = false
        local playing = br_core.sound_get_tag("level31_hum", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level31_hum", player, {
                name = "br_traffic_distant_cc0_klangfabrik",
                level = 31,
                gain = (br_sounds.master or 1) * (0.5 * math.random() + 0.3),
                fade = 0.3,
                -- single_sound = true,
                max_hear_distance = 50,
                to_player = player:get_player_name(),
                time = 10,
            })
        end
    end
})



-- poolrooms
br_core.register_level({
    level = 37,
    desc = "Level 37",
    danger = 0,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 20,
    biome_roughness = 896,
    biome_pixelization = 4,
    sun = {
        visible = true,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#def",
        type = "plain"
    },
})

br_core.register_biome({
    level = 37,
    desc = "Level 37 columns",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 10 then
            return false
        else
            return true
        end
    end,
    schems = {
        {name=sch("37_channel_0")},
    }
})
br_core.register_biome({
    level = 37,
    desc = "Level 0 main",
    danger = 0,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("37_channel_0"), prevalence=3},
        {name=sch("37_channel_1"), prevalence=3},
        {name=sch("37_channel_2"), prevalence=2},
        {name=sch("37_channel_3"), prevalence=1},
        {name=sch("37_pool_0"), prevalence=2},
        {name=sch("37_stair_0"), prevalence=2},
    }
})
br_core.register_biome({
    level = 37,
    desc = "Level 0 closed",
    danger = 0,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("37_channel_0"), rotation="0", prevalence=1},
        {name=sch("37_channel_2"), rotation="0", prevalence=1},
        {name=sch("37_closed_0"), prevalence=5},
        {name=sch("37_closed_1"), prevalence=4},
        {name=sch("37_closed_1_portal"), prevalence=1},
        {name=sch("37_pool_0"), prevalence=2},
        {name=sch("37_stair_0"), prevalence=2},
    }
})

br_core.register_level_sound({
    level = 37,
    time = 40,
    on_play = function(self, dtime, player)
        local can_play = false
        local playing = br_core.sound_get_tag("37_dripsandwater", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("37_dripsandwater", player, {
                name = "br_dripsandwater_WTFPL_Sumianvoice",
                level = 37,
                gain = (br_sounds.master or 1) * math.random(),
                fade = 0.2,
                -- single_sound = true,
                max_hear_distance = 50,
                to_player = player:get_player_name(),
                -- pos = get_centre_of_segment(player:get_pos()),
                time = 40,
            })
        end
    end
})





br_core.register_level({
    level = 45,
    desc = "Level 45",
    secret = true,
    danger = 0,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 41,
    sun = {
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
		base_color = "#ffffff",
		type = "regular",
        clouds = false,
		sky_color = {
			day_sky = "#92b9f2" ,
			day_horizon = "#010c22",
			dawn_sky = "#92b9f2" ,
			dawn_horizon = "#010c22",
			night_sky = "#006bff" ,
			night_horizon = "#010c22",
			indoors = "#010c22",
			fog_sun_tint = "#010c22",
			fog_moon_tint = "#010c22",
			fog_tint_type = "custom",
		}
	},
})

br_core.register_biome({
    level = 45,
    desc = "Level 45 doors",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos) -- must be deterministic
        if vector.distance(pos, vector.new(0, pos.y, 0)) > 10 then
            return false
        else
            return true
        end
    end,
    schems = {
        {name=sch("45_doors_0"), no_rotation=true},
        {name=sch("45_sparse_0"), no_rotation=true},
        {name=sch("45_sparse_0"), no_rotation=true},
    }
})

br_core.register_biome({
    level = 45,
    desc = "Level 45 doors with portal",
    danger = 0,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("45_sparse_0"), no_rotation=true},
        {name=sch("45_sparse_0"), no_rotation=true},
        {name=sch("45_sparse_0"), no_rotation=true},
        {name=sch("45_sparse_0_portal"), no_rotation=true},
        {name=sch("45_doors_0"), no_rotation=true},
        {name=sch("45_doors_0"), no_rotation=true},
        {name=sch("45_doors_0"), no_rotation=true},
    }
})

br_core.register_level_sound({
    level = 45,
    time = 7,
    on_play = function(self, dtime, player)
        local can_play = false
        local playing = br_core.sound_get_tag("level45_wind", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level45_wind", player, {
                name = "br_wind_noise_WTFPL_Sumianvoice",
                level = 45,
                gain = (br_sounds.master or 1) * (0.2 * math.random() + 0.1),
                fade = 0.15,
                to_player = player:get_player_name(),
                time = 7,
            })
        end
    end
})



-- ???
br_core.register_level({
    level = 51,
    desc = "Level 51 area",
    danger = 1,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 2,
    biome_roughness = 8.654,
    sun = {
        visible = false,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#001",
        type = "plain"
    },
})
br_core.register_biome({
    level = 51,
    desc = "Level 51",
    danger = 1,
    on_generate = nil,
    segheight = 10,
    can_generate = function(pos) -- must be deterministic
        return true
    end,
    schems = {
        {name=sch("51_shelves_0"),  vertical_segments={1}, prevalence=1, skip_above=3},
        {name=sch("51_shelves_1"),  vertical_segments={1}, prevalence=1, skip_above=3},
        {name=sch("51_shelves_0_portal"),  vertical_segments={1}, prevalence=1, skip_above=3, can_generate = function(pos)
            return ((pos.x % 2 == 0) and (pos.z % 2 == 0))
        end},
        {name=sch("51_shelves_0"),  vertical_segments={1}, prevalence=3, skip_above=3, rotation="0"},
        {name=sch("51_shelves_1"),  vertical_segments={1}, prevalence=3, skip_above=3, rotation="0"},
    }
})
br_core.register_biome({
    level = 51,
    desc = "Level 51",
    danger = 1,
    on_generate = nil,
    segheight = 10,
    schems = {
        {name=sch("51_shelves_0"),  vertical_segments={1}, prevalence=1, skip_above=3},
        {name=sch("51_shelves_1"),  vertical_segments={1}, prevalence=1, skip_above=3},
        {name=sch("51_piles_0"),  vertical_segments={1}, prevalence=9, skip_above=3},
        {name=sch("51_piles_0_portal"),  vertical_segments={1}, prevalence=1, skip_above=3},
        {name=sch("51_car_0"),  vertical_segments={1}, prevalence=1, skip_above=3, can_generate = function(pos)
            return ((pos.x % 3 == 0) and (pos.z % 2 == 0))
        end},
    }
})




-- hospital
br_core.register_level({
    level = 62,
    desc = "Level 62 hospital",
    danger = 2,
    grace_dist = 5,
    biome = {},
    segsize = 40,
    base_height = 2,
    sun = {
        visible = false,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    sky = {
        clouds = false,
        base_color = "#eee",
        type = "plain"
    },
})
br_core.register_biome({
    level = 62,
    desc = "Level 62",
    danger = 2,
    on_generate = nil,
    schems = {
        {name=sch("62_corridors_0"), rotation="0"},
        {name=sch("62_corridors_0"), rotation="90"},
        {name=sch("62_corridors_1"), rotation="0"},
        {name=sch("62_corridors_1"), rotation="90"},
        {name=sch("62_corridors_2"), rotation="0"},
        {name=sch("62_corridors_2"), rotation="90"},
        {name=sch("62_corridors_closed_0"), rotation="0", prevalence=3},
        {name=sch("62_corridors_closed_0"), rotation="90", prevalence=3},
    }
})

br_core.register_level_sound({
    level = 62,
    time = 8,
    on_play = function(self, dtime, player)
        local can_play = false
        local playing = br_core.sound_get_tag("level62_hum", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then can_play = true end
        if can_play then
            br_core.sound_play("level62_hum", player, {
                name = "br_fluorescent_hum",
                level = 62,
                gain = (br_sounds.master or 1) * (0.04 * math.random() + 0.001),
                fade = 0.01,
                to_player = player:get_player_name(),
                time = 8,
            })
        end
    end
})




if true then
local l96_pattern = {}

local function l96_pattern_add(def)
    for i=1, def.prevalence or 1 do
        l96_pattern[#l96_pattern+1] = def.pattern
    end
end
l96_pattern_add({
    prevalence = 9,
    pattern = {
        {"SW","SS","SS","SE","TV"},
        {"WW","OO","OO","EE","TV"},
        {"WW","ST","OO","EE","TV"},
        {"NW","NN","NN","NE","TV"},
        {"TH","TH","TH","TH","TX"},
    },
})
l96_pattern_add({
    prevalence = 6,
    pattern = {
        {"SW","SS","SS","SE","TV"},
        {"WW","ST","OO","NE","TV"},
        {"WY","OO","EE","PP","TV"},
        {"NW","NN","NE","PP","TV"},
        {"TH","TH","TH","TH","TX"},
    },
})
l96_pattern_add({
    prevalence = 1,
    pattern = {
        {"SW","SE","PP","PP","TV"},
        {"WY","EE","PP","PP","TV"},
        {"WW","EE","PP","PP","TV"},
        {"NW","NE","PP","PP","TV"},
        {"TH","TH","TH","TH","TX"},
    },
})
l96_pattern_add({
    prevalence = 1,
    pattern = {
        {"PP","PP","PP","SW","SE"},
        {"SW","SE","PP","NW","NE"},
        {"WY","EE","PP","PP","PP"},
        {"NW","NE","PP","PP","PP"},
        {"TH","TH","TH","TH","TX"},
    },
})
l96_pattern_add({
    prevalence = 3,
    pattern = {
        {"PP","PP","PP","PP","PP"},
        {"PP","PP","PP","PP","PP"},
        {"PP","PP","PP","PP","PP"},
        {"PP","PP","PP","PP","PP"},
        {"PP","PP","PP","PP","PP"},
    },
})

local length = 5
local width = 5
local function l96_glitchy_version(x, z, perlin)
    local len = 2
    local roughness = 21964.27896 * vector.distance(vector.new(x, 0, z), vector.new(0, 0, 0))
    local variant = math.floor(perlin.variant.data3d[x%len+1][2][z%len+1] * roughness) % #l96_pattern + 1
    return "ALL"
    --l96_pattern[variant][z%length+1][x%width+1]
end

-- streets
br_core.register_level({
    level = 96,
    desc = "Level 96 streets",
    danger = 2,
    grace_dist = 5,
    biome = {},
    segsize = 16,
    base_height = 10,
    on_generate = function(pos, perlin)
        local x, z = pos.x, pos.z
        local roughness = 21964.27896
        local variant = math.floor(perlin.variant.data3d[2][2][2] * roughness) % #l96_pattern + 1

        -- weird "doesn't care about walls" version
        if vector.distance(pos, vector.new(0,0,0)) > 500
        and x%60 > 6 and z%60 > 6 then
            return l96_glitchy_version(x, z, perlin)
        end

        return l96_pattern[variant][z%length+1][x%width+1]
    end,
    sun = {
        visible = true,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    clouds = {
        density = 0.6,
        thickness = 2.3,
        ambient = "#fff",
        color = "#ffffff20",
        height = 96 * br_core.level_height - br_core.offset + 100,
        speed = {x=2.7, z=3}
    },
    sky = {
		base_color = "#ffffff",
		type = "regular",
		clouds = true,
		sky_color = {
			day_sky = "#80aed9" ,
			day_horizon = "#8086a1",
			dawn_sky = "#979" ,
			dawn_horizon = "#fb9",
			night_sky = "#263965" ,
			night_horizon = "#263965",
			indoors = "#263965",
			fog_sun_tint = "#263965",
			fog_moon_tint = "#263965",
			fog_tint_type = "custom",
		}
	},
})
local l96pl = {}
local l96timer = -1
br_core.register_on_changed_level(function(player, from_level, to_level)
    if to_level == 96 then
        l96pl[player] = true
        player:override_day_night_ratio(1)
    elseif from_level == 96 then
        l96pl[player] = nil
        player:override_day_night_ratio(1)
    end
end)
minetest.register_globalstep(function(dtime)
    local day_period = 600
    if l96timer == -1 then
        l96timer = world_storage:get_key("l96timer") or day_period * 0.4
    end
    l96timer = (l96timer + dtime) % day_period
    if l96timer - dtime < 0 and world_storage then
        world_storage:set_key("l96timer", l96timer)
    end
    for player,_ in pairs(l96pl) do
        if player and player.is_player and player:is_player() then
            player:override_day_night_ratio(math.sin(l96timer / day_period * 3.14))
        end
    end
end)
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "NW",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_corner_0"), rotation="180", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "NE",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_corner_0"), rotation="270", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "SE",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_corner_0"), rotation="0", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "SW",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_corner_0"), rotation="90", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "NN",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_wall_0"), rotation="180", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_1"), rotation="180", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2"), rotation="180", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2_portal"), rotation="180", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_stair_continuous_wall_0"), rotation="0", vertical_segments={0,1}, skip_above=8},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "EE",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), rotation="270", vertical_segments={1}},
        --
        {name=sch("96_wall_0"), rotation="270", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_1"), rotation="270", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2"), rotation="270", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2_portal"), rotation="270", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_stair_continuous_wall_0"), rotation="90", vertical_segments={0,1}, skip_above=8},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "SS",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_wall_0"), rotation="0", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_1"), rotation="0", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2"), rotation="0", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2_portal"), rotation="0", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_stair_continuous_wall_0"), rotation="180", vertical_segments={0,1}, skip_above=8},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "WW",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_wall_0"), rotation="90", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_1"), rotation="90", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2"), rotation="90", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_wall_2_portal"), rotation="90", vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_stair_continuous_wall_0"), rotation="270", vertical_segments={0,1}, skip_above=8},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "WY",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_stair_continuous_wall_0"), rotation="270", vertical_segments={0,1}, skip_above=8},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "TV",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        {name=sch("16x8_air"), rotation="0", vertical_segments={0,0,1,1,1,1,1,1,1,0}},
        --
        {name=sch("96_road_0"), rotation="0", vertical_segments={0,1,0,0,0,0,0,0,0,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "TH",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        {name=sch("16x8_air"), rotation="0", vertical_segments={0,0,1,1,1,1,1,1,1,0}},
        --
        {name=sch("96_road_0"), rotation="90", vertical_segments={0,1,0,0,0,0,0,0,0,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "TX",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        {name=sch("16x8_air"), rotation="0", vertical_segments={0,0,1,1,1,1,1,1,1,0}},
        --
        {name=sch("96_crossroad_0"), rotation="0", vertical_segments={0,1,0,0,0,0,0,0,0,0}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "OO",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        {name=sch("16x8_air"), rotation="0", vertical_segments={0,0,0,0,0,0,0,0,0,1}},
        --
        {name=sch("96_middle_0"), free_rotation=true, vertical_segments={0,1,1,1,1,1,1,1,1,0}, prevalence=1},
        {name=sch("96_middle_1"), free_rotation=true, vertical_segments={0,1,1,1,1,1,1,1,1,0}, prevalence=1},
        {name=sch("96_middle_2"), free_rotation=true, vertical_segments={0,1,1,1,1,1,1,1,1,0}, prevalence=3},
        {name=sch("96_middle_0_portal"), free_rotation=true, vertical_segments={0,1,1,1,1,1,1,1,1,0}},
        {name=sch("96_stair_0"), free_rotation=true, vertical_segments={0,1,1,1,1,1,1,1,0,0}, skip_above=1, prevalence=3},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "PP",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        {name=sch("16x8_air"), rotation="0", vertical_segments={0,0,1,1,1,1,1,1,1,0}},
        --
        {name=sch("96_park_0"), vertical_segments={0,1}},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "ST",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        {name=sch("16x8_air"), rotation="0", vertical_segments={0,0,1,1,1,1,1,1,1,0}},
        --
        {name=sch("96_stair_continuous_0"), vertical_segments={0,1}, skip_above=8},
    }
})
br_core.register_biome({
    level = 96,
    desc = "Level 96",
    name = "ALL",
    danger = 2,
    on_generate = nil,
    segheight = 8,
    schems = {
        {name=sch("96_underground_0"), vertical_segments={1}},
        --
        {name=sch("96_stair_continuous_0"), vertical_segments={0,1}, skip_above=8, prevalence=9},
        {name=sch("96_blank_ruin"), vertical_segments={0,0,0,0,0,0,0,0,0,1}, prevalence=3},
        {name=sch("96_blank_ruin"), vertical_segments={0,0,0,0,0,0,0,0,1,0}, prevalence=8, skip_above=1},
        {name=sch("96_blank_ruin"), vertical_segments={0,0,0,0,0,0,0,1,0,0}, prevalence=5, skip_above=2},
        --
        {name=sch("96_blank_ruin"), vertical_segments={0,0,0,0,0,1,0,0,0,0}, prevalence=9, skip_above=4},
        --
        {name=sch("96_blank_ruin"), vertical_segments={1,0,0,0,0,0,0,0,0,0}, prevalence=5, skip_above=9},
        {name=sch("96_blank_ruin"), vertical_segments={0,0,1,0,0,0,0,0,0,0}, prevalence=5, skip_above=8},
            {name=sch("96_blank_ruin"), vertical_segments={1}, prevalence=2, skip_above=9, can_generate=function(pos)
            if (pos.x % 60 < 12) and (pos.z % 60 < 12) then return false
            else return true end
        end},
        --
        {name=sch("96_corner_0"), vertical_segments={0,1,1,1,1,1,1,1,1,1}},
        {name=sch("96_middle_0"), free_rotation=true, vertical_segments={1,1,1,1,1,1,1,1,1,1}, prevalence=1},
        {name=sch("96_middle_1"), free_rotation=true, vertical_segments={1,1,1,1,1,1,1,1,1,1}, prevalence=1},
        {name=sch("96_middle_2"), free_rotation=true, vertical_segments={1,1,1,1,1,1,1,1,1,1}, prevalence=3},
        {name=sch("96_middle_0_portal"), free_rotation=true, vertical_segments={1,1,1,1,1,1,1,1,1,1}},
        --
        {name=sch("96_stair_0"), free_rotation=true, vertical_segments={1,1,1,1,1,1,1,1,0,0}, skip_above=1, prevalence=3},
        {name=sch("96_stair_0"), free_rotation=true, vertical_segments={1,0,0,0,0,0,0,0,0,0}, prevalence=3, skip_above=1},
    }
})
end





if true then
-- streets
br_core.register_level({
    level = 167,
    desc = "Level 167",
    danger = 2,
    grace_dist = 5,
    biome = {},
    segsize = 16,
    biome_roughness = 2.728,
    -- level_height = 160,
    biome_pixelization = 3,
    base_height = 17,
    -- on_generate = function(pos, perlin)
    -- end,
    sun = {
        visible = true,
        texture = "blank.png",
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    clouds = {
        density = 0.6,
        thickness = 2.3,
        ambient = "#333",
        color = "#33333320",
        height = 167 * br_core.level_height - br_core.offset + 80 + 50,
        speed = {x=6.9, z=12}
    },
    sky = {
		base_color = "#ffffff",
		type = "regular",
		clouds = true,
		sky_color = {
			day_sky = "#b1b2b1" ,
			day_horizon = "#6a6270",
			dawn_sky = "#5d5d5f" ,
			dawn_horizon = "#6e5766",
			night_sky = "#fff" ,
			night_horizon = "#fff",
			indoors = "#3b3b3b",
			fog_sun_tint = "#263965",
			fog_moon_tint = "#263965",
			fog_tint_type = "custom",
		}
	},
})
local l167pl = {}
local l167timer = -1
br_core.register_on_changed_level(function(player, from_level, to_level)
    if to_level == 167 then
        l167pl[player] = true
        player:override_day_night_ratio(1)
    elseif from_level == 167 then
        l167pl[player] = nil
        player:override_day_night_ratio(1)
    end
end)
minetest.register_globalstep(function(dtime)
    local day_period = 600
    if l167timer == -1 then
        l167timer = world_storage:get_key("l167timer") or day_period * 0.2
    end
    l167timer = (l167timer + dtime) % day_period
    if l167timer - dtime < 0 and world_storage then
        world_storage:set_key("l167timer", l167timer)
    end
    for player,_ in pairs(l167pl) do
        if player and player.is_player and player:is_player() then
            player:override_day_night_ratio( 0.2 + math.sin(l167timer / day_period * 3.14) * 0.3)
            local pos = player:get_pos()
            for i=0, 3 do
                local light = get_natural_light(pos, 0.5)
                if light >= 14 then
                    local pp = vector.new(
                        (rand_range(pos.x-40, pos.x+40)),
                        (rand_range(pos.y-2,  pos.y+18)),
                        (rand_range(pos.z-40, pos.z+40))
                    )
                    local vel = vector.new(
                        (rand_range(3, 8)),
                        (rand_range(-1, 1)),
                        (rand_range(3, 8))
                    )
                    pp = vector.offset(pp, -10, 0, -10)
                    minetest.add_particle({
                        pos = pp,
                        velocity = vel,
                        playername = player:get_player_name(),
                        expirationtime = 4,
                        collisiondetection = true,
                        collision_removal = true,
                        texture = "br_dust_particle.png",
                        size = 1
                    })
                end
            end
        end
    end
end)
br_core.register_biome({
    level = 167,
    desc = "Level 167",
    -- name = "NW",
    danger = 2,
    on_generate = nil,
    segheight = 16,
    can_generate = function(pos)
        return (pos.x % 73 > 20 and pos.z % 92 < 20)
    end,
    schems = {
        {name=sch("167_main_0"),      free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_main_1"),      free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_main_2"),      free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_main_3"),      free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_main_3_portal"),    vertical_segments={1,1,1,1,1}, prevalence=2, can_generate=function(pos)
            return (pos.x%3==0 or pos.z%2==0)
        end},
        {name=sch("167_main_4"),      free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_main_5"),      free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_apartment_1"), free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=3},
        {name=sch("167_flat_0"),      free_rotation=true, vertical_segments={0,0,0,1,1}, prevalence=7},
        {name=sch("167_flat_1"),      free_rotation=true, vertical_segments={0,0,0,1,1}, prevalence=7},
        {name=sch("167_flat_2"),      free_rotation=true, vertical_segments={0,0,0,1,1}, prevalence=7},
        {name=sch("167_flat_3"),      free_rotation=true, vertical_segments={0,0,0,1,1}, prevalence=7},
        {name=sch("167_stairs_0"),    free_rotation=true, vertical_segments={1,1,1,1,0}, skip_above=1},
        {name=sch("167_stairs_1"),    free_rotation=true, vertical_segments={1,1,1,1,0}, skip_above=1},
        {name=sch("167_stairs_2"),    free_rotation=true, vertical_segments={1,1,1,1,0}, skip_above=1},
    }
})
br_core.register_biome({
    level = 167,
    desc = "Level 167",
    -- name = "NW",
    danger = 2,
    on_generate = nil,
    segheight = 16,
    can_generate = function(pos)
        return not (pos.x % 73 < 20 and pos.z % 92 < 20)
    end,
    schems = {
        -- {name=sch("16x16x16_null"), vertical_segments={0,0,0,0,0}},
        {name=sch("167_main_0"),    free_rotation=true, vertical_segments={1,1,0,0,0}},
        {name=sch("167_main_1"),    free_rotation=true, vertical_segments={1,1,0,0,0}},
        {name=sch("167_main_2"),    free_rotation=true, vertical_segments={1,1,0,0,0}},
        {name=sch("167_main_3"),    vertical_segments={1,1,0,0,0}},
        {name=sch("167_main_3_portal"),    vertical_segments={1,1,0,0,0}, can_generate=function(pos)
            return (pos.x%3==0 or pos.z%2==0)
        end},
        {name=sch("167_main_1"),    free_rotation=true, vertical_segments={1,1,1,0,0}},
        {name=sch("167_main_5"),    free_rotation=true, vertical_segments={1,1,0,0,0}},
        {name=sch("167_tall_block_0"), free_rotation=true, vertical_segments={0,0,1,0,0}, skip_above=1},
        {name=sch("167_tall_block_1"), free_rotation=true, vertical_segments={0,0,1,0,0}, skip_above=1},
        {name=sch("167_flat_0"),    free_rotation=true, vertical_segments={0,1,1,0,0}, prevalence=4},
        {name=sch("167_flat_1"),    free_rotation=true, vertical_segments={0,1,1,0,0}, prevalence=4},
        {name=sch("167_flat_2"),    free_rotation=true, vertical_segments={0,1,1,0,0}, prevalence=4},
        {name=sch("167_flat_3"),    free_rotation=true, vertical_segments={0,1,1,0,0}, prevalence=7},
        {name=sch("167_stairs_0"),  free_rotation=true, vertical_segments={1,1,0,0,0}, skip_above=1},
        {name=sch("167_stairs_1"),  free_rotation=true, vertical_segments={1,1,0,0,0}, skip_above=1},
        {name=sch("167_stairs_2"),  free_rotation=true, vertical_segments={1,1,0,0,0}, skip_above=1},
    }
})
br_core.register_biome({
    level = 167,
    desc = "Level 167",
    -- name = "NW",
    danger = 2,
    on_generate = nil,
    segheight = 16,
    prevalence = 3,
    can_generate = function(pos)
        return (pos.x % 73 < 20 and pos.z % 92 < 20)
    end,
    schems = {
        -- {name=sch("16x16x16_null"), vertical_segments={0,0,0,0,0}},
        {name=sch("167_apartment_0"), free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=4},
        {name=sch("167_apartment_0_portal"), free_rotation=true, vertical_segments={1,1,1,1,1}, prevalence=1, can_generate=function(pos)
            return (pos.x%2==0 or pos.z%3==0)
        end},
        {name=sch("167_apartment_1"), free_rotation=true, vertical_segments={1,1,1,1,1}},
        {name=sch("167_apartment_2"), free_rotation=true, vertical_segments={1,1,1,1,1}},
        {name=sch("167_apartment_3"), free_rotation=true, vertical_segments={1,1,1,1,1}},
        {name=sch("167_flat_3"),      free_rotation=true, vertical_segments={1,1,}, prevalence=2, skip_above=4},
        {name=sch("167_stairs_0"),    free_rotation=true, vertical_segments={1,1,1,1,0}, skip_above=1},
        {name=sch("167_stairs_1"),    free_rotation=true, vertical_segments={1,1,1,1,0}, skip_above=1},
        {name=sch("167_stairs_2"),    free_rotation=true, vertical_segments={1,1,1,1,0}, skip_above=1},
    }
})
minetest.register_ore({
    ore_type       = "stratum",
    ore            = "br_core:concrete_ruined_grey",
    wherein        = {"air", "group:liquid", "group:solid"},
    y_min = 167 * br_core.level_height - br_core.offset,
    y_max = 167 * br_core.level_height - br_core.offset,
})
-- br_core.add_loot_area(167, "br_core:loot_area_0")
br_core.add_loot(167, {node_name = "br_core:loot_crate_0"})

br_core.register_level_sound({
    level = 167,
    time = 10,
    scare_timer = {},
    wind_gust_timer = {},
    on_play = function(self, dtime, player)
        local playing = br_core.sound_get_tag("level167_wind", player)
        local light = get_natural_light(player:get_pos(), 0.5) or 0
        if (playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1))
        and light > 13 then
            br_core.sound_play("level167_wind", player, {
                name = "br_wind_noise_WTFPL_Sumianvoice",
                level = 167,
                gain = (br_sounds.master or 1) * (0.5 * math.random() + 0.5) * (light-4) / 10,
                fade = 0.5,
                to_player = player:get_player_name(),
                time = 7,
            })
        elseif playing and #playing >= 1 and light < 13 then
            br_core.sound_stop_tag("level167_wind", player, {
                fade = 0.5,
            })
        end
        playing = br_core.sound_get_tag("level167_traffic", player)
        if playing == nil or #playing < 1
        or (playing and playing[1].fading and #playing <= 1) then
            br_core.sound_play("level167_traffic", player, {
                name = "br_traffic_distant_cc0_klangfabrik",
                level = 167,
                gain = (br_sounds.master or 1) * (0.2 * math.random() + 0.1),
                fade = 0.3,
                pitch = 0.8,
                max_hear_distance = 50,
                to_player = player:get_player_name(),
                time = 14,
            })
        end

        if self.scare_timer[player] == nil then self.scare_timer[player] = math.random(120, 200) end
        self.scare_timer[player] = self.scare_timer[player] - dtime
        playing = br_core.sound_get_tag("level167_beep", player)
        if (self.scare_timer[player] < 0) and (playing == nil or #playing < 1) then
            self.scare_timer[player] = math.random(60, 240)
            br_core.sound_play("level167_beep", player, {
                name = "br_level_0_beep",
                level = 167,
                gain = (br_sounds.master or 1) * 0.2,
                to_player = player:get_player_name(),
                time = 1,
            })
        end


        if self.wind_gust_timer[player] == nil then self.wind_gust_timer[player] = math.random(10, 100) end
        self.wind_gust_timer[player] = self.wind_gust_timer[player] - dtime
        playing = br_core.sound_get_tag("level167_wind_gust", player)
        if ((self.wind_gust_timer[player] < 0) and (playing == nil or #playing < 1))
        and light > 13 then
            self.wind_gust_timer[player] = math.random(10, 100)
            br_core.sound_play("level167_wind_gust", player, {
                name = "br_wind_noise_heavy_WTFPL_Sumianvoice",
                level = 167,
                gain = (br_sounds.master or 1) * 0.2,
                to_player = player:get_player_name(),
                fade = 0.2,
                time = 31,
            })
        elseif playing and #playing >= 0 and light < 13 then
            br_core.sound_stop_tag("level167_wind_gust", player, {
                fade = 0.2,
            })
        elseif self.wind_gust_timer[player] < 0 then
            self.wind_gust_timer[player] = math.random(10, 100)
        end
    end
})
end





br_core.register_level({
    level = 180,
    desc = "Level 180 \'the end\'",
    secret = true,
    danger = 0,
    grace_dist = 5,
    biome = {},
    segsize = 20,
    base_height = 42,
    biome_roughness = 2981.7138976,
    sun = {
        visible = true,
        sunrise_visible = false,
    },
    moon = {
        visible = false,
    },
    clouds = {
        density = 0.6,
        thickness = 2.3,
        ambient = "#fff",
        color = "#ffffff20",
        height = 180 * br_core.level_height - br_core.offset + 42 - 4.37 + 90,
        speed = {x=2.7, z=3}
    },
    sky = {
		base_color = "#ffffff",
		type = "regular",
		clouds = true,
		sky_color = {
			day_sky = "#3383a6" ,
			day_horizon = "#3383a6",
			dawn_sky = "#000" ,
			dawn_horizon = "#000",
			night_sky = "#263965" ,
			night_horizon = "#263965",
			indoors = "#263965",
			fog_sun_tint = "#263965",
			fog_moon_tint = "#263965",
			fog_tint_type = "custom",
		}
	},
})

br_core.register_biome({
    level = 180,
    desc = "Level 180 house",
    danger = 0,
    on_generate = nil,
    prevalence = 1,
    can_generate = function(seg_pos) -- must be deterministic
        if (seg_pos.x%18 == 0) and (seg_pos.z%18 == 0) then
            return true
        else
            return false
        end
    end,
    schems = {
        {name=sch("180_house_0")},
    }
})

br_core.register_biome({
    level = 180,
    desc = "Level 180",
    danger = 0,
    on_generate = nil,
    can_generate = function(pos) -- must be deterministic
        return true
    end,
    prevalence = 20,
    schems = {
        {name=sch("180_open_0")},
    }
})

br_core.register_biome({
    level = 180,
    desc = "Level 180",
    danger = 0,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("180_open_0")},
        {name=sch("180_open_0")},
        {name=sch("180_open_0")},
        {name=sch("180_open_0")},
        {name=sch("180_open_1")},
        {name=sch("180_open_2")},
        {name=sch("180_open_3")},
        {name=sch("180_open_4")},
        {name=sch("180_arch_0")},
        {name=sch("180_path_0")},
    }
})

br_core.register_biome({
    level = 180,
    desc = "Level 180",
    danger = 0,
    on_generate = nil,
    prevalence = 1,
    schems = {
        {name=sch("180_arch_0")},
        {name=sch("180_path_0")},
        {name=sch("180_path_0")},
        {name=sch("180_path_0")},
        {name=sch("180_path_1")},
        {name=sch("180_path_1")},
    }
})
br_core.register_on_changed_level(function(player, from_level, to_level)
    if to_level == 180 and from_level ~= -1 then
        player:set_pos(vector.new(50,br_spawn.get_level_y(180),50))
    end
end)


local t_180_origin_sound = 6
local pl_180 = {}
br_core.register_on_changed_level(function(player, from_level, to_level)
    if to_level == 180 then
        pl_180[player] = true
    elseif from_level == 180 then
        pl_180[player] = nil
    end
end)

minetest.register_globalstep(function(dtime)
    t_180_origin_sound = t_180_origin_sound - dtime
    if t_180_origin_sound < 0 then t_180_origin_sound = t_180_origin_sound + (30 + math.random()*60) else return end
    for player,_ in pairs(pl_180) do repeat
        if core.is_player(player) then
            local pos = player:get_pos()
            local dist = vector.distance(pos, vector.new(-19, pos.y, -16))
            if dist < 80 then break end
            local dir = vector.copy(pos)
            dir.y = 0
            dir = vector.direction(dir, vector.new(
                -math.random() * 32,
                -math.random() * 32,
                -math.random() * 32
            ))
            core.sound_play("br_level_0_beep", {
                pos = pos + vector.multiply(dir, 20),
                to_player = player:get_player_name(),
                max_hear_distance = 999,
                gain = 0.3,
                start_time = 14.8,
            }, true)
        end
    until true end
end)

-- sort levels after all mods loaded
minetest.register_on_mods_loaded(function()
    br_core.sort_levels()
end)




br_core.achievements.register_default(0)
br_core.achievements.register_default(1)
br_core.achievements.register_default(2)
br_core.achievements.register_default(5)
br_core.achievements.register_default(12)
br_core.achievements.register_default(17)
br_core.achievements.register_default(31)
br_core.achievements.register_default(37)
br_core.achievements.register_default(45)
br_core.achievements.register_default(51)
br_core.achievements.register_default(62)
br_core.achievements.register_default(96)
br_core.achievements.register_default(167)
br_core.achievements.register_default(180)
