local mod_name = minetest.get_current_modname()
local mod_path = minetest.get_modpath(mod_name)
local S = minetest.get_translator(mod_name)

aom_igm.registered_settings = {}
aom_igm.registered_settings_array = {}
aom_igm.registered_settings_by_category = {}
aom_igm.player_meta = {}
local pl = aom_igm.player_meta

-- set player meta so that settings are persistent
function aom_igm.save_player(player)
    local meta = player:get_meta()
    meta:set_string("aom_igm:settings", minetest.serialize(pl[player].settings))
end

-- make sure the player is tracked
function aom_igm.check_player(player)
    if not pl[player] then
        local meta = player:get_meta()
        local deser = minetest.deserialize(meta:get_string("aom_igm:settings"), true)
        pl[player] = {
            settings = deser or {},
        }
        if deser then
            for i, v in pairs(pl[player].settings) do
                if not aom_igm.registered_settings[i] then
                    pl[player].settings[i] = nil
                end
            end
        end
        aom_igm.save_player(player)
    end
    return pl[player]
end

-- gets the player setting, or the default if not set
function aom_igm.get_setting(player, settingname, default)
    local setting = aom_igm.registered_settings[settingname]
    if not setting then return default end
    local pi = aom_igm.check_player(player)
    local plval = pi.settings[settingname]
    if plval == nil then plval = setting.default end
    return plval
end

-- don't let player set arbitrary values
local function sanitise_value(value, val_type)
    if type(value) == val_type then return value end
    if val_type == "number" then return tonumber(value) end
    if val_type == "boolean" then return (value == "true") or not (value == "false") end
end

-- set a player setting with a raw value
function aom_igm.set_setting(player, settingname, value)
    local setting = aom_igm.registered_settings[settingname]
    if not setting then return nil end
    local pi = aom_igm.check_player(player)
    if value ~= nil then
        value = sanitise_value(value, setting.type)
    end
    local old_value = pl[player].settings[settingname]
    pi.settings[settingname] = value
    aom_igm.save_player(player)

    local plval = pl[player].settings[settingname]
    if plval == nil then plval = setting.default end
    aom_igm.on_change_setting(player, settingname, plval, old_value)
end

-- register a player setting, using type(default) as a type
function aom_igm.register_setting(name, default)
    if aom_igm.registered_settings[name] then return end -- don't allow re-registering
    local t = type(default)
    aom_igm.registered_settings[name] = {type=t, default=default}
    table.insert(aom_igm.registered_settings_array, {name=name, def=aom_igm.registered_settings[name]})
end

aom_igm.registered_on_change_setting = {}
-- runs when a setting for a player is changed --> callback(player, settingname, new_value, old_value)
function aom_igm.register_on_change_setting(settingname, callback)
    if not aom_igm.registered_on_change_setting[settingname] then aom_igm.registered_on_change_setting[settingname] = {} end
    table.insert(aom_igm.registered_on_change_setting[settingname], callback)
end

aom_igm.registered_on_change_any_setting = {}
-- runs when a setting for a player is changed --> callback(player, settingname, new_value, old_value)
function aom_igm.register_on_change_any_setting(callback)
    table.insert(aom_igm.registered_on_change_any_setting, callback)
end

function aom_igm.on_change_setting(player, settingname, new_value, old_value)
    for i, callback in ipairs(aom_igm.registered_on_change_any_setting) do
        callback(player, settingname, new_value, old_value)
    end
    for i, callback in ipairs(aom_igm.registered_on_change_setting[settingname] or {}) do
        callback(player, settingname, new_value, old_value)
    end
end

function aom_igm.sort_by_category()
    local cat = aom_igm.registered_settings_by_category
    for i, def in ipairs(aom_igm.registered_settings_array) do
        local catname = string.split(def.name, "_")[1] --(ΦωΦ)
        if not cat[catname] then cat[catname] = {} end
        table.insert(cat[catname], def)
    end
end
minetest.register_on_mods_loaded(aom_igm.sort_by_category)


