
# Procedural Event Bus
This is a global set of functions that do what they say on the tin mostly. The exception is `CONDITIONAL`. Signals and listeners mean not only can you sometimes save on constant checks by using one-off events instead, but you can also allow extending, avoid some dependencies, and generally make things more reactive and less constant polling.

## SIGNAL
```lua
SIGNAL(tag, ...)
```

## LISTEN
If any function calls `SIGNAL(tag)`, it will call the function you have passed to `LISTEN(tag, callback)`
```lua
LISTEN(tag, function() end)
```

## CONDITIONAL
If any signal returns `==false`, it will return `false` immediately. Otherwise, if one or more callback returns `true` it returns true, and if nothing gets returned then it returns `false`. 

```lua
local mybool = CONDITIONAL(tag, ...)
```

Example:
```lua
LISTEN("can_be_damaged", function(self)
    if self._invulnerable then
        return false
    else
        return true
    end
end)

[...]

local is_damaged = CONDITIONAL("can_be_damaged", self)
```
This is good for conditions that need to be able to be extended, such as when an object should enter a "sleep" state, or to test if it is allowed to take some action.


## Getting a standalone copy of event bus
It is only recommended for internal use where you know a feature / mod will not need to be accessed externally, but it is possible to get your own copy of the eventbus. This is not really necessary since Lua tables are super fast, but you can do it if you really super duper want to. This is not OOP, it returns a full copy of the functions with their own scope, so it is not memory optimised or anything. Don't make 1,000,000 of them please.

You can do this by getting a copy:
```lua
local signal_copy = SIGNALS.new()
signal_copy.SIGNAL("on_something_happen", 38.2, 4)
```
Or you can insert the functions into an existing table (recommended):
```lua
my_mod = {}
SIGNALS.new(my_mod)
my_mod.SIGNAL("on_something_happen", 38.2, 4)
```
