
# Purpose
1. provide an item entity that is extensible so you don't need to override stuff and make jank workarounds
2. make a `builtin` alternative but in a permissive license

It is not "to be compatible with existing item entity mods". Item entity mods that override the item entity will likely completely eradicate or severely break this mod.

# Implemented behaviors
- physics, gravity
- sliding on slippery (and slight sliding on everything else)
- items getting pushed out of solids
- merging with nearby items
- items laying flat (optional)
- floating in water (optional)
- items pushed by liquids (optional)
- items burning in `fire` or `burns_items` group nodes (optional)
- picking up nearby items (optional)
- makes sounds when hitting the ground (optional)

Default behaviour is similar to builtin, but with extra features like floating in water and burning in fire.

# NOTE
All listeners MUST return `"_cancel"` if using `self.object:remove()`. This is what tells the system that it should not run more code for this signal call. 

# How it works
All interactions and hooks are using signals, so that you can stick new bits on without having to override the item entity. You can create your own signal and listener at any time. Signals that have no listeners just terminate with no effect, and do not throw an error. You don't need to use these listeners all the time like this mod tries to however, you could use procedural code just fine, but these are how you hook into the item entity's callbacks without having to override anything.

`aom_item_entity.LISTEN("mytag", ...)` is the same as `aom_item_entity.register_on_mytag(...)` would be in most Minetest dev.

# Variables and settings
```lua
-- uses aom_settings if available, else the built in settings table
-- gets a value from settings, else default
aom_item_entity.get(player, key, default)
-- sets a setting; if present in aom_settings, sets that key there
aom_item_entity.set(player, key, value)
```

# List of built in signals
```lua
-- MAIN BUILTIN SIGNALS
------------------------
-- called every frame for each item entity
aom_item_entity.LISTEN("on_step", function(self, dtime, moveresult, ...)
end)
-- item is itemstring or ItemStack
aom_item_entity.LISTEN("set_item", function(self, item, ...)
end)
-- when starting out with staticdata; don't use unless you really need it
aom_item_entity.LISTEN("on_activate", function(self, staticdata, dtime_s)
end)
-- data is a table of data to save
-- vars listed in `aom_item_entity.staticdata_load_list` are already loaded by this point
aom_item_entity.LISTEN("get_staticdata", function(self, data)
end)
-- called when a player or other thing punches / attacks this
aom_item_entity.LISTEN("on_punch", function(self, puncher, time_from_last_punch, tool_capabilities, dir, damage, ...)
end)
-- makes the item able to phase through solids
aom_item_entity.LISTEN("enable_physics", function(self)
end)
-- stops the item phasing through solids
aom_item_entity.LISTEN("disable_physics", function(self)
end)

-- EXTRA BUILTIN SIGNALS
------------------------
-- called while item is moving to player to be picked up
aom_item_entity.LISTEN("on_magnet", function(self, dtime)
end)
-- tells item to get picked up by this player
aom_item_entity.LISTEN("set_magnet", function(self, player)
end)
-- return == true to sleep, == false to force no sleep
aom_item_entity.LISTEN("should_sleep", function(self, dtime, moveresult)
end)
-- means this item has entered "don't bother" mode
aom_item_entity.LISTEN("on_sleep", function(self)
end)
-- means item has gone back to doing calculations and movement again
aom_item_entity.LISTEN("on_wake", function(self)
end)
-- means item has hit the ground after being in the air (called once)
aom_item_entity.LISTEN("on_landed", function(self, dtime, moveresult, ...)
end)

-- EXTRA SIGNALS used for specific things
-----------------------------------------
-- when set_properties is used, this is called
aom_item_entity.LISTEN("get_custom_params", function(self)
end)
-- when this item has just had another item merge into it; `self` is the resulting merged item that has the full stack
-- NOTE: is called after `set_item`
aom_item_entity.LISTEN("on_merged_with", function(self, object)
end)
-- when items lay flat / hit floor if enabled
aom_item_entity.LISTEN("item_stick", function(self)
end)
-- when item stops laying flat
aom_item_entity.LISTEN("item_unstick", function(self)
end)
-- for lay flat items that rotate mid air
aom_item_entity.LISTEN("do_idle_rotation", function(self, dtime)
end)
```

# Item definitions
There are some callbacks when an item hits certain events.
```lua
local itemdef = {
    -- these are all `no return` or `"_cancel"` to stop further callbacks running
    -- called every frame for every item entity of this item
    _on_item_step = function (self, dtime, moveresult, ...)
    end,
    -- called once when the item is first spawned in
    _on_item_drop = function(self, itemstack, dropper)
    end,
    -- called once AFTER the item enters the inventory, and BEFORE it calls `object:remove()`
    -- `itemstack` is the itemstack that was added to the inventory, but changes to this itemstack don't have effect
    _on_item_pickup = function(self, itemstack, picker)
    end,
}
```
