-- carts.lua

local cart_entity = {
	initial_properties = {
		physical = false, -- otherwise going uphill breaks
		collisionbox = {-0.5, -0.5, -0.5, 0.5, 0.5, 0.5},
		selectionbox = {-9/16, -0.5, -15/16, 9/16, 0.5, 15/16, rotate=true},
		visual = "mesh",
		mesh = "mg_cart.glb",
		visual_size = {x=10, y=10},
		textures = {"mg_cart_wood.png", "mg_cart_frame.png", "mg_cart_wood.png"},
	},
	velocity = {x=0, y=0, z=0},
	old_dir = {x=1, y=0, z=0},
	old_pos = nil,
	old_switch = 0,
	railtype = nil,
	attached_items = {},
	cargo = ""
}

local function update_lvl(cart)
	local stack = ItemStack(cart.cargo)

	local texture = "mg_cart_wood.png"
	local ore = mg_core.ores.registered_ores[stack:get_name()]
	if ore ~= nil then
		local nodename = mg_core.ores.levels[ore.level].node
		local node = core.registered_nodes[nodename]
		local item = core.registered_craftitems[stack:get_name()]
		texture = node.tiles[1] .. "^" .. item.inventory_image
	end

	local frame = math.round((stack:get_count() / stack:get_stack_max()) * 6)
	--core.log(frame)
	cart.object:set_animation({x=frame, y=frame}, 1, 0)
	cart.object:set_properties({
		textures = {"mg_cart_wood.png", "mg_cart_frame.png", texture},
	})
end

function cart_entity:on_rightclick(clicker)
	local item = clicker:get_wielded_item()
	local cargo = ItemStack(self.cargo)
	local c_name = cargo:get_name()

	local is_ore = false
	if mg_core.ores.registered_ores[item:get_name()] ~= nil then
		is_ore = true
	end

	if item:get_name() == "" then -- take the item from cargo and give it to the player
		--core.log("no item, taking cargo")
		clicker:get_inventory():add_item("main", cargo)
		self.cargo = ""
		update_lvl(self)
	elseif c_name == item:get_name() and is_ore then
		--core.log("cargo of same type, adding to cart")
		clicker:get_inventory():remove_item("main", item)
		cargo:set_count(cargo:get_count() + item:get_count())
		self.cargo = c_name .. " " .. cargo:get_count()
		update_lvl(self)
	elseif c_name == "" and is_ore then
		--core.log("no cargo already loaded, adding to cart")
		clicker:get_inventory():remove_item("main", item)
		self.cargo = item:get_name() .. " " .. item:get_count()
		update_lvl(self)
	end
	--core.show_formspec(clicker:get_player_name(), "cart_inventory", cart_formspec(self, clicker))
end

function cart_entity:on_activate(staticdata, dtime_s)
	self.object:set_armor_groups({immortal=1})
	if string.sub(staticdata, 1, string.len("return")) ~= "return" then
		return
	end
	local data = core.deserialize(staticdata)
	if type(data) ~= "table" then
		return
	end
	self.railtype = data.railtype
	self.old_dir = data.old_dir or self.old_dir
	if data.cargo ~= nil then
		self.cargo = data.cargo
	else
		self.cargo = ""
	end
	update_lvl(self)
end

function cart_entity:get_staticdata()
	return core.serialize({
		railtype = self.railtype,
		old_dir = self.old_dir,
		cargo = self.cargo
	})
end

function cart_entity:on_punch(puncher, time_from_last_punch, tool_capabilities, direction)
	local pos = self.object:get_pos()
	local vel = self.object:get_velocity()
	if not self.railtype or vector.equals(vel, {x=0, y=0, z=0}) then
		local node = core.get_node(pos).name
		self.railtype = core.get_item_group(node, "connect_to_raillike")
	end
	-- Punched by non-player
	if not puncher or not puncher:is_player() then
		local cart_dir = carts:get_rail_direction(pos, self.old_dir, nil, nil, self.railtype)
		if vector.equals(cart_dir, {x=0, y=0, z=0}) then
			return
		end
		if not vector.equals(self.velocity, vector.zero()) then -- stop the cart if it's moving
			self.velocity = vector.zero()
			self.punched = false
		else -- cart can only be pushed from a stop
			self.velocity = vector.multiply(cart_dir, 2)
			self.punched = true
		end
		return
	end
	-- Player digs cart by sneak-punch
	if puncher:get_player_control().sneak then
		-- Pick up cart
		local inv = puncher:get_inventory()
		if not core.is_creative_enabled(puncher:get_player_name())
				or not inv:contains_item("main", "mg_carts:cart") then
			local leftover = inv:add_item("main", "mg_carts:cart")
			-- If no room in inventory add a replacement cart to the world
			if not leftover:is_empty() then
				core.add_item(self.object:get_pos(), leftover)
			end
		end
		if self.cargo ~= "" then
			local stack = ItemStack(self.cargo)
			local leftover = inv:add_item("main", stack)
			core.item_drop(leftover, puncher, pos)
		end
		self.object:remove()
		return
	end
	-- Player punches cart to alter velocity
	--[[if puncher:get_player_name() == self.driver then
		if math.abs(vel.x + vel.z) > carts.punch_speed_max then
			return
		end
	end]]

	if not vector.equals(self.velocity, vector.zero()) then
		self.old_dir = self.velocity
		self.velocity = vector.zero()
		self.punched = false
	else
		local punch_dir = carts:velocity_to_dir(puncher:get_look_dir())
		punch_dir.y = 0
		local cart_dir = carts:get_rail_direction(pos, punch_dir, nil, nil, self.railtype)
		if vector.equals(cart_dir, {x=0, y=0, z=0}) then
			return
		end

		local punch_interval = 1
		-- Faulty tool registrations may cause the interval to be set to 0 !
		if tool_capabilities and (tool_capabilities.full_punch_interval or 0) > 0 then
			punch_interval = tool_capabilities.full_punch_interval
		end
		time_from_last_punch = math.min(time_from_last_punch or punch_interval, punch_interval)
		local f = 2 * (time_from_last_punch / punch_interval)

		self.velocity = vector.multiply(cart_dir, f)
		self.old_dir = cart_dir
		self.punched = true
	end
end

local function rail_on_step_event(handler, obj, dtime)
	if handler then
		handler(obj, dtime)
	end
end

-- sound refresh interval = 1.0sec
local function rail_sound(self, dtime)
	if not self.sound_ttl then
		self.sound_ttl = 1.0
		return
	elseif self.sound_ttl > 0 then
		self.sound_ttl = self.sound_ttl - dtime
		return
	end
	self.sound_ttl = 1.0
	if self.sound_handle then
		local handle = self.sound_handle
		self.sound_handle = nil
		core.after(0.2, core.sound_stop, handle)
	end
	local vel = self.object:get_velocity()
	local speed = vector.length(vel)
	if speed > 0 then
		self.sound_handle = core.sound_play(
			"mg_carts_cart_moving", {
			object = self.object,
			gain = (speed / carts.speed_max) / 2,
			loop = true,
		})
	end
end

local function get_railparams(pos)
	local node = core.get_node(pos)
	return carts.railparams[node.name] or {}
end

local v3_len = vector.length
local function rail_on_step(self, dtime)
	local vel = self.object:get_velocity()
	if self.punched then
		vel = vector.add(vel, self.velocity)
		self.object:set_velocity(vel)
		self.old_dir.y = 0
	elseif vector.equals(vel, {x=0, y=0, z=0}) then
		return
	end

	local pos = self.object:get_pos()
	local dir = carts:velocity_to_dir(vel)
	local dir_changed = not vector.equals(dir, self.old_dir)
	local update = {}

	if self.old_pos and not self.punched and not dir_changed then
		local flo_pos = vector.round(pos)
		local flo_old = vector.round(self.old_pos)
		if vector.equals(flo_pos, flo_old) then
			-- Do not check one node multiple times
			return
		end
	end

	local ctrl, player

	-- Get player controls
	if self.driver then
		player = core.get_player_by_name(self.driver)
		if player then
			ctrl = player:get_player_control()
		end
	end

	local stop_wiggle = false
	if self.old_pos and not dir_changed then
		-- Detection for "skipping" nodes (perhaps use average dtime?)
		-- It's sophisticated enough to take the acceleration in account
		local acc = self.object:get_acceleration()
		local distance = dtime * (v3_len(vel) + 0.5 * dtime * v3_len(acc))

		local new_pos, new_dir = carts:pathfinder(
			pos, self.old_pos, self.old_dir, distance, ctrl,
			self.old_switch, self.railtype
		)

		if new_pos then
			-- No rail found: set to the expected position
			pos = new_pos
			update.pos = true
			dir = new_dir
		end
	elseif self.old_pos and self.old_dir.y ~= 1 and not self.punched then
		-- Stop wiggle
		stop_wiggle = true
	end

	local railparams

	-- dir:         New moving direction of the cart
	-- switch_keys: Currently pressed L(1) or R(2) key,
	--              used to ignore the key on the next rail node
	local switch_keys
	dir, switch_keys = carts:get_rail_direction(
		pos, dir, ctrl, self.old_switch, self.railtype
	)
	dir_changed = not vector.equals(dir, self.old_dir)

	local acc = 0
	if stop_wiggle or vector.equals(dir, {x=0, y=0, z=0}) then
		dir = vector.new(self.old_dir)
		vel = {x = 0, y = 0, z = 0}
		local pos_r = vector.round(pos)
		if not carts:is_rail(pos_r, self.railtype)
				and self.old_pos then
			pos = self.old_pos
		elseif not stop_wiggle then
			-- End of rail: Smooth out.
			pos = pos_r
			dir_changed = false
			dir.y = 0
		else
			pos.y = math.floor(pos.y + 0.5)
		end
		update.pos = true
		update.vel = true
	else
		-- Direction change detected
		if dir_changed then
			vel = vector.multiply(dir, math.abs(vel.x + vel.z))
			update.vel = true
			if dir.y ~= self.old_dir.y then
				pos = vector.round(pos)
				update.pos = true
			end
		end
		-- Center on the rail
		if dir.z ~= 0 and math.floor(pos.x + 0.5) ~= pos.x then
			pos.x = math.floor(pos.x + 0.5)
			update.pos = true
		end
		if dir.x ~= 0 and math.floor(pos.z + 0.5) ~= pos.z then
			pos.z = math.floor(pos.z + 0.5)
			update.pos = true
		end

		-- Slow down or speed up..
		acc = dir.y * -4.0

		-- Get rail for corrected position
		railparams = get_railparams(pos)

		-- no need to check for railparams == nil since we always make it exist.
		local speed_mod = railparams.acceleration
		if speed_mod and speed_mod ~= 0 then
			-- Try to make it similar to the original carts mod
			acc = acc + speed_mod
		else
			-- Handbrake or coast
			if ctrl and ctrl.down then
				acc = acc - 3
			else
				acc = acc - 0.4
			end
		end
	end

	-- Limit cart speed
	local vel_len = vector.length(vel)
	if vel_len > carts.speed_max then
		vel = vector.multiply(vel, carts.speed_max / vel_len)
		update.vel = true
	end
	if vel_len >= carts.speed_max and acc > 0 then
		acc = 0
	end

	self.object:set_acceleration(vector.multiply(dir, acc))

	self.old_pos = vector.round(pos)
	self.old_dir = vector.new(dir)
	self.old_switch = switch_keys

	if self.punched then
		-- Collect dropped items
		for _, obj_ in pairs(core.get_objects_inside_radius(pos, 1)) do
			local ent = obj_:get_luaentity()
			-- Careful here: physical_state and disable_physics are item-internal APIs
			if ent and ent.name == "__builtin:item" and ent.physical_state then
				ent:disable_physics()
				obj_:set_attach(self.object, "", {x=0, y=0, z=0}, {x=0, y=0, z=0})
				self.attached_items[#self.attached_items + 1] = obj_
			end
		end
		self.punched = false
		update.vel = true
	end

	railparams = railparams or get_railparams(pos)

	if not (update.vel or update.pos) then
		rail_on_step_event(railparams.on_step, self, dtime)
		return
	end

	local yaw = 0
	if dir.x < 0 then
		yaw = 0.5
	elseif dir.x > 0 then
		yaw = 1.5
	elseif dir.z < 0 then
		yaw = 1
	end
	self.object:set_yaw(yaw * math.pi)
	
	local rot = 0
	if dir.y == -1 then
		rot = -0.75

	elseif dir.y == 1 then
		rot = 0.75
	end

	local r = self.object:get_rotation()
	self.object:set_rotation(vector.new(rot, r.y, r.z))
	--if rot ~= 0 then self.object:set_pos(vector.offset(self.object:get_pos(), 0, 0.5, 0)) end
	--[[
	local anim = {x=0, y=0}
	if dir.y == -1 then
		anim = {x=2, y=2}
	elseif dir.y == 1 then
		anim = {x=4, y=4}
	end
	local rot = self.object:get_rotation()
	self.object:set_rotation(vector.new(rot.x, rot.y, rot.z))
	self.object:set_animation(anim, 1, 0)
	]]

	if update.vel then
		self.object:set_velocity(vel)
	end
	if update.pos then
		if dir_changed then
			self.object:set_pos(pos)
		else
			self.object:move_to(pos)
		end
	end

	-- call event handler
	rail_on_step_event(railparams.on_step, self, dtime)
end

function cart_entity:on_step(dtime)
	rail_on_step(self, dtime)
	rail_sound(self, dtime)
end

core.register_entity("mg_carts:cart", cart_entity)

core.register_craftitem("mg_carts:cart", {
	description = "Cart" .. "\n" .. "(Sneak+Click to pick up",
	inventory_image = "mg_cart.png",
	wield_image = "mg_cart.png",
	on_place = function(itemstack, placer, pointed_thing)
		local under = pointed_thing.under
		local node = core.get_node(under)
		local udef = core.registered_nodes[node.name]
		if udef and udef.on_rightclick and
				not (placer and placer:is_player() and
				placer:get_player_control().sneak) then
			return udef.on_rightclick(under, node, placer, itemstack,
				pointed_thing) or itemstack
		end

		if pointed_thing.type ~= "node" then
			return
		end
		local c = nil
		if carts:is_rail(pointed_thing.under) then
			c = core.add_entity(pointed_thing.under, "mg_carts:cart")
		elseif carts:is_rail(pointed_thing.above) then
			c = core.add_entity(pointed_thing.above, "mg_carts:cart")
		else
			return
		end

		local inv = core.create_detached_inventory(tostring(c:get_guid()), {
			on_put = function(inv, listname, index, stack, player)
				local sz = stack:get_count() / stack:get_stack_max()
				local frame = math.round(sz * 7)
				c:set_animation({x=frame,y=frame}, 1, 0)

				return stack:get_count()
			end,
			on_move = function(inv, from_list, from_index, to_list, to_index, count, player)
				local stack = inv:get_stack(from_list, from_index)
				local sz = stack:get_count() / stack:get_stack_max()
				local frame = math.round(sz * 7)
				c:set_animation({x=frame,y=frame}, 1, 0)

				return count
			end,
		})
		inv:set_size("cargo", 1)

		core.sound_play({name = "default_place_node_metal", gain = 0.5},
			{pos = pointed_thing.above}, true)

		local player_name = placer and placer:get_player_name() or ""
		if not core.is_creative_enabled(player_name) then
			itemstack:take_item()
		end
		return itemstack
	end,
})