--[[
(C) TPH/tph9677/TubberPupperHusker/TubberPupper/Damotrix
MIT License
https://opensource.org/license/mit

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

events_api = {
    --- checks if provided value is a null string character \0
    --- @param val any value to be determined whether or not is equal to nil (checks if `nil` or null character)
    is_nil = function(val)
        return val == nil or val == '\0'
    end,
    --- converts variable arguments into an iteratable table, converts nil to null string character
    convert_variable_args = function(...)
        -- add values to this table to return
        local cnv = {} -- converted
        -- iterates over provided arguments
        for i=1, select('#', ...) do
            local val = select(i, ...) -- gets value at position of provided arguments
            -- convert to null string character if nil (so that it actually goes into the table)
            cnv[#cnv + 1] = val == nil and '\0' or val
        end
        -- return converted table
        return cnv
    end
}

-- localize
local is_nil = events_api.is_nil
local cva = events_api.convert_variable_args


--- function for creating events and their handling
---@param name string name of the event, do NOT include "on_" in the name if you opt for autosetup
---@param def? table optional definition for additional automated setup
---@return table eventdata provides data.funcs (table), data.handler + data.register functions for handling events. Provides data.name as the name the event was registered with.
events_api.create = function(name, def)
    if type(name) ~= "string" then
        error("events_api: attempted to register event without a string name.")
    end
    local data = {
        name = name, -- name of the event
        -- funcs should be an array table, use `table.remove` to reliably remove certain functions
        funcs = {} -- list of functions to be ran by `data.handler`
    }
    -- what should be called to add functions to `data.funcs`
    -- should be added to your mod global as `mymod.register_on_myevent`
    data.register = function(func)
        if type(func) ~= "function" then return end -- do an error
        if type(data.funcs) ~= "table" then
            error("events_api: registration function ran for '"..(type(data.name) == "string" and data.name or name)..
              "', however there exists no funcs table. Fatal error.")
        end
        data.funcs[#data.funcs + 1] = func
        -- function for disconnecting from list of functions
        return function()
            for ind,ffunc in ipairs(data.funcs) do -- index, found func
                -- if we have the same ID
                if ffunc == func then
                    return table.remove(data.funcs, ind)
                end
            end
        end
    end
    -- handling the running of each and every function in `data.funcs`
    -- provided parameters will be ran to each and every function
    data.handler = function(...)
        if type(data.funcs) ~= "table" then
            error("events_api: handler ran for '"..(type(data.name) == "string" and data.name or name)..
              "', however there exists no funcs table. Fatal error.")
        end
        -- if `prior_to_each_handler_func` specified, run a function that lets you modify the handler parameters
        -- prior to each function being ran
        -- similar to `after_event`, provides data of function as first parameter (data.funcs id and func itself)
        -- SHOULD RETURN ALL PARAMETERS IN THE SAME ORDER!
        local ptehf = data.prior_to_each_handler_func
        -- if `after_event` specified, returns an array of tables with an id correlating to a funcs index,
        -- and a returned of data returned by function
        -- this will run even if all the functions return nil
        local after_event = data.after_event
        -- adds any return result from the function to an array (if after_event)
        local returns = after_event and {} -- initialize only if function exists
        -- iterates over, calling each function
        for id,func in ipairs(data.funcs) do
            -- oh you want to do quite the intervention
            local prefunc = ptehf and cva(
              ptehf( {id=id, func=func}, ... )
            )
            -- time to store what our funcs are doing
            if after_event then
                -- what we got from func returns
                local got
                -- if we ran prior purification, give unpacked prefunc stuff
                if prefunc then
                    got = cva( func( unpack(prefunc) ) )
                -- did not do any purification
                else
                    got = cva( func( ... ) )
                end
                -- store return(ed) results, function index (id), and actual function
                table.insert(returns, {returned=got, id=id, func=func} )
            -- call function normally
            else
                -- if we ran prior purification, give unpacked prefunc stuff
                if prefunc then
                    func(unpack(prefunc))
                -- else run with the given parameters
                else
                    func(...)
                end
            end
        end
        -- run after event if exists
        -- first is all the functions, then the rest is the events' parameters
        -- "returns" should be called "funcs" in your function
        if after_event then
            after_event(returns, ...)
        end
    end
    -- if setup wanted and globals given for setup
    if type(def) == "table" then
        -- add to provided funcs table
        -- optional table, can be used to add data.funcs to a provided table
        if type(def.funcsgroup) == "table" then
            def.funcsgroup[name] = data.funcs
        end
        -- add to provided handlers table
        -- optional table, can be used to add data.handler to a provided table
        if type(def.handlers) == "table" then
            def.handlers[name] = data.handler
        end
        -- create function for registration in mod's global
        if type(def.global) == "table" then
            -- add register function
            def.global["register_on_"..name] = data.register
        end
        -- got function for doing stuff on event end
        if type(def.after_event) == "function" then
            data.after_event = def.after_event
        end
        -- got function handling changing the parameters for each ran function callback
        if type(def.prior_to_each_handler_func) == "function" then
            data.prior_to_each_handler_func = def.prior_to_each_handler_func
        end
    end
    -- give data a special metatable to permit calling the data to run the handler
    data = setmetatable(data, {
        __call = function(t, ...) -- t is the data table, we can ignore it
            if type(data.handler) ~= "function" then return end -- do an error in the future
            return data.handler(...)
        end
    })
    -- return created event data
    return data
end