--[[
 exile_songs;
 Plays ambient music for Exile (or any game technically, though should be for Exile!!!)
 tracks specifically crafted for the environment of Exile
 
 even will play any music you add to /sounds/ ! So feel free to customize your experience :3
 
    Copyright (C) 2024 TPH (see readme.txt for more info)

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
--]]

local is_v4 = minetest.get_modpath("tgcr") and true

local random = math.random

exile_snow = {
    gravity = 9.8,
    -- snowballs
    power_start = 7,
    power_max = 40,
    power_add = 0.5,
    charge_up_delay = 0.05
}
exile_snow.gravity = 9.8 -- 9.8
exile_snow.term_vel = math.sqrt(2*exile_snow.gravity*20)

-- p1 is thrower, p2 is target
local messages = {
    "p1 absolutely DECIMATED p2 with a snowball!",
    "p1 nailed p2 right in the noggin with a snowball!",
    "p2's brain is freezing over from p1's snowball, ouch!",
    "p2 better hope that p1's snowball didn't have ice!",
    "p2 got a full serving of snowball from p1!",
    "p2 was unexpectedly whacked in the face by p1's snowball!",
    "p2's face was sent to the ice realm by p1's snowball!",
    "p2 was nearly knocked unconscious by p1's vicious snowball!",
    "p2 was quickly cooled down by a hurling snowball from p1!",
    "p2 couldn't even open their eyes before a snowball from p1 did it for them!",
    "p1's snowball told p2 to chill out",
    "p1 thought p2 needed a snowday to enjoy",
    -- p1 excluded
    "p2 got a face full of snow!"
}

local function sound_play(pos, data)
    if type(data) ~= "table" then return end -- not data we can use
    if not data.name then return end -- no name
    -- ensure no accidental overwrite
    data = table.copy(data)
    -- range function
    local function get_range(value)
        -- if value is a table and its index 1 and 2 are numbers then
        --   return a randomized value between them
        return type(value) == 'table' and type(value[1]) == "number"
            and type(value[2]) == "number" and
            (value[1]+random()*(value[2]-value[1]) ) or value
    end
    -- figure out range of gain, fade, pitch, and hear distance
    data.gain = get_range(data.gain)
    data.fade = get_range(data.fade)
    data.pitch = get_range(data.pitch)
    data.max_hear_distance = get_range(data.max_hear_distance)
    -- use position or object
    if type(pos) == "table" then
        data.pos = pos
    else
        data.object = pos
    end
    return core.sound_play(data.name, data)
end

-- custom functionality for detecting if player is in creative
-- used to check if we should deplete
local creative_mode = minetest.settings:get_bool("creative_mode")
local function player_in_creative(player)
  -- get the player by name if string
  player = type(player) == "userdata" and player or type(player) == "string" and core.get_player_by_name(player)
  -- if player is a player...
  if core.is_player(player) then
      if core.check_player_privs(player,"creative") or creative_mode then
          return true
      end
  end
  return false
end

core.register_entity("exile_snow:snowball", {
    initial_properties = {
        visual = "sprite",
        visual_size = {x=0.5, y=0.5},
        collisionbox = {-0.15, -0.15, -0.15, 0.15, 0.15, 0.15},
        textures = {"exile_snow_ball.png"},
        physical = true,
        collide_with_objects = true,
        static_save = false,
    },
    _desc = "Snowball Coming At Ya",
    sounds = {
        hit = {name = "exile_snow_ball_hit", gain = 0.5, pitch = {0.85, 1.1}}
    },
    -- what happens when a snowball meets an object
    snowball_hit = function(self, target)
        local is_player = core.is_player(target)
        -- check if we're a player and if there's an owner_forcefield
        if is_player and self.owner_forcefield then
            -- if time is over the limit of owner_forcefield, remove
            if core.get_server_uptime() > self.owner_forcefield then
                self.owner_forcefield = nil
            -- protected from our own snowball!
            -- check if thrower value exists and if target's name equals it
            elseif self.thrower and self.thrower:get_player_name() == target:get_player_name() then
                return
            end
        end
        local pos = self.object:get_pos()
        pos = {x = pos.x, y = pos.y, z = pos.z} -- WE HAVE TO DO THIS BECAUSE PARTICLE EMITTERS HAVE THE BIG STOOPID!
        local hit_prtc = { -- hit particles
            pos = pos,
            amount = 3,
            time = 0.1, -- how long it takes emitter to fully spawn all particles
            minxexptime = 4,
            maxexptime = 4,
            collisiondetection = true,
            -- fly up on collision
            minvel = {x = -2, y = 1, z = -2},
            maxvel = {x = 2, y = 3.5, z = 2},
            minacc = {x = 0, y = -exile_snow.gravity, z = 0},
            minsize = 0.3,
            maxsize = 0.6,
            drag = {x = 3, y = 0, z = 3} -- lower velocity to the sides
        }
        hit_prtc.maxacc = hit_prtc.minacc
        local bounds = 3 -- used in sheet'ing the desired image
        -- spawn 10 particle emitters per hit for a proper randomized image assortment
        for i = 1, 10 do
            -- random bounds
            -- subtract 1 because that's how the function works
            local rbounds = {random(0,bounds-1), random(0,bounds-1)} -- x, y
            -- faster to use concat instead of ..
            hit_prtc.texture = table.concat({
                "nodes_nature_snow.png^[sheet:", bounds, "x", bounds, ":", rbounds[1], ",", rbounds[2]
            })
            core.add_particlespawner(hit_prtc)
        end
        -- play sound at pos
        sound_play(pos, self.sounds.hit)
        -- show snow covered hud if player and not dead
        -- thrown at face or from above
        if is_player and target:get_hp() > 0 then
            local pprops = target:get_properties() -- player properties
            local ppos = target:get_pos()
            local selfpos = self.object:get_pos()
            -- above head
            local ontop = ppos.y + (pprops.eye_height * 1.15)
            -- get about mid-face height
            ppos.y = ppos.y + (pprops.eye_height * 0.95)
            local canhud = selfpos.y > ontop
            -- do a more complex calculation if above is untrue (thrown at face perhaps?)
            -- only check if y is less than 0.65 units apart
            if not canhud and math.abs(selfpos.y - ppos.y) < 0.65 then
                local infront = vector.add(ppos, vector.multiply(target:get_look_dir(), 0.25))
                -- at our face!
                if vector.distance(selfpos, infront) < 0.6 then
                    canhud = true
                end
            end
            -- snow covered view
            if canhud then
                local despawn_time = 3 -- how long until vanishing begins
                -- creates 12 transparent huds to make a "full" opaque image
                for i=0, 1.2, 0.1 do
                    local hud = target:hud_add({
                        hud_elem_type = "image",
                        text = "exile_snow_snowhud.png",
                        position = {x = 0.5, y = 0.5},
                        scale = {x=-100, y=-100}
                    })
                    -- that are then gradually removed one after another to create a vanishing effect
                    minetest.after(despawn_time + i, function()
                        -- incase player leaves
                        if type(target) == "userdata" then
                            target:hud_remove(hud)
                        end
                    end)
                end
                if core.is_player(self.thrower) then
                    local pnames = {self.thrower:get_player_name(), target:get_player_name()}
                    local basecolor = core.get_color_escape_sequence("#69cde4") -- somewhat desaturated cyan
                    -- hit ourself
                    if pnames[1] == pnames[2] then
                        core.chat_send_player(pnames[1], table.concat({basecolor, "Congrats, you hit yourself!"}))
                    -- we didn't hit ourselves, we hit someone ELSE muhaha!
                    else
                        local message = messages[random(1,#messages)]
                        local snowcolor = core.get_color_escape_sequence("#ddf4f6") -- frosty ice colour
                        -- use table.concat to more efficiently merge color sequences
                        message = table.concat({basecolor, message})
                        pnames[1] = table.concat(
                          {core.get_color_escape_sequence("#1bed1f"), pnames[1], basecolor}) -- greenish colour
                        pnames[2] = table.concat(
                          {core.get_color_escape_sequence("#c8061e"), pnames[2], basecolor}) -- reddish colour
                        message = message:gsub("p1", pnames[1])
                        message = message:gsub("p2", pnames[2])
                        message = message:gsub("snowball", table.concat({snowcolor, "snowball", basecolor}))
                        -- all shall know your defeat
                        core.chat_send_all(message)
                    end
                end
            end
        end
        -- delete snowball
        self.object:remove()
    end,
    -- destroy snowball if punched
    on_punch = function(self, puncher)
        self:snowball_hit()
    end,
    on_step = function(self, dtime, moveresult)
        -- touched something
        if moveresult and (moveresult.touching_ground or moveresult.collides) then
            local collider = moveresult.collisions and moveresult.collisions[1] or {}
            return self:snowball_hit(collider.type == "node" and collider.node_pos or
              collider.object)
        end
        -- otherwise accelerate downward
        local acc = self.object:get_acceleration()
        -- if y velocity is greater than terminal velocity
        if acc.y > -exile_snow.term_vel then
            -- decrease y acceleration by a tenth of gravity
            -- ensure value is multiplied by dtime divided by 5 centiseconds
            acc.y = acc.y - ( (exile_snow.gravity/10) * (dtime/0.05) )
            -- if below term vel set to it
            if acc.y < -exile_snow.term_vel then
                acc.y = -exile_snow.term_vel
            end
            self.object:set_acceleration(acc)
        end
    end
})

local is_throwing = {}

local function remove_from_throwing(user)
    is_throwing[user:get_player_name()] = nil
end

local function snowball_charge_up(itemstack, user, info)
    -- set up info, and modify
    info = info or {init=true} -- set init for first initialization
    -- don't run code if we're already throwing a snowball
    if info.init then
        local pname = user:get_player_name()
        if is_throwing[pname] then return end
        is_throwing[pname] = true
    end
    info.power = info.power or exile_snow.power_start - 1 -- we add to in next line
    -- ensure at max if over otherwise add
    info.power = info.power >= exile_snow.power_max and exile_snow.power_max or info.power + exile_snow.power_add
    -- used for verifying if same snowball
    info.index = info.index or user:get_wield_index()
    info.itemname = info.itemname or itemstack:get_name()
    -- check if we're using the same snowball (as long as not on first initialization)
    -- TODO: see about making some form of callback for a failed snowball?
    if not info.init then
        if user:get_wield_index() ~= info.index then return remove_from_throwing(user) end -- moved slots
        local citem = user:get_wielded_item()
        if citem:get_name() ~= info.itemname then return remove_from_throwing(user) end -- dropped snowballs or they got changed somehow
    else
        info.init = nil -- we've finished initialization, remove from info
    end
    -- get player's control
    local ctrl = user:get_player_control()
    -- we're still amping up!
    if ctrl.LMB then
        core.after(exile_snow.charge_up_delay, snowball_charge_up, itemstack, user, info)
        return
    end
    -- we've released! bombs away
    remove_from_throwing(user)
    local init_props = user:get_properties() -- get object properties of player
    local ppos = user:get_pos()
    -- 85% of player's eye height
    ppos.y = ppos.y + (init_props.eye_height * 0.85)
    local lookdir = vector.multiply(user:get_look_dir(), 1) -- 100% of a node infront
    local pos = vector.add(ppos, lookdir) -- where we'll spawn
    local ball = core.add_entity(pos, "exile_snow:snowball")
    if not ball then return end -- no entity spawned, return
    -- add an acceleration of Y + 1 and 14 nodes ahead of player
    -- TODO: see about a start of 6 and optional ways to charge it up
    ball:set_acceleration( vector.add(vector.new(0,1,0), vector.multiply(user:get_look_dir(), info.power) ) )
    -- if entity is found, apply thrower value and forcefield check
    local ent = ball:get_luaentity()
    if ent then
        -- we da thrower of this snowball
        ent.thrower = user
        -- forcefield of 1 second after initial throw
        ent.owner_forcefield = core.get_server_uptime() + 1
    end
    sound_play(user, itemstack:get_definition().sounds.throw) -- play throw sound
    -- take 1 from itemstack per snowball if not in creative
    if not player_in_creative(user) then
        itemstack:take_item()
        user:set_wielded_item(itemstack)
    end
end

core.register_craftitem("exile_snow:snowball", {
    description = "Snowball",
    inventory_image = "exile_snow_ball.png",
    stack_max = 48,
    sounds = {
        throw = {name = "exile_snow_throw", gain = 0.5, pitch = {0.9, 1.1}}
    },
    -- itemstack mechanics handled in snowball_charge_up
    on_use = function(itemstack, user, pointed_thing)
        snowball_charge_up(itemstack, user)
    end
})

-- recipes
crafting.register_recipe({
    output = "exile_snow:snowball 12",
    type = {is_v4 and "hand" or "crafting_spot"},
    items = {"nodes_nature:snow"},
    level = 1,
    always_known = true
})
crafting.register_recipe({
    output = "exile_snow:snowball 24",
    type = {is_v4 and "hand" or "crafting_spot"},
    items = {"nodes_nature:snow_block"},
    level = 1,
    always_known = true
})