--[[
(C) TPH/tph9677/TubberPupperHusker/TubberPupper/Damotrix
MIT License
https://opensource.org/license/mit

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

-- for saving info, as all we need is stuff we can get on mod load
local info = {
    OS = "Unknown", -- default in case of error, assume Linux if unknown
    has_backslash = false, -- for OS' like Windows, becareful!
    arch = "Unknown" -- if JIT is enabled, gets arch from there
}

local S = core.get_translator("operating_system_detector")

-- get modpath for detection
local modpath = core.get_modpath("operating_system_detector") -- all we need to do our magic
info.pathstring = modpath

-- split into its constituent parts
local path = modpath:split("/")
-- only windows will have this unique problem (backslashes)
if #path == 1 then
    path = modpath:split("\\")
    -- if it has backslashes, this will be more than 1
    if #path > 1 then
        -- Windows partitions begin with "letter:" e.g. "C:"
        if path[1]:match("%w:") then
            info.OS = "Windows"
        end
        info.has_backslash = true
    -- no path could be grabbed
    else
        path = nil
    end
end
-- table of paths
info.path = path
-- still more searching to be had
if info.OS == "Unknown" then
    -- set up by Android app installation
    if path[1] == "storage" and path[2] == "emulated" and modpath:match("Android") then
        info.OS = "Android"
    -- Linux (?) - BSD has the same thing unfortunately
    elseif path[1] == "home" then
        info.OS = "Linux"
    -- Mac (Apple)
    elseif path[1] == "Users" then
        info.OS = "MacOS"
    -- iOS (Apple)
    elseif path[1] == "private" and path[2] == "mobile" then
        info.OS = "iOS"
    end
end
-- you know what to do, boys!
if modpath:match("multicraft") then
    info.MultiCraft = true
end
-- we can determine if it's iPad from MultiCraft
if info.OS == "Unknown" and info.MultiCraft then
    info.OS = "iPad" -- could be iOS, but we'll go iPad anyways
end

-- we can use the jit.os command if available lol
if jit then
    local OS = jit.os
    -- OSX is macOS, and other is an unsupported detection
    info.OS = OS == "OSX" and "macOS" or OS == "Other" and info.OS or OS
    -- add architecture
    info.arch = jit.arch or "Unknown"
end

-- we're mobile!!!
info.mobile = (info.OS == "iPad" or info.OS == "iOS" or info.OS == "Android")

--core.log(modpath)
--core.log(info.OS)

-- global for use
os_detector = {
    get_OS = function()
        return info.OS
    end,
    -- returns "Unknown" if LuaJIT is not available
    get_architecture = function()
        return info.arch
    end,
    get_path_string = function()
        return info.pathstring
    end,
    -- lowercase permits making a "case insensitive" variant where everything is lowercase
    get_path = function(lowercase)
        if lowercase then
            local pathstring = info.pathstring:lower()
            return info.has_backslash and pathstring:split("\\") or pathstring:split("/")
        end
        return path and table.copy(info.path)
    end,
    -- unique little operating system, aren't ya?
    has_backslash = function()
        return info.has_backslash
    end,
    -- MULTICRAFT! AAAAAAAAAAAAAA
    is_multicraft = function()
        return info.MultiCraft or false
    end,
    is_mobile = function()
        return info.mobile
    end
}
-- backwards compatibility
OS_detector = os_detector

-- chat commands
local chatcommands = {
    os = {
        description = S("Gets detected operating system type from modpath"),
        func = function(player, params)
            return true, S("Current detected operating system is @1",info.OS)
        end,
    },
    is_mobile = {
        description = S("Returns whether or not your detected operating system is a mobile device"),
        func = function(player, params)
              -- is mobile
            return true, info.mobile and S("Current operating system is detected as being a mobile device. @1",
                S("Current detected operating system is @1", info.OS)) or
              -- not mobile
              S("Current operating system is not a mobile device. @1",
                S("Current detected operating system is @1", info.OS))
        end,
    },
    architecture = {
        description = S("Gets JIT detected architecture of operating system"),
        func = function(player, params)
            return true, info.arch == "Unknown" and S("Cannot detect system architecture, LuaJIT is not enabled") or
              S("Current detected system architecture is @1", info.arch)
        end
    },
}
-- won't permit leaking modpath on servers
if core.is_singleplayer() then
    chatcommands.modpath = {
        description = S("Returns modpath string of Operating System Detector mod"),
        func = function(player, params)
            return true, info.pathstring
        end
    }
end


local cmddef = {
    description = S("Handles command structure for Operating System Detector commands"),
    -- handles command functionality
    func = function(name, param)
        if type(param) ~= "string" then return false end
        param = param:lower():split(" ")
        local cmd = param[1]
        if not cmd then
            return false, S("No command issued")
        end

        if chatcommands[cmd] then
            param[1] = nil -- clear from params list
            return chatcommands[cmd].func(name, param)
        else
            return false, S("'@1' is not a valid command!")
        end
    end
}

core.register_chatcommand("osd", cmddef)
-- register variations
for cmd,data in pairs(chatcommands) do
    core.register_chatcommand("osd "..cmd, {description=data.description})
end
-- permit alternative name for `is_mobile`
chatcommands.mobile = chatcommands.is_mobile
-- permit alternative name for `architecture`
chatcommands.arch = chatcommands.architecture
