tph_doors = {}

-- tph_doors "is_protected"
-- returns is_protected boolean, name
-- case insensitive unless modified
function tph_doors.is_protected(pos, player)
  local name = type(player) == "string" and player or type(player) == "table" or type(player) == "userdata"
  and player.get_player_name and player:get_player_name() or ""
  if minetest.check_player_privs(name,"protection_bypass") then return false, name end
  local meta = minetest.get_meta(pos)
  if meta:contains("owner") then
    if string.lower(meta:get_string("owner")) == string.lower(name) then
      return false, name
    end
    return true, name
  end
  return false, name
  --return minetest.is_protected(pos, name), name
end

-- tph_doors "set_protection"
-- modify this function to determine how it should set protection stuff
function tph_doors.set_protection(pos, player)
  local meta = minetest.get_meta(pos)
  local name = type(player) == "string" and player or type(player) == "table" or type(player) == "userdata"
  and player.get_player_name and player:get_player_name() or nil
  if not name or name == "" then return false end -- failure
  meta:set_string("owner",name)
  return true -- success
end

-- what'd happen realistically - you can close a locked door but can't open a locked one
function tph_doors.can_interact_realistic(pos, clicker, itemstack, nodemeta)
  -- if it's not even protected, then our work here is done
  if not tph_doors.is_protected(pos, clicker) then
    return true
  end
  if nodemeta:get_int("state") == 1 then -- opened locked doors can close and lock
    return true
  end
  return false
end

-- absolutely no interaction under any circumstances IF protected
function tph_doors.can_interact_no_interact(pos, clicker, itemstack, nodemeta)
  if not tph_doors.is_protected(pos, clicker) then
    return true
  end
  return false
end

function tph_doors.open_close(pos, node, clicker, itemstack, pointed_thing, force_interact)
  local def = minetest.registered_nodes[node.name]
  -- specify whether or not the door should open or close
  local meta = minetest.get_meta(pos)
  local can_interact = force_interact == true and true or type(def.door_interact) == "function" and def.door_interact(pos, clicker, itemstack, meta) or type(def.door_interact) ~= "function" and true or false
  if not can_interact then
    return
  end
  local door_data = meta:to_table() -- use less metatable function calls by converting to table
  if not door_data then return end -- error, don't continue
  local state = tonumber(door_data.fields.state) or 0 -- 0 for closed, 1 for open
  -- fix conversion issues
  if state > 1 then
    state = state == 2 and 0 or state == 3 and 1 or 0
  end
  local closedp2 = tonumber(door_data.fields.p2) or nil -- store closed door param2
  local sounds = def.sounds
  if state == 1 then -- open to close
    local can_close = force_interact or type(def._can_close) == "function" and def._can_close(pos, clicker) or true
    if not can_close then return end -- can close custom function
    if type(def._door_closed) == "function" then
      def._door_closed(pos,clicker,table.copy(node))
    end
    if sounds.door_close then
      local sound = table.copy(sounds.door_close)
      sound.pos = pos
      minetest.sound_play(sound.name,sound)
    end
    -- closing
    door_data.fields.state = 0
  else -- close to open
    local can_open = force_interact or type(def._can_open) == "function" and def._can_open(pos, clicker) or true
    if not can_open then return end -- can open custom function
    if type(def._door_opened) == "function" then
      def._door_opened(pos,clicker,table.copy(node))
    end
    if sounds.door_open then
      local sound = table.copy(sounds.door_open)
      sound.pos = pos
      minetest.sound_play(sound.name,sound)
    end
    -- opening
    door_data.fields.state = 1
  end
  local door_type = node.name:sub(#node.name-1,#node.name) == "_b" and 2 or 1 -- "_b" is 2, normal is 1
  local p2 = node.param2
  -- closedp2 or if nil; get a closed door's p2, or figure out one from an open door
  closedp2 = closedp2 or (state == 0 and p2) or (door_type == 1 and p2 - 1 or p2 + 1)
  -- fix closedp2 to a proper param2
  closedp2 = (closedp2 < 0 and 3 or closedp2 > 3 and 0) or closedp2
  p2 = state == 1 and closedp2 or (door_type == 1 and p2 + 1 or p2 - 1)
  -- set closed param2 and convert table back to metatable
  door_data.fields.p2 = closedp2
  meta:from_table(door_data)
  -- fix param2 to a proper param2
  node.param2 = (p2 < 0 and 3 or p2 > 3 and 0) or p2
  -- set door node between normal or _b variant
  node.name = door_type == 2 and node.name:sub(1,#node.name-2) or node.name.."_b"
  minetest.swap_node(pos,node)
end

-- gets the true collision height of a node
local function get_node_height(def)
  local height = 0
  local function get_height(box)
    if box and box.fixed then
      -- several boxes
      if type(box.fixed[1]) == "table" then
        local points = {10000000,-10000000} -- lowest, heighest (greater than reasonable minetest bounds lol)
        for _,box_t in pairs(box.fixed) do
          -- if Y of bottom part of box is less than, then set lowest
          if box_t[2] < points[1] then
            points[1] = box_t[2]
          -- if Y of top part of box is lesser than, then set lowest
          elseif box_t[5] < points[1] then
            points[1] = box_t[5]
          end
          -- if Y of bottom part of box is greater than, then set highest
          if box_t[2] > points[2] then
            points[2] = box_t[2]
          -- if Y of top part of box is greater than, then set highest
          elseif box_t[5] > points[2] then
            points[2] = box_t[5]
          end
        end
        -- remove negatives
        height = math.abs(points[1]) + math.abs(points[2])
      else -- singular box
        height = math.abs(box.fixed[2]) + math.abs(box.fixed[5])
      end
    end
  end
  -- check collision + selection boxes, and node_box
  for _,box in pairs({"collision_box","selection_box","node_box"}) do
    if def[box] then
      get_height(def[box])
    end
  end
  return math.max(height,1) -- a node should always be respected at 1
end

local function flip_box(box)
  local function flip_box(b) -- reverse order and flip negative/positive
    local old = table.copy(b)
    b[1] = -old[4]
    b[4] = -old[1]
    return b
  end
  -- can flip
  if type(box) == "table" and type(box.fixed) == "table" then
    if type(box.fixed[1]) == "table" then -- it's a range of boxes
      for bi,b in pairs(box.fixed) do -- LOCAL: box index, box
        box.fixed[bi] = flip_box(b)
      end
    else -- only 1 box
      box.fixed = flip_box(box.fixed)
    end
  end
  return box
end

function tph_doors.register_door(name,def)
  -- errors
  if type(name) ~= "string" then
    error(debug.traceback("tph_doors: invalid name for registration, got '"..type(name).."'",2))
  end
  if type(def) ~= "table" then
    error(debug.traceback("tph_doors: invalid definition for registration, got '"..type(def).."'",2))
  end
  -- main definition registration
  def.groups = def.groups or {}
  def.groups.door = 1
  def.stack_max = def.stack_max or 12
  def.paramtype = def.paramtype or "light"
  def.paramtype2 = def.paramtype2 or "facedir"
  if type(def.sunlight_propagates) ~= "boolean" then def.sunlight_propagates = true end
  if type(def.walkable) ~= "boolean" then def.walkable = true end
  if type(def.is_ground_content) ~= "boolean" then def.is_ground_content = false end
  if type(def.buildable_to) ~= "boolean" then def.buildable_to = false end
  def.sounds = def.sounds or {}
  -- close and open sounds
  def.sounds.door_close = def.sounds.door_close or {
    name = "doors_door_close", gain = 0.2, max_hear_distance = 10
  }
  def.sounds.door_open = def.sounds.door_open or {
    name = "doors_door_open", gain = 0.2, max_hear_distance = 10
  }
  -- conversion setup
  if not def.drawtype then -- assume we're doing conversion (MTG door-wise)
    def.drawtype = "mesh"
    def.mesh = "door_a.b3d"
    def.selection_box = {type = "fixed", fixed = {-1/2,-1/2,-1/2,1/2,3/2,-6/16}}
    def.collision_box = {type = "fixed", fixed = {-1/2,-1/2,-1/2,1/2,3/2,-6/16}}
  end
  -- function setup
  def.on_rightclick = tph_doors.open_close
  -- on_place with an optional 4th parameter - param2 override for doors
  def.on_place = function(itemstack, placer, pointed_thing, p2_override)
    local pos = pointed_thing and (pointed_thing.under or pointed_thing.above)
    if not pos then return itemstack end
    local place_name = name
    -- player functionality
    local dir = 0
    if minetest.is_player(placer) then
      local sneaking = placer:get_player_control().sneak
      -- right click functionality
      local function rmb_func(ndef)
        if ndef and type(ndef.on_rightclick) == "function" then
          return ndef.on_rightclick(pos,minetest.get_node(pos),placer,itemstack,pointed_thing) or itemstack
        end
        return false
      end
      -- check if can replace or place through node
      local function check(ndef)
        if (not ndef) or (ndef and ndef.buildable_to) then--and not minetest.get_item_group(def.name,"door_no_replace") == 1 then
          return true
        end
      end
      -- actual code
      local unoccupied_space = 0
      for i=0,2 do
        local ndef = minetest.registered_nodes[minetest.get_node(pos).name]
        if check(ndef) then
          unoccupied_space = unoccupied_space + 1
        end
        local rmb_success = (sneaking ~= true and rmb_func(ndef)) or nil
        if rmb_success then
          return rmb_success
        elseif minetest.is_protected(pos,placer:get_player_name()) then
          return itemstack
        end
        if unoccupied_space >= 2 then
          pos.y = pos.y - 1 -- one too high, go down
          break
        end
        pos.y = pos.y + get_node_height(ndef)
      end
      if unoccupied_space < 2 then return itemstack end
      -- getting direction
      dir = minetest.dir_to_facedir(placer:get_look_dir())
      -- ability to put door in accordance to other doors
      local ref = {
        {x = -1, z = 0}, -- 1
				{x = 0, z = 1}, -- 2
				{x = 1, z = 0}, -- 3
				{x = 0, z = -1}, -- 4
      }
      local aside = {
        x = pos.x + ref[dir + 1].x,
        y = pos.y,
        z = pos.z + ref[dir + 1].z
      }
      aside = minetest.get_node(aside).name
      if minetest.get_item_group(aside, "door") == 1 then
        if aside:sub(#aside-1,#aside) ~= "_b" then
          place_name = place_name.."_b"
        end
      else -- check other side
        -- added to prevent error lol
        ref[0] = ref[4]
        ref[-1] = ref[3]
        local otherside = {
          x = pos.x + ref[dir - 1].x,
          y = pos.y,
          z = pos.z + ref[dir - 1].z
        }
        otherside = minetest.get_node(otherside).name
        if minetest.get_item_group(otherside, "door") == 1 then
          if otherside:sub(#otherside-1,#otherside) ~= "_b" then
            place_name = place_name.."_b"
          end
        end
      end
    end
    -- doesn't require player
    minetest.set_node(pos,{name=place_name,param2=p2_override or dir})
    if def.after_place_node then
      def.after_place_node(pos, placer, itemstack, pointed_thing)
    end
    return itemstack
  end
  -- only allow rotation if no exclusion
  if not def._no_rotation then
    -- rotation
    def.on_rotate = function(pos, node, user, mode, new_param2)
      new_param2 = tonumber(new_param2) or -1 -- no errors allowed
      -- only rotate if valid param2
      if new_param2 >= 0 and new_param2 <= 3 then
        local meta = minetest.get_meta(pos)
        -- only rotate if closed
        if meta:get_int("state") == 0 then
          meta:set_int("p2",new_param2)
          node.param2 = new_param2
          minetest.swap_node(pos,node)
        end
      end
      return false
    end
  end
  -- server protected
  if def.server_protected then
    def.on_blast = function() end
    def.protected = true
  end
  -- can_dig functionality
  def.can_dig = def.can_dig or function(pos, clicker)
    if not tph_doors.is_protected(pos, clicker) then
      return true
    else
      return false
    end
  end
  -- get or create on_blast
  def.on_blast = def.on_blast or function(pos, intensity)
    minetest.remove_node(pos)
    return def.drops or {def.drop}
  end
  -- redefine on_blast to include can_blast
  local old_blast = def.on_blast
  def.on_blast = function(pos, intensity)
    -- if result is false from can_blast, check if it isn't a function and return true, otherwise return false
    local can_boom = type(def.can_blast) == "function" and def.can_blast(pos) or type(def.can_blast) ~= "function" and true or false
    if can_boom then return old_blast(pos, intensity) end
  end
  -- after_place_node for protected doors
  def.after_place_node = def.after_place_node or def.protected and function(pos, placer, itemstack, pointed_thing)
    local pn = type(placer) == "string" and placer or placer.get_player_name and placer:get_player_name() or ""
    tph_doors.set_protection(pos, placer)
  end
  minetest.register_node(name,def)
  -- register "_b" variant
  def = table.copy(def)
  def.groups.not_in_creative_inventory = 1
  def.mesh = def.mesh and (def._b_mesh or def.mesh:gsub("_a.","_b.")) or nil
  def.node_box = def.node_box and (def._b_node_box or flip_box(def.node_box)) or nil
  if def.node_box then -- refresh collision boxes for nodebox
    -- doesn't properly flip noedboxes if selection box isn't removed
    def.collision_box = nil
    def.selection_box = nil
  end
  def.collision_box = def.collision_box and (def._b_collision_box or flip_box(def.collision_box)) or nil
  def.selection_box = def.selection_box and (def._b_selection_box or flip_box(def.selection_box)) or nil
  def.tiles = def.tiles and (def._b_tiles or def.tiles)
  def.drop = name
  minetest.register_node(name.."_b",def)
end

-- alias conversion - convert all old doors to new
function tph_doors.alias_conversion(basename) -- basename should be the same name you used to define the door
  assert(type(basename) == "string","tph_doors.alias_conversion: improper name given for alias conversion, got "..type(basename))
  minetest.register_alias(basename.."_a",basename)
  -- don't need to change b
  minetest.register_alias(basename.."_c",basename)
  minetest.register_alias(basename.."_d",basename.."_b")
end
-- you will need to create a function (ABM or LBM?) to delete now impervious and transparent "doors:hidden" nodes