--[[
    Input-held Eating Mod for Minetest
    Copyright (C) 2024 TPH/TubberPupperHusker <damotrixrob@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
--]]
local vers = tonumber(minetest.get_version().string) or 0
tph_eating = {
  eating_time = 0.5,
  eating_repeats = 4,
  use_key = "RMB", -- SHOULD BE STRING TO INDEX
  use_function = "on_secondary_use", -- SHOULD BE STRING TO INDEX
  entity_use_function = "on_rightclick", -- SHOULD BE STRING TO INDEX ON ENTITES
  burping = true,
  silly = false,
}
tph_eating.eating_sound = {
  name = "tph_eating_chew",
  gain = 0.5,
  pitch = {1,1.05}
}
tph_eating.finished_sound = {
  name = "tph_eating_burp",
  gain = 0.35,
  pitch = {0.93,1.03}
}
tph_eating.players_eating = {}

tph_eating.creative_mode = minetest.settings:get_bool("creative_mode")
-- custom function for detecting if player is in creative
tph_eating.player_in_creative = function(player)
  -- get the player by name if string
  if (type(player) == "string") then
    player = minetest.get_player_by_name(player)
  end
  -- if player is a player...
  if (minetest.is_player(player)) then
    if (minetest.check_player_privs(player,"creative") or tph_eating.creative_mode == true) then
      return true
    end
  end
  return false
end

-- for getting an edible item's or node's definition (use itemstack:get_definition() instead for after code loads)
local function get_def(item)
  if type(item) == "table" then
    -- ensure getting a proper table lol
    item = item.name 
  end
  if type(item) == "string" then
    -- check through several tables for the item
    return minetest.registered_items[item] or minetest.registered_craftitems[item] or minetest.registered_tools[item] or minetest.registered_nodes[item]
  end
  return item
end

local function play_sound(sound_def,pos)
  if type(sound_def) ~= "table" then
    -- you gave me something that doesn't work
    error(debug.traceback("tph_eating.play_sound: provided sound_def is not a table! Got type: "..type(sound_def),2))
  end
  local sound_name = sound_def.name -- # permits using a table as so: sound_table = {name = "sound", 1 = {stuff}, 2 = {stuff} }
  -- if more than 1 playable sound that is
  if #sound_def > 1 then
    -- more than 1 sound to play
    local playable_sounds = {}
    for _,sound in pairs(sound_def) do
      if type(sound) == "table" and (sound.name or sound_name) then
        playable_sounds[#playable_sounds + 1] = sound
      end
    end
    if #playable_sounds <= 0 then
      -- did not get proper sound defs
      error(debug.traceback("tph_eating.sound_play: more than 1 sound provided in sound_def, however no proper sound defs inside!",2))
    end
    sound_def = playable_sounds[math.random(1,#playable_sounds)]
    sound_name = sound_def.name or sound_name
  end
  if type(sound_name) ~= "string" then
    -- no name
    error(debug.traceback("tph_eating.sound_play: improper name provided for sound!",2))
  end
  sound_name = string.gsub(sound_name," ","") -- remove spaces
  if sound_name == "" then -- you didn't want to play this sound
    return -1
  end
  sound_def = table.copy(sound_def) -- prevent overwriting tables
  sound_def.name = sound_name -- apply modified name rid of any errors
  -- verify or eliminate pos
  if type(pos) == "table" or type(pos) == "userdata" then
    pos = {x = tonumber(pos.x), y = tonumber(pos.y), z = tonumber(pos.z)}
    if not pos.x or not pos.y or not pos.z then
      pos = nil
    end
  else
    pos = nil
  end
  sound_def.pos = pos
  -- utilize singular number or randomized value
  local function get_range(value)
    -- if value is a table and its index 1 and 2 are numbers then return a randomized value between them, otherwise return value
    return type(value) == 'table' and tonumber(value[1]) and tonumber(value[2]) and
    (value[1]+math.random()*(value[2]-value[1]) ) or value
  end
  -- automatically set each number table value (in case of future new values)
  for valuename,value in pairs(sound_def) do
    if type(value) == "table" and tonumber(value[1]) and tonumber(value[2]) then
      sound_def[valuename] = get_range(value)
    end
  end

  return minetest.sound_play(sound_def.name,sound_def)
end
function tph_eating.sound_play(...) -- sound_def, pos
  return play_sound(...)
end

local function get_eating_information(item, noerror)
  if type(item) == "userdata" and item["get_definition"] then
    item = item:get_definition()
  end
  item = type(item) == "table" and item or get_def(item)
  if type(item) ~= "table" and not noerror then
    error("tph_eating: Could not get eating information for "..tostring(item))
  end
  local eating_time = tonumber(item._eating_time) or tph_eating.eating_time
  eating_time = math.max(eating_time,0.001) -- minimum of 1 millisecond
  local eating_repeats = tonumber(item._eating_repeats) or tph_eating.eating_repeats
  eating_repeats = math.max(eating_repeats,2) -- minimum of 2 eating repeats
  local sounds = item.sounds 
  local eating_sound = tph_eating.eating_sound
  local finished_sound = tph_eating.finished_sound
  if sounds then
    eating_sound = (type(sounds.eating_chew) == "function" or type(sounds.eating_chew) == "table") and sounds.eating_chew or
    eating_sound or {}
    finished_sound = (type(sounds.eating_finished) == "function" or type(sounds.eating_finished) == "table") and
    sounds.eating_finished or finished_sound or {}
  end
  -- # accepts getting a sound table from a function or table
  if type(eating_sound) == "function" then
    eating_sound = eating_sound()
    if type(eating_sound) ~= "table" then
      eating_sound = {name = "tph_eating_chew"}
    end
  end
  if type(finished_sound) == "function" then
    finished_sound = finished_sound()
    if type(finished_sound) ~= "table" then
      finished_sound = {name = "tph_eating_burp"}
    end
  end
  -- you don't like my burping lol, let's modify eating_sound for a finished_sound if you didn't specify one
  if finished_sound.name == "tph_eating_burp" and tph_eating.burping ~= true then
    finished_sound = {
      name = eating_sound.name,
      gain = eating_sound.gain,
      -- table, number, or nil - depends on eating_sound
      pitch = type(eating_sound.pitch) == "table" and {eating_sound.pitch[1]*0.78,eating_sound.pitch[1]*0.9} or
      type(eating_sound.pitch) == "number" and eating_sound.pitch * 0.8 or eating_sound.pitch
    }
  end

  return {
    eating_time = eating_time,
    eating_repeats = eating_repeats,
    eating_sound = eating_sound,
    finished_sound = finished_sound,
  }
end
function tph_eating.get_eating_information(...) -- item, noerror (prevents error)
  return get_eating_information(...)
end

-- local to script variable for quick access
local players_eating = tph_eating.players_eating

-- remove player from table global of concurrent players indulging in food
local function clear_eating(player)
  -- get string of player
  player = type(player) == "string" and player or minetest.is_player(player) and player:get_player_name()
  if player then
    players_eating[player] = nil
  end
end
-- allow mods to clear player eating
function tph_eating.clear_eating(...)
  clear_eating(...)
end
-- stop a player's nutritional indulgence
local function cease_eating(player)
  player = type(player) == "string" and player or minetest.is_player(player) and player:get_player_name()
  local data = players_eating[player]
  if data then
    data.force_finish = true
  end
end
-- allow mods to stop a player from eating
function tph_eating.cease_eating(...)
  return cease_eating(...)
end

-- MAIN EATING FUNCTION
local function eating(player, itemstack)
  if type(itemstack) ~= "userdata" or not minetest.is_player(player) then
    -- you're no fun, you gave me a bad itemstack or player
    clear_eating(player)
    return
  end
  local pname = player:get_player_name()
  local data = players_eating[pname] or {}
  data.iteration = tonumber(data.iteration) or 1
  data.tool_def = data.tool_def or itemstack:get_definition()
  if not data.tool_def then
    -- can't be eating with no tooldef
    clear_eating(player)
    return
  end
  -- tool def interactions
  data.img = data.img or data.tool_def._eating_image or data.tool_def.inventory_image
  if data.img == "" and data.tool_def.tiles then
    -- support for getting texture from node if inventory_image doesn't exist
    local tiles = data.tool_def.tiles
    data.img = tiles[math.random(1,#tiles)].name or tiles[math.random(1,#tiles)] -- get any provided texture
  end
  -- eating_info for getting stable _eating values
  data.eating_info = data.eating_info or tph_eating.get_eating_information(data.tool_def)
  local eating_time = data.eating_info.eating_time
  local eating_repeats = data.eating_info.eating_repeats
  -- first time?
  if data.iteration == 1 then
    -- # clear_eating not necessary for returns since player has not yet been added to eating global
    -- custom function for specifying whether or not the eating should commence
    if type(data.tool_def._eating_condition) == "function" and data.tool_def._eating_condition(player, itemstack) ~= true then
      return
    end
    data.index = tonumber(data.index) or player:get_wield_index()
    -- allow for custom "eating_initiated" function to be defined upon eating start
    itemstack = type(data.tool_def._eating_initiated) == "function" and data.tool_def._eating_initiated(player, itemstack, data) or itemstack
    if type(itemstack) ~= "userdata" then
      -- no breaking allowed
      return
    end
  end
  local w_item = player:get_wielded_item()
  local control = player:get_player_control()
  -- ate item fully, no longer eating or selecting food item, or dropped item
  if (itemstack:get_name() == "" or itemstack:is_empty()) or not control[tph_eating.use_key] or
  not (data.index or data.index ~= player:get_wield_index()) or
  itemstack:get_name() ~= w_item:get_name() or data.force_finish then
    -- custom failed eat function
    if type(data.tool_def._eating_failed) == "function" then
      data.tool_def._eating_failed(player, itemstack, data)
    end
    clear_eating(player)
    return
  else
    -- update itemstack to wielded item
    itemstack = w_item
  end
  local itr_pos = player:get_pos() -- interaction pos
  if type(data.height) ~= "number" then
    local collbox = player:get_properties().collisionbox
    data.height = (collbox[2] + collbox[5])*0.67 -- height for where particles should be (67% of total player height)
  end
  -- # height can be set in _eating_ongoing if player's true height changes (such as in a chair or bed)
  if type(data.img) == "string" then
    local prtc_pos = {x=0,y=data.height,z=0.5} -- particle position
    local bounds = {4,4} -- crop image bounds (X,Y), should not be lower than 1
    local prtc_amt = 2
    if (data.iteration + 1) >= eating_repeats then
      -- create more particles upon finishing
      prtc_amt = 4
    end

    local prtc_def = {
      pos = prtc_pos,
      attached = player,
      time = (eating_time), -- last as long as it takes for a player to eat again
      amount = prtc_amt,
      collisiondetection = true,
      collision_removal = true,
      vel = {min={x = -1.5, y = 1, z = -1.5},max={x = 1.5, y = 3.5, z = 1.5}  },
      acc = {x = 0, y = -11, z = 0},
      size = {min=0.8, max=1.5},
      --glow = 5 -- makes particles not so dark at night lol
    }
    if vers < 5.6 then
      -- compatibility for v5.0.1 to before v5.6
      prtc_def.minpos = prtc_pos
      prtc_def.maxpos = prtc_pos
      prtc_def.minvel = prtc_def.vel.min
      prtc_def.maxvel = prtc_def.vel.max
      prtc_def.minacc = prtc_def.acc
      prtc_def.maxacc = prtc_def.acc
      prtc_def.minsize = prtc_def.size.min
      prtc_def.maxsize = prtc_def.size.max
    end

    for i = 1, 6, 1 do
      local img_crop = data.img.."^[sheet:"..tostring(bounds[1]).."x"..tostring(bounds[2])..":"..math.random(bounds[1])..","..math.random(bounds[2])
      prtc_def.texture = img_crop
      minetest.add_particlespawner(prtc_def)
    end
  end
  if data.iteration >= eating_repeats then
    -- custom eating success function, sounds will have to be played in function
    local tempstack = type(data.tool_def._eating_success) == "function" and data.tool_def._eating_success(player, itemstack, data) or nil
    -- continue taking itemstack as normal if no itemstack return or set itemstack to return if provided itemstack
    if type(tempstack) ~= "userdata" then
      if tph_eating.player_in_creative(player) ~= true then
        itemstack:take_item()
      end
    else
      itemstack = tempstack
    end
    play_sound(data.eating_info.finished_sound, itr_pos)
    player:set_wielded_item(itemstack)
    clear_eating(player)
    return
  else
    -- do not run custom function if began eating
    if data.iteraton ~= 1 then
      -- custom function "_eating_ongoing" to run each time it iterates over eating, expects itemstack in return or nil
      itemstack = type(data.tool_def._eating_ongoing) == "function" and data.tool_def._eating_ongoing(player, itemstack, data) or itemstack
      -- # could add a custom eating animation, get time with data.eating_time
    end
    play_sound(data.eating_info.eating_sound, itr_pos)
  end
  -- ending iteration
  data.iteration = data.iteration + 1
  players_eating[pname] = data
  minetest.after(eating_time, eating, player, itemstack)
end
-- permits mods to use the eating function instead of using an eating hook
function tph_eating.eating_func(...)
  return eating(...)
end

function tph_eating.add_eating_hook(item,forcereplace,success_function)
  item = get_def(item)
  -- can't append
  if type(item) ~= "table" then
    return
  end
  -- run potential errors
  if type(tph_eating.use_function) ~= "string" then
    error(debug.traceback("tph_eating: 'use_function' should be string for indexing, got type: "..type(tph_eating.use_function),2))
  end
  if type(tph_eating.use_key) ~= "string" then
    error(debug.traceback("tph_eating: 'use_key' should be string for indexing, got type: "..type(tph_eating.use_key),2))
  end
  -- get old function to run
  local old_func = item[tph_eating.use_function]
  if forcereplace then
    -- or forcibly eradicate the old function
    old_func = nil
  end
  local override = {} -- create override table to properly index with tph_eating.use_function
  override[tph_eating.use_function] = function(itemstack, user, pointed_thing)
    if type(itemstack) ~= "userdata" or type(user) ~= "userdata" then
      return itemstack
    end
    local use_result = type(old_func) == "function" and old_func(itemstack, user, pointed_thing) or nil
    if minetest.is_player(itemstack) then -- itemstack arg is player
      local temp = user
      user = itemstack
      itemstack = temp
    end
    if use_result ~= nil and use_result ~= itemstack then
      return itemstack
    end
    -- pointing at an entity with an on_rightclick function
    if type(pointed_thing) == "table" and pointed_thing.ref then
      local entity = pointed_thing.ref:get_luaentity()
      if entity then
        use_result = type(entity[tph_eating.entity_use_function]) == "function" and entity[tph_eating.entity_use_function](entity, user, itemstack, pointed_thing) or false
        if use_result ~= false or use_result ~= itemstack then
          return itemstack
        end
      end
    end
    if not players_eating[user:get_player_name()] then
      eating(user, itemstack)
    end
    return itemstack
  end
  if type(success_function) == "function" then
    override["_eating_success"] = success_function
  end
  minetest.override_item(item.name,override)
end

-- MOD SUPPORT (place lua files with necessary functions for connection to tph_eating, inside of compatibility folder)
local path = minetest.get_modpath("tph_eating")
local support_list = minetest.get_dir_list(path.."/compatibility")
for _,file in pairs(support_list) do
  -- run any lua file inside of "tph_eating/compatibility"
  if string.match(file,".lua") then
    dofile(path.."/compatibility/"..file)
  end
end