-- no exact point to this really, but it's nice to have a warning before things get ugly
if core.global_exists("typeof") then
    error("typeof: global exists already - causing crash to prevent incompatibility or issues")
end

core.log("info", "typeof: loading getmetatable references")

local modstorage = core.get_mod_storage()

local refs = {
    areastore = getmetatable(AreaStore()),
    settings = getmetatable(core.settings),
    meta = {
        storage = getmetatable(modstorage)
    }
}
local tablerefs = {}

-- split each dividable type into individual functions
-- tables
local function is_table(value)
    -- a vector
    -- voxelarea has the same values as a regular vector, compatibility for <5.5.0
    if vector and vector.check and vector.check(value) then
        return "vector"
    -- voxelarea has the same values as a regular vector, so check for `ystride`
    elseif (not value.ystride) and
      -- not a true vector type, for <5.5.0 or vectors created within a table
      type(value.x) == "number" and type(value.y) == "number" and type(value.z) == "number" then
        return "pseudovector"
    end
    -- special tables with metatables ; voxel area
    local special = getmetatable(value)
    if special == tablerefs.voxelarea then
        return "voxelarea"
    -- has a metatable, but let's do a more sophisticated look into this
    elseif special then
        -- hmm, might be a registered item
        if value.name then
            local reg_item = core.registered_items[value.name]
            -- oh wow, we are!
            -- each item definition has its own unique metatable
            if reg_item and getmetatable(reg_item) == special then
                return "itemdefinition", (value.type or "none")
            end
        end
        -- nothing crazy, just a table with a metatable
        return "table", "metatable"
    end
    -- default
    return "table"
end

-- userdatas
-- players, objects, metarefs
local function is_userdata(value)
    -- player
    if core.is_player(value) then
        return "player"
    -- object
    elseif type(value.get_properties) == "function" and type(value.set_animation) == "function" and
      type(value.set_detach) == "function" then
        return "object"
    end
    -- check metatable references
    local metaref = getmetatable(value)
    if metaref then
        -- check list of userdata metatable references
        for name, refdata in pairs(refs) do
            if refdata == metaref then
                return name
            end
        end
        -- metadataref
        if type(value.contains) == "function" and type(value.to_table) == "function" and
          type(value.set_int) == "function" then
            for name, refdata in pairs(refs.meta) do
                if refdata == metaref then
                    return "meta", name
                end
            end
            -- hmm... couldn't figure out what type of metadataref this was
            return "meta"
        end
    end
    -- default
    return "userdata"
end

-- numbers
-- permit returning whether or not this is a float (decimal points) or integer
local function is_number(value)
    if math.ceil(value) ~= value then
        return "number", "float"
    else
        return "number", "integer"
    end
end

-- returns a more complex assortment of types
-- e.g. vector will be "vector" instead of "table" or player will be "player" instead of "userdata"
-- sorted into: main type, subdivision type (2 return values)
-- type, subdivision = typeof(value)
-- subdivision can be nil
function typeof(value)
    local gottype = type(value) -- so we aren't calling `type()` constantly lol
    if gottype == "table" then return is_table(value) end
    if gottype == "userdata" then return is_userdata(value) end
    if gottype == "number" then return is_number(value) end
    -- return detected type if not a table, userdata, or number
    return gottype
end

-- itemstack
local item = ItemStack(next(core.registered_items).name)
refs.itemstack = getmetatable(item)
refs.meta.itemstack = getmetatable(item:get_meta())
-- get post-load metatable references
core.register_on_mods_loaded(function()
    -- node specific (ran on a delay for world load)
    core.after(0.01, function()
        -- positions
        local positions = {vector.new(0,0,0), vector.new(1,1,1)}
        refs.voxelmanip = getmetatable(VoxelManip(positions[1], positions[2]))
        tablerefs.voxelarea = VoxelArea and getmetatable(
        VoxelArea:new({MinEdge = positions[1], MaxEdge = positions[2]}))
        refs.nodetimer = getmetatable(core.get_node_timer(positions[1]))
        local meta = core.get_meta(positions[1])
        refs.meta.node = getmetatable(meta)
        refs.inventory = getmetatable(meta:get_inventory())
        -- debug message
        core.log("info", "typeof: finished loading metatable references (excluding player meta)")
    end)
end)

core.register_on_joinplayer(function(plr)
    if refs.meta.player then return end
    refs.meta.player = getmetatable(plr:get_meta())
    core.log("info", "typeof: loaded player meta metatable reference")
    -- awful debug to check values
    --[[
    for e,v in pairs(refs) do
        core.log(e.." : "..tostring(v))
        if e == "meta" then
            for e2,v2 in pairs(v) do
                core.log(e.." ;; "..e2.." : "..tostring(v2))
            end
        end
    end
    --]]
    -- check if player meta metatable has been successfully added
    --[[
    core.after(1, function()
        core.log(dump({typeof(plr:get_meta())}))
    end)
    --]]
end)

-- uncomment this to test *almost* each typeof

--[[
core.after(3, function()
    local pos = {vector.new(0,0,0), vector.new(1,1,1)}
    core.log(dump({typeof(1)}))
    core.log(dump({typeof(1.2)}))
    core.log(typeof(AreaStore()))
    core.log(typeof(VoxelArea:new(pos[1], pos[2])))
    core.log(typeof(core.settings))
    core.log(typeof(vector.new(0,0,0)))
    core.log(typeof({x=1,y=1,z=1}))
    local item = ItemStack(next(core.registered_items).name)
    core.log(typeof(item))
    core.log(dump({typeof(item:get_meta())}))
    core.log(dump({typeof(modstorage)}))
    core.log(typeof(VoxelManip(vector.new(0,0,0),vector.new(10,10,10))))
    local meta = core.get_meta(vector.new(0,0,0))
    core.log(dump({typeof(meta)}))
    core.log(typeof(meta:get_inventory()))
    core.log(typeof(core.get_node_timer(vector.new(0,0,0))))
    -- node def, item def, tool def
    -- next provides the index, so we'll have to check the tables for real
    local ndef, idef, tdef = next(core.registered_nodes), next(core.registered_craftitems), next(core.registered_tools)
    ndef, idef, tdef = core.registered_items[ndef], core.registered_items[idef], core.registered_tools[tdef]
    core.log(ndef.name.." : "..dump({typeof(ndef)}))
    core.log(idef.name.." : "..dump({typeof(idef)}))
    core.log(tdef.name.." : "..dump({typeof(tdef)}))
    -- check what a none will provide too
    core.log("nil : "..dump({typeof(item:get_definition())}))
end)
--]]