local S = beds.get_translator
local is_54 = core.has_feature("direct_velocity_on_players")
local is_pova = core.get_modpath("pova")
local pi = math.pi
local is_sp = core.is_singleplayer()
local enable_respawn = core.settings:get_bool("enable_bed_respawn") ~= false
local chat_msg = core.settings:get_bool("bed_show_sleep_msg")

-- Helper functions

local function get_look_yaw(pos)

	local rotation = core.get_node(pos).param2

	if rotation > 3 then
		rotation = rotation % 4 -- Mask colorfacedir values
	end

	if rotation == 1 then
		return pi / 2, rotation
	elseif rotation == 3 then
		return -pi / 2, rotation
	elseif rotation == 0 then
		return pi, rotation
	else
		return 0, rotation
	end
end


local function is_night_skip_enabled()

	local enable_night_skip = core.settings:get_bool("enable_bed_night_skip")

	if enable_night_skip == nil then
		enable_night_skip = true
	end

	return enable_night_skip
end


local function check_in_beds(players)

	local in_bed = beds.player

	if not players then
		players = core.get_connected_players()
	end

	for n, player in pairs(players) do

		local name = player and player:get_player_name()

		if not in_bed[name] then return false end
	end

	return #players > 0
end


local function lay_down(player, pos, bed_pos, state, skip)

	local name = player:get_player_name()
	local hud_flags = player:hud_get_flags()

	if not player or not name then return end

	-- stand up
	if state ~= nil and not state then

		if not beds.player[name] then
			return false -- player not in bed, do nothing
		end

		beds.bed_position[name] = nil

		-- skip here to prevent sending player specific changes (used for leaving players)
		if skip then return end

		player:set_pos(beds.pos[name])

		-- physics, eye_offset, etc
		local physics_override = beds.player[name].physics_override

		beds.player[name] = nil

		if is_pova then
			pova.del_override(name, "force")
			pova.do_override(player)
		else
			player:set_physics_override({
				speed = physics_override.speed,
				jump = physics_override.jump,
				gravity = physics_override.gravity
			})
		end

		player:set_eye_offset({x = 0, y = 0, z = 0}, {x = 0, y = 0, z = 0})
		player:set_look_horizontal(math.random(1, 180) / 100)

		player_api.player_attached[name] = false
		player_api.set_animation(player, "stand" , 30)

		hud_flags.wielditem = true

	else -- lay down

		-- Check if bed is occupied
		for _, other_pos in pairs(beds.bed_position) do

			if vector.distance(bed_pos, other_pos) < 0.1 then

				core.chat_send_player(name, S("This bed is already occupied!"))

				return false
			end
		end

		-- Check if player is moving
		if is_54 and vector.length(player:get_velocity()) > 0.08 then

			core.chat_send_player(name,
					S("You have to stop moving before going to bed!"))

			return false
		end

		-- Check if player is attached to an object
		if player:get_attach() then return false end

		-- player already in bed, do nothing
		if beds.player[name] then return false end

		beds.pos[name] = pos
		beds.bed_position[name] = bed_pos
		beds.player[name] = {physics_override = player:get_physics_override()}

		local yaw, param2 = get_look_yaw(bed_pos)

		player:set_look_horizontal(yaw)

		local dir = core.facedir_to_dir(param2)

		-- p.y is just above the nodebox height of the 'Simple Bed' (the highest bed),
		-- to avoid sinking down through the bed.
		local p = {
			x = bed_pos.x + dir.x / 2,
			y = bed_pos.y + 0.07,
			z = bed_pos.z + dir.z / 2
		}

		if is_pova then
			pova.add_override(name, "force", {speed = 0, jump = 0, gravity = 0})
			pova.do_override(player)
		else
			player:set_physics_override({speed = 0, jump = 0, gravity = 0})
		end

		player:set_pos(p)

		player_api.player_attached[name] = true
		player_api.set_animation(player, "lay" , 0)

		hud_flags.wielditem = false

		if chat_msg then
			core.chat_send_all(S("@1 has gone to bed.", name))
		end
	end

	player:hud_set_flags(hud_flags)
end


local function get_player_in_bed_count()

	local c = 0

	for _, _ in pairs(beds.player) do
		c = c + 1
	end

	return c
end


local div = tonumber(core.settings:get("bed_sleep_divide")) or 2

local function update_formspecs(finished)

	local ges = #core.get_connected_players()
	local player_in_bed = get_player_in_bed_count()
	local is_majority = (ges / div) <= player_in_bed
	local form_n
	local esc = core.formspec_escape

	if finished then
		form_n = beds.formspec .. "label[2.7,9;" .. esc(S("Good morning.")) .. "]"
	else
		form_n = beds.formspec .. "label[2.2,9;" ..
			esc(S("@1 of @2 players are in bed", player_in_bed, ges)) .. "]"

		if is_majority and is_night_skip_enabled() then
			form_n = form_n .. "button_exit[2,6;4,0.75;force;" ..
				esc(S("Force night skip")) .. "]"
		end
	end

	for name,_ in pairs(beds.player) do
		core.show_formspec(name, "beds_form", form_n)
	end
end

-- Public functions

function beds.kick_players()

	for name, _ in pairs(beds.player) do

		local player = core.get_player_by_name(name)

		lay_down(player, nil, nil, false)
	end
end


function beds.skip_night()
	core.set_timeofday(0.23)
end


local update_scheduled = false

local function schedule_update()

	if update_scheduled then
		-- there already is an update scheduled; don't schedule more to prevent races
		return
	end

	update_scheduled = true

	core.after(2, function()

		update_scheduled = false

		if not is_sp then
			update_formspecs(is_night_skip_enabled())
		end

		if is_night_skip_enabled() then

			-- skip the night and let all players stand up
			beds.skip_night()
			beds.kick_players()
		end
	end)
end


function beds.kick_player_at(kick_pos)

	for name, bed_pos in pairs(beds.bed_position) do

		if vector.equals(bed_pos, kick_pos) then

			if beds.player[name] then

				local player = core.get_player_by_name(name)

				if not player then
					return false
				end

				lay_down(player, nil, nil, false)

				core.close_formspec(name, "beds_form")

				core.log("info", "[beds] Kicked " .. name
					.. " out of bed at " .. core.pos_to_string(kick_pos))

				return true
			end
		end
	end

	return false
end


function beds.on_rightclick(pos, player)

	local name = player:get_player_name()
	local ppos = player:get_pos()
	local tod = core.get_timeofday()

	if tod > beds.day_interval.start and tod < beds.day_interval.finish then

		if beds.player[name] then
			lay_down(player, nil, nil, false)
		end

		core.chat_send_player(name, S("You can only sleep at night."))

		return
	end

	-- move to bed
	if not beds.player[name] then

		lay_down(player, ppos, pos)

		beds.set_spawns() -- save respawn positions when entering bed
	else
		lay_down(player, nil, nil, false)
	end

	-- check for custom on_sleep function
	local nod = core.get_node(pos).name
	local def = core.registered_nodes[nod]
	local on_sleep = def and def.on_sleep

	-- if on_sleep returns True then skip sleeping
	if on_sleep and on_sleep(pos, player) then

		lay_down(player, nil, nil, false)

		return
	end

	if not is_sp then
		update_formspecs(false)
	end

	if check_in_beds() then
		schedule_update()
	end
end


function beds.can_dig(bed_pos)

	-- Check all players in bed which one is at the expected position
	for _, player_bed_pos in pairs(beds.bed_position) do

		if vector.equals(bed_pos, player_bed_pos) then
			return false
		end
	end

	return true
end

-- Callbacks

-- Only register respawn callback if respawn enabled
if enable_respawn then

	-- set respawn flag to true by default
	core.register_on_joinplayer(function(player)

		if not player then return end

		local name = player:get_player_name()
		beds.respawn[name] = true
	end)

	-- respawn player at bed if enabled and valid position is found
	core.register_on_respawnplayer(function(player)

		if not player then return end

		local name = player:get_player_name()
		local pos = beds.spawn[name]

		-- check if respawn flag is true (for mini-game support, can be set to false)
		if pos and beds.respawn[name] then
			player:set_pos(pos)
			return true
		end
	end)
end


core.register_on_leaveplayer(function(player)

	if not player then return end

	local name = player:get_player_name()

	lay_down(player, nil, nil, false, true)

	beds.player[name] = nil

	if check_in_beds() then
		schedule_update()
	end
end)


core.register_on_dieplayer(function(player)

	local name = player:get_player_name()
	local in_bed = beds.player
	local pos = player:get_pos()
	local yaw = get_look_yaw(pos)

	if in_bed[name] then
		lay_down(player, nil, pos, false)
		player:set_look_horizontal(yaw)
		player:set_pos(pos)
	end
end)


core.register_on_player_receive_fields(function(player, formname, fields)

	if formname ~= "beds_form" then
		return
	end

	-- Because "Force night skip" button is a button_exit, it will set fields.quit
	-- and lay_down call will change value of player_in_bed, so it must be taken
	-- earlier.
	local last_player_in_bed = get_player_in_bed_count()

	if fields.quit or fields.leave then
		lay_down(player, nil, nil, false)
		update_formspecs(false)
	end

	if fields.force then

		-- check if enough players are sleeping to skip night (was half)
		local is_majority = (
				#core.get_connected_players() / div) <= last_player_in_bed

		if is_majority and is_night_skip_enabled() then
			update_formspecs(true)
			beds.skip_night()
			beds.kick_players()
		else
			update_formspecs(false)
		end
	end
end)
