
-- global

invisibility = {
	effect_time = core.settings:get("invisibility.effect_time") or 180 -- 3 mins
}

-- translation and player table

local S = core.get_translator("invisibility")
local players = {}

-- reset player invisibility if they go offline or die

core.register_on_leaveplayer(function(player)

	local name = player:get_player_name()

	if players[name] then players[name] = nil end
end)

core.register_on_dieplayer(function(player)
	invisibility.invisible(player, nil)
end)

-- creative check

local creative_mode_cache = core.settings:get_bool("creative_mode")
local function is_creative(name)
	return creative_mode_cache or core.check_player_privs(name, {creative = true})
end

-- invisibility functions

function invisibility.is_visible(player_name)

	if players[player_name] then return false end

	return true
end

function invisibility.invisible(player, toggle)

	if not player then return false end

	local name = player:get_player_name()

	players[name] = toggle

	local prop

	if toggle == true then -- hide player and name tag

		prop = {visual_size = {x = 0, y = 0}}

		player:set_nametag_attributes({
			color = {a = 0, r = 255, g = 255, b = 255}
		})
	else -- show player and tag
		prop = {visual_size = {x = 1, y = 1}}

		player:set_nametag_attributes({
			color = {a = 255, r = 255, g = 255, b = 255}
		})
	end

	player:set_properties(prop)
end

-- invisibility potion

core.register_node("invisibility:potion", {
	description = S("Invisibility Potion"),
	drawtype = "plantlike",
	tiles = {"invisibility_potion.png"},
	inventory_image = "invisibility_potion.png",
	wield_image = "invisibility_potion.png",
	paramtype = "light",
	is_ground_content = false,
	walkable = false,
	selection_box = {type = "fixed", fixed = {-0.25, -0.5, -0.25, 0.25, 0.3, 0.25}},
	groups = {vessel = 1, dig_immediate = 3, attached_node = 1},
	sounds = default.node_sound_glass_defaults(),

	on_use = function(itemstack, user)

		local pos = user:get_pos()
		local name = user:get_player_name()

		-- are we already invisible?
		if players[name] then

			core.chat_send_player(name, S(">>> You are already invisible!"))

			return itemstack
		end

		-- make player invisible
		invisibility.invisible(user, true)

		-- play sound
		core.sound_play("pop", {
				pos = pos, gain = 1.0, max_hear_distance = 5}, true)

		-- display 10 second warning
		core.after(invisibility.effect_time - 10, function()

			if players[name] and user:get_pos() then

				core.chat_send_player(name,
						S(">>> You have 10 seconds before invisibility wears off!"))
			end
		end)

		-- make player visible 5 minutes later
		core.after(invisibility.effect_time, function()

			if players[name] and user:get_pos() then

				-- show hidden player
				invisibility.invisible(user, nil)

				-- play sound
				core.sound_play("pop", {
						pos = user:get_pos(), gain = 1.0, max_hear_distance = 5}, true)
			end
		end)

		-- take potion, return empty bottle (and rest of potion stack)
		if not is_creative(name) then

			local item_count = user:get_wielded_item():get_count()
			local inv = user:get_inventory()
			local giving_back = "vessels:glass_bottle"

			if inv and item_count > 1 then

				if inv:room_for_item("main", {name = "vessels:glass_bottle"}) then

					inv:add_item("main", {name = "vessels:glass_bottle"})
				else
					pos.y = pos.y + 1

					core.add_item(pos, {name = "vessels:glass_bottle"})
				end

				giving_back = "invisibility:potion " .. tostring(item_count - 1)

			end

			return ItemStack(giving_back)
		end

	end
})

-- craft recipe

core.register_craft( {
	output = "invisibility:potion",
	type = "shapeless",
	recipe = {
		"default:sapling", "default:junglesapling", "default:pine_sapling",
		"default:acacia_sapling", "default:aspen_sapling", "default:bush_sapling",
		"default:acacia_bush_sapling", "vessels:glass_bottle", "flowers:mushroom_red"
	}
})

-- vanish command (admin only)

core.register_chatcommand("vanish", {
	params = "<name>",
	description = S("Make player invisible"),
	privs = {server = true},

	func = function(name, param)

		-- player online
		if param ~= "" and core.get_player_by_name(param) then

			name = param

		-- player not online
		elseif param ~= "" then

			return false, S("Player @1 is offline!", param)
		end

		local player = core.get_player_by_name(name)
		local msg

		-- hide / show player
		if players[name] then

			invisibility.invisible(player, nil) ; msg = "visible"
		else
			invisibility.invisible(player, true) ; msg = "invisible"
		end

		return true, S("Player @1 is now @2!", name, S(msg))
	end
})
