# Charged Models for Luanti

A powerful Luanti mod that allows you to attach animated 3D models to held items with a sophisticated keyframe animation system. Create dynamic, interactive items with custom animations triggered by player input.

## Features

- **Attach 3D models to any held item** - Replace default 2D inventory sprites with full 3D models
- **Sequential keyframe animation system** - Define smooth animations with position, rotation, and easing
- **Input-driven animations** - Trigger animations based on player keybinds (dig, place, jump, sneak, etc.)
- **Hold mechanics** - Support for charge-up animations that hold at the final frame or loop
- **Multi-model support** - Attach multiple models to a single item (dual wielding, complex items)
- **Look adjustment** - Models automatically adjust based on where the player is looking
- **Callback system** - Hook into animation events (on_start, on_complete, on_hold, on_release)
- **Automatic item display** - Built-in system to display held items/nodes as cubes with proper textures

## Installation

1. Download or clone this repository into your Luanti mods folder
2. Rename the folder to `charged_models`
3. Enable the mod in your world settings

## Quick Start

### Basic Model Registration

```lua
minetest.register_on_mods_loaded(function()
    -- Create a simple idle animation
    charged_models.create_animation("sword_idle", {
        trigger = "none",
        loop = true,
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 1.5, position = {x = 3, y = 13.2, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 3.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}}
        }
    })
    
    -- Create a model configuration
    charged_models.create_model_config("basic_sword", {
        mesh = "sword.gltf",
        textures = {"sword_texture.png"},
        scale = {x = 10, y = 10, z = 10},
        animations = {
            idle = "sword_idle"
        }
    })
    
    -- Register to an item
    charged_models.register_item_model("default:sword_steel", "basic_sword")
end)
```

## API Reference

### Animation System

#### `charged_models.create_animation(name, config)`

Creates a named animation that can be referenced by models.

**Parameters:**
- `name` (string) - Unique identifier for the animation
- `config` (table) - Animation configuration

**Config Options:**
- `trigger` (string) - When to play: `"none"`, `"press_dig"`, `"hold_dig"`, `"release_dig"`, `"press_place"`, `"hold_place"`, `"release_place"`, `"press_jump"`, `"hold_sneak"`, etc.
- `transform_keyframes` (table) - Array of keyframe definitions (see below)
- `loop` (boolean) - Whether animation loops (default: false)
- `loop_speed` (number) - Speed multiplier for loops (default: 1.0)
- `hold_behavior` (string) - `"none"`, `"hold_last_frame"`, or `"loop_until_release"`
- `speed` (number) - Animation playback speed (default: 30)
- `blend` (number) - Blending factor between frames (default: 0)
- `keyframes` (table) - Model animation keyframes `{start = 0, stop = 30}`
- Callbacks: `on_start`, `on_complete`, `on_hold`, `on_release` (functions)

**Transform Keyframe Structure:**
```lua
{
    time = 0.5,  -- Time in seconds from animation start
    position = {x = 3, y = 13, z = 3},  -- Optional position
    rotation = {x = 0, y = 45, z = 0},  -- Optional rotation (degrees)
    easing = "ease_out"  -- "linear", "ease_in", "ease_out", "ease_in_out"
}
```

### Model Configuration

#### `charged_models.create_model_config(name, config)`

Creates a named model configuration template.

**Single Model Config:**
```lua
charged_models.create_model_config("my_tool", {
    mesh = "tool.gltf",
    textures = {"tool_texture.png"},
    position = {x = 3, y = 13, z = 3},
    rotation = {x = 0, y = 0, z = 0},
    scale = {x = 10, y = 10, z = 10},
    look_adjustment = {
        enabled = true,
        factor = 5.0,
        max_vertical_offset = 5.0,
        min_vertical_offset = -2.0,
        rotation_factor = 20.0
    },
    animations = {
        idle = "my_idle_animation",
        attack = "my_attack_animation"
    }
})
```

**Multi-Model Config (Dual Wielding):**
```lua
charged_models.create_model_config("dual_swords", {
    -- Left sword
    {
        mesh = "sword.gltf",
        textures = {"cold.png"},
        position = {x = -2, y = 13, z = 3},
        scale = {x = 8, y = 8, z = 8},
        animations = {
            slash = {  -- Inline animation
                trigger = "press_place",
                transform_keyframes = { ... }
            }
        }
    },
    -- Right sword
    {
        mesh = "sword.gltf",
        textures = {"cold.png"},
        position = {x = 2, y = 13, z = 3},
        scale = {x = 8, y = 8, z = 8},
        animations = { ... }
    }
})
```

### Item Registration

#### `charged_models.register_item_model(item_name, config_name)`

Associates an item with a model configuration.

```lua
charged_models.register_item_model("default:sword_steel", "basic_sword")
```

### Utility Functions

- `charged_models.get_animation(name)` - Retrieve animation config
- `charged_models.get_model_config(name)` - Retrieve model config
- `charged_models.list_model_configs()` - List all registered configs
- `charged_models.has_item_model(item_name)` - Check if item has model
- `charged_models.get_item_model_config(item_name)` - Get item's model config
- `charged_models.unregister_item_model(item_name)` - Remove model from item

## Included Examples

The mod comes with several demonstration examples:

### 1. **Wooden Pickaxe** (`mtg_wooden_pickaxe_example.lua`)
Replaces the default wooden pickaxe with a 3D model featuring:
- Realistic mining animation that loops while holding left-click
- Thrust-forward motion with natural pullback
- Sound effects synced to animation timing
- Demonstrates `hold_dig` trigger and `loop_until_release` behavior

### 2. **Dual Swords** (`dual_swords_example.lua`)
Shows multi-model capabilities:
- Left sword and right sword models
- Independent animations per model
- Left-click slashes right sword, right-click slashes left sword
- Idle breathing animation

### 3. **Charged Blade** (`charged_greatsword_overhead_particles_example.lua`)
Advanced charge-up mechanic with particle effects:
- Hold left-click to charge (sword raises overhead)
- Release to unleash devastating attack with particle explosion
- Charging particles during hold phase
- Shockwave and impact effects on release
- Sound effects and callbacks

### 4. **Bare Fists** (`two_handed_fist_example.lua`)
Two-handed combat system:
- Left and right hand models
- Alternating punch animations
- Synchronized idle breathing

### 5. **Charged Hand** (`charged_hand.lua`)
Complete charge-punch implementation:
- Charge meter HUD display
- Hold to charge, release to punch
- Damage scaling based on charge amount
- Only breaks blocks when fully charged
- Raycast-based targeting

### 6. **Torch with Fire** (`torch_example.lua`)
Dynamic torch with particle effects:
- Flickering idle animation
- Real-time fire and smoke particles
- Ember sparks that obey physics
- Wave and raise animations

### 7. **Automatic Item Display** (`cube-entities.lua`)
Displays any held item as a 3D cube:
- Automatic texture mapping from inventory images
- Node vs item detection
- Configurable positioning per item
- Only shows for items without custom models

## Animation Triggers

Available trigger strings for the `trigger` parameter:

- **None**: `"none"` - Auto-play (for idle animations)
- **Dig (Left Click)**: `"press_dig"`, `"hold_dig"`, `"release_dig"`
- **Place (Right Click)**: `"press_place"`, `"hold_place"`, `"release_place"`
- **Movement**: `"press_jump"`, `"hold_jump"`, `"press_sneak"`, `"hold_sneak"`
- **Directional**: `"press_up"`, `"press_down"`, `"press_left"`, `"press_right"`
- **Other**: `"press_aux1"`, `"hold_aux1"`, `"press_zoom"`

## Easing Functions

Control animation smoothness with easing:

- `"linear"` - Constant speed
- `"ease_in"` - Slow start, fast end
- `"ease_out"` - Fast start, slow end
- `"ease_in_out"` - Slow start and end, fast middle

## Model Format

Models should preferably be in **glTF 2.0** format (`.gltf`) otherwise **Wavefront** format (`.obj`) also works. Place model files in your mod's `models/` directory and textures in `textures/`.

## Dependencies
None

## License

Source code: MIT
Media assets (models, sounds, textures): CC0

## Contributing

Contributions, bug reports, and feature requests are welcome! Share your creative uses of the animation system.

## Credits
Created by TomCon