

-- Example: Animation Completion Detection with Key State Checking
minetest.register_on_mods_loaded(function()
    
    -- Animation that detects completion and checks for key combinations
    charged_models.create_animation("combo_attack", {
        trigger = "press_dig",
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 0.3, position = {x = 4, y = 14, z = 2}, rotation = {x = -45, y = 30, z = 0}, easing = "ease_out"},
            {time = 0.6, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in"}
        },
        on_complete = function(self)
            -- Animation completed - now check if specific keys are held
            local parent = self.object:get_attach()
            if not parent or not parent:is_player() then return end
            
            local player_name = parent:get_player_name()
            local controls = parent:get_player_control()
            
            -- Check if jump key is held when animation completes
            if controls.jump then
                minetest.chat_send_player(player_name, "Combo detected! Jump was held during attack completion!")
                
                -- Trigger a special combo animation
                self:play_animation("combo_finisher")
                
                -- Spawn special particles
                local pos = parent:get_pos()
                pos.y = pos.y + 2
                
                minetest.add_particlespawner({
                    amount = 20,
                    time = 0.1,
                    minpos = {x = pos.x - 1, y = pos.y - 1, z = pos.z - 1},
                    maxpos = {x = pos.x + 1, y = pos.y + 1, z = pos.z + 1},
                    minvel = {x = -2, y = 1, z = -2},
                    maxvel = {x = 2, y = 3, z = 2},
                    minacc = {x = 0, y = -2, z = 0},
                    maxacc = {x = 0, y = -2, z = 0},
                    minexptime = 0.5,
                    maxexptime = 1.0,
                    minsize = 1,
                    maxsize = 3,
                    collisiondetection = false,
                    glow = 14,
                    texture = "default_cloud.png^[colorize:#gold:200"
                })
                
            elseif controls.sneak then
                minetest.chat_send_player(player_name, "Stealth combo! Sneak was held during attack!")
                
                -- Different effect for sneak combo
                minetest.sound_play("default_dig_dig_immediate", {
                    to_player = player_name,
                    gain = 0.5,
                    pitch = 1.5
                })
                
            else
                minetest.chat_send_player(player_name, "Basic attack completed. Try holding jump or sneak for combos!")
            end
        end
    })
    
    -- Combo finisher animation (triggered when jump is held)
    charged_models.create_animation("combo_finisher", {
        trigger = "none",  -- Manually triggered
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 0.2, position = {x = 3, y = 16, z = 3}, rotation = {x = 0, y = 180, z = 0}, easing = "ease_out"},
            {time = 0.4, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 360, z = 0}, easing = "ease_in"}
        },
        on_complete = function(self)
            local parent = self.object:get_attach()
            if parent and parent:is_player() then
                minetest.chat_send_player(parent:get_player_name(), "Combo finisher completed!")
            end
        end
    })
    
    -- Multi-stage animation that checks keys at each stage
    charged_models.create_animation("multi_stage_attack", {
        trigger = "press_place",
        transform_keyframes = {
            -- Stage 1: Wind up
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 0.5, position = {x = 2, y = 14, z = 4}, rotation = {x = -30, y = -45, z = 0}, easing = "ease_in"},
            -- Stage 2: Strike
            {time = 0.8, position = {x = 5, y = 12, z = 1}, rotation = {x = 45, y = 45, z = 0}, easing = "ease_out"},
            -- Stage 3: Recovery
            {time = 1.2, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in_out"}
        },
        on_start = function(self)
            local parent = self.object:get_attach()
            if parent and parent:is_player() then
                minetest.chat_send_player(parent:get_player_name(), "Multi-stage attack started! Hold different keys for effects!")
            end
        end,
        on_complete = function(self)
            local parent = self.object:get_attach()
            if not parent or not parent:is_player() then return end
            
            local player_name = parent:get_player_name()
            local controls = parent:get_player_control()
            
            -- Check multiple key combinations
            local held_keys = {}
            if controls.jump then table.insert(held_keys, "jump") end
            if controls.sneak then table.insert(held_keys, "sneak") end
            if controls.aux1 then table.insert(held_keys, "aux1") end
            if controls.zoom then table.insert(held_keys, "zoom") end
            
            if #held_keys > 0 then
                local key_list = table.concat(held_keys, ", ")
                minetest.chat_send_player(player_name, "Multi-stage attack completed with keys held: " .. key_list)
                
                -- Different effects based on key combinations
                if controls.jump and controls.sneak then
                    minetest.chat_send_player(player_name, "Ultimate combo: Jump + Sneak!")
                    -- Trigger ultimate effect
                elseif controls.aux1 and controls.zoom then
                    minetest.chat_send_player(player_name, "Precision combo: Aux1 + Zoom!")
                    -- Trigger precision effect
                end
            else
                minetest.chat_send_player(player_name, "Multi-stage attack completed normally.")
            end
        end
    })
    
    -- Animation with hold detection and key state monitoring
    charged_models.create_animation("charged_spin", {
        trigger = "hold_aux1",
        hold_behavior = "loop_until_release",
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 0.5, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 90, z = 0}, easing = "linear"},
            {time = 1.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 180, z = 0}, easing = "linear"},
            {time = 1.5, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 270, z = 0}, easing = "linear"},
            {time = 2.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 360, z = 0}, easing = "linear"}
        },
        on_start = function(self)
            local parent = self.object:get_attach()
            if parent and parent:is_player() then
                minetest.chat_send_player(parent:get_player_name(), "Charging spin attack! Hold aux1 to continue spinning!")
            end
        end,
        on_hold = function(self)
            -- Called every loop while key is held
            local parent = self.object:get_attach()
            if not parent or not parent:is_player() then return end
            
            local controls = parent:get_player_control()
            local player_name = parent:get_player_name()
            
            -- Check for additional keys while spinning
            if controls.jump then
                -- Add upward movement while spinning and jumping
                local current_pos = self.object:get_pos()
                if current_pos then
                    current_pos.y = current_pos.y + 0.1
                    self.object:set_pos(current_pos)
                end
            end
            
            -- Spawn particles while spinning
            local pos = parent:get_pos()
            pos.y = pos.y + 1.5
            
            minetest.add_particlespawner({
                amount = 2,
                time = 0.1,
                minpos = {x = pos.x - 0.5, y = pos.y, z = pos.z - 0.5},
                maxpos = {x = pos.x + 0.5, y = pos.y, z = pos.z + 0.5},
                minvel = {x = -1, y = 0, z = -1},
                maxvel = {x = 1, y = 0, z = 1},
                minacc = {x = 0, y = 0, z = 0},
                maxacc = {x = 0, y = 0, z = 0},
                minexptime = 0.3,
                maxexptime = 0.6,
                minsize = 0.5,
                maxsize = 1.5,
                collisiondetection = false,
                glow = 8,
                texture = "default_cloud.png^[colorize:#cyan:150"
            })
        end,
        on_release = function(self)
            local parent = self.object:get_attach()
            if parent and parent:is_player() then
                minetest.chat_send_player(parent:get_player_name(), "Spin attack released!")
            end
        end
    })
    
    -- Create test weapon with all the new animations
    charged_models.create_model_config("test_combo_weapon", {
        mesh = "sword.gltf",
        textures = {"cold.png"},
        position = {x = 3, y = 13, z = 3},
        scale = {x = 10, y = 10, z = 10},
        animations = {
            idle = "idle_breath",  -- Reuse existing idle animation
            combo_attack = "combo_attack",
            combo_finisher = "combo_finisher",
            multi_stage = "multi_stage_attack",
            spin = "charged_spin"
        }
    })
    
    -- Register to a test item
    charged_models.register_item_model("charged_models:test_combo_weapon", "test_combo_weapon")
    
end)

-- Register the test weapon
minetest.register_tool("charged_models:test_combo_weapon", {
    description = "Test Combo Weapon\n" ..
                 "Left click: Combo attack (hold jump/sneak for combos)\n" ..
                 "Right click: Multi-stage attack (hold various keys)\n" ..
                 "Hold Aux1: Spinning attack (hold jump to rise)",
    inventory_image = "cold.png",
    tool_capabilities = {
        full_punch_interval = 1.0,
        max_drop_level = 1,
        groupcaps = {
            cracky = {times = {[1] = 2.0, [2] = 1.0, [3] = 0.5}, uses = 20, maxlevel = 2},
        },
        damage_groups = {fleshy = 4},
    },
})
-- Simple Charge Animation Example
-- Demonstrates hold-to-complete mechanic with proper state management

-- Create the charging animation
charged_models.create_animation("simple_charge", {
    trigger = "press_dig",  -- Start on dig press
    keyframes = {
        start = 0,
        stop = 20  -- 20 frame animation
    },
    speed = 10,  -- Slow animation (2 seconds total)
    blend = 0.1,
    loop = false,
    hold_behavior = "hold_last_frame",  -- Hold at end
    
    -- Simple transform keyframes
    transform_keyframes = {
        -- Start position
        {
            time = 0,
            position = {x = 3, y = 13, z = 3},
            rotation = {x = 0, y = 0, z = 0},
            easing = "linear"
        },
        -- End position (charged)
        {
            time = 2.0,  -- 2 seconds to complete
            position = {x = 2, y = 12, z = 2},
            rotation = {x = -15, y = 5, z = 0},
            easing = "ease_out"
        }
    },
    
    on_start = function(self)
        local player = self.object:get_attach()
        if player then
            minetest.chat_send_player(
                player:get_player_name(),
                "⚡ Charging... hold dig key!"
            )
        end
        self.charge_complete = false
        self.notified_complete = false
    end,
    
    on_hold = function(self)
        -- This is called when animation reaches end and is holding
        if not self.charge_complete then
            self.charge_complete = true
            local player = self.object:get_attach()
            if player and not self.notified_complete then
                minetest.chat_send_player(
                    player:get_player_name(),
                    "✅ CHARGED! Release dig to activate!"
                )
                self.notified_complete = true
            end
        end
    end,
    
    on_release = function(self)
        local player = self.object:get_attach()
        if not player then return end
        
        local player_name = player:get_player_name()
        
        if self.charge_complete then
            -- Success! Animation completed before release
            minetest.chat_send_player(player_name, "🎯 SUCCESS! Full power activated!")
            
            -- Add particle effect
            local pos = player:get_pos()
            pos.y = pos.y + 1.5
            minetest.add_particlespawner({
                amount = 15,
                time = 0.3,
                minpos = {x = pos.x - 0.3, y = pos.y, z = pos.z - 0.3},
                maxpos = {x = pos.x + 0.3, y = pos.y, z = pos.z + 0.3},
                minvel = {x = -2, y = 1, z = -2},
                maxvel = {x = 2, y = 3, z = 2},
                minacc = {x = 0, y = -5, z = 0},
                maxacc = {x = 0, y = -5, z = 0},
                minexptime = 0.5,
                maxexptime = 1.5,
                minsize = 0.5,
                maxsize = 2,
                texture = "default_item.png^[colorize:yellow:100"
            })
        else
            -- Failed! Released too early
            minetest.chat_send_player(player_name, "❌ FAILED! Released too early!")
        end
        
        -- Reset state
        self.charge_complete = false
        self.notified_complete = false
    end
})

-- Simple idle animation (non-looping)
charged_models.create_animation("simple_idle", {
    trigger = "none",
    keyframes = {
        start = 0,
        stop = 0
    },
    speed = 30,
    blend = 0.1,
    loop = false,
    hold_behavior = "none",
    
    transform_keyframes = {
        {
            time = 0,
            position = {x = 3, y = 13, z = 3},
            rotation = {x = 0, y = 0, z = 0},
            easing = "linear"
        }
    }
})

-- Create model configuration
charged_models.create_model_config("simple_charge_model", {
    position = {x = 3, y = 13, z = 3},
    rotation = {x = 0, y = 0, z = 0},
    scale = {x = 8, y = 8, z = 8},
    mesh = "hand.gltf",
    textures = {"default_wood.png"},
    look_adjustment = {
        enabled = true,
        factor = 2.0,
        max_vertical_offset = 2.0,
        min_vertical_offset = -1.0,
        rotation_factor = 10.0
    },
    physical = false,
    collide_with_objects = false,
    pointable = false,
    static_save = false,
    animations = {
        idle = "simple_idle",
        simple_charge = "simple_charge"
    }
})

-- Register test item
minetest.register_craftitem("charged_models:charge_test", {
    description = "Charge Test Item\nPress and hold dig key to charge!\nRelease when fully charged for success.",
    inventory_image = "default_stick.png^[colorize:blue:50",
    stack_max = 1,
    
    on_use = function(itemstack, user, pointed_thing)
        if user and user:is_player() then
            minetest.chat_send_player(
                user:get_player_name(),
                "Press and HOLD dig key to charge!"
            )
        end
        return itemstack
    end
})

-- Register the item with the model
charged_models.register_item_model("charged_models:charge_test", "simple_charge_model")

-- Simple craft recipe
minetest.register_craft({
    output = "charged_models:charge_test",
    recipe = {
        {"default:stick"},
        {"default:stick"},
        {"default:stick"}
    }
})

minetest.log("action", "[Charged Models] Simple charge example loaded!")
-- Example implementations for the Charged Models API
-- This file demonstrates various animation patterns and use cases

-- Example 1: Charge-up Staff - Animation that charges up and holds on last frame until release
-- This creates a magical staff that glows brighter as you hold the dig key

-- First, create the charge-up animation configuration
charged_models.create_animation("charge_up", {
    trigger = "hold_dig",  -- Triggered when holding the dig key
    transform_keyframes = {
        -- Start position (normal staff position)
        {
            time = 0,
            position = {x = 3, y = 13, z = 3},
            rotation = {x = 0, y = 0, z = 0},
            easing = "linear"
        },
        -- Slight movement and rotation as charging begins
        {
            time = 0.5,
            position = {x = 2, y = 14, z = 2},
            rotation = {x = 10, y = 5, z = 0},
            easing = "ease_out"
        },
        -- More dramatic movement as charge builds
        {
            time = 1.0,
            position = {x = 1, y = 15, z = 1},
            rotation = {x = 20, y = 10, z = 5},
            easing = "ease_in_out"
        },
        -- Final charged position - this frame will be held until release
        {
            time = 1.5,
            position = {x = 0, y = 16, z = 0},
            rotation = {x = 30, y = 15, z = 10},
            easing = "ease_in"
        }
    },
    speed = 30,
    blend = 0.2,
    loop = false,
    hold_behavior = "hold_last_frame",  -- This makes it hold on the last frame
    
    -- Callback functions for different events
    on_start = function(entity)
        minetest.log("action", "Staff charging started!")
        -- You could add particle effects here
    end,
    
    on_hold = function(entity)
        -- This is called while holding on the last frame
        -- You could add continuous particle effects or sounds here
        local parent = entity.object:get_attach()
        if parent and parent:is_player() then
            -- Add some particle effects at the staff tip
            minetest.add_particlespawner({
                amount = 5,
                time = 0.1,
                minpos = parent:get_pos(),
                maxpos = parent:get_pos(),
                minvel = {x = -0.5, y = 0.5, z = -0.5},
                maxvel = {x = 0.5, y = 1.5, z = 0.5},
                minacc = {x = 0, y = -0.5, z = 0},
                maxacc = {x = 0, y = -0.5, z = 0},
                minexptime = 1,
                maxexptime = 2,
                minsize = 1,
                maxsize = 3,
                texture = "default_item_smoke.png^[colorize:#00ff00:128",
            })
        end
    end,
    
    on_release = function(entity)
        minetest.log("action", "Staff charge released!")
        -- You could trigger a spell effect here
        local parent = entity.object:get_attach()
        if parent and parent:is_player() then
            -- Create a burst effect when released
            minetest.add_particlespawner({
                amount = 20,
                time = 0.5,
                minpos = parent:get_pos(),
                maxpos = parent:get_pos(),
                minvel = {x = -2, y = -2, z = -2},
                maxvel = {x = 2, y = 2, z = 2},
                minacc = {x = 0, y = 0, z = 0},
                maxacc = {x = 0, y = 0, z = 0},
                minexptime = 0.5,
                maxexptime = 1.5,
                minsize = 2,
                maxsize = 5,
                texture = "default_item_smoke.png^[colorize:#ffff00:200",
            })
        end
    end,
    
    on_complete = function(entity)
        minetest.log("action", "Staff animation completed")
    end
})

-- Create an idle animation for when not charging
charged_models.create_animation("staff_idle", {
    trigger = "none",  -- This will be the default idle state
    transform_keyframes = {
        {
            time = 0,
            position = {x = 3, y = 13, z = 3},
            rotation = {x = 0, y = 0, z = 0},
            easing = "linear"
        }
    },
    speed = 30,
    loop = false,
    hold_behavior = "none"
})

-- Create the model configuration for the charge staff
charged_models.create_model_config("charge_staff_model", {
    position = {x = 3, y = 13, z = 3},
    rotation = {x = 0, y = 0, z = 0},
    scale = {x = 10, y = 10, z = 10},
    mesh = "hand.gltf",  -- You can replace this with a staff model
    textures = {"hand_skin.png^[colorize:#8B4513:128"},  -- Brown-ish color for wood
    look_adjustment = {
        enabled = true,
        factor = 5.0,
        max_vertical_offset = 5.0,
        min_vertical_offset = -2.0,
        rotation_factor = 20.0
    },
    physical = false,
    collide_with_objects = false,
    pointable = false,
    static_save = false,
    animations = {
        idle = "staff_idle",
        charge = "charge_up"
    }
})

-- Register a test item that uses this configuration
minetest.register_craftitem("charged_models:charge_staff", {
    description = "Charge Staff (Test Item)\nHold dig to charge up, release to cast!",
    inventory_image = "default_stick.png^[colorize:#8B4513:128",
    stack_max = 1,
    
    on_use = function(itemstack, user, pointed_thing)
        if user and user:is_player() then
            minetest.chat_send_player(user:get_player_name(), 
                "Hold the dig key to charge the staff, release to cast!")
        end
        return itemstack
    end
})

-- Register the item with the model system
charged_models.register_item_model("charged_models:charge_staff", "charge_staff_model")

-- Example 2: Simple sword with attack animation
charged_models.create_animation("sword_attack", {
    trigger = "press_dig",  -- Triggered on dig key press
    transform_keyframes = {
        -- Starting position
        {
            time = 0,
            position = {x = 3, y = 13, z = 3},
            rotation = {x = 0, y = 0, z = 0},
            easing = "linear"
        },
        -- Wind up
        {
            time = 0.1,
            position = {x = 5, y = 15, z = 5},
            rotation = {x = -30, y = -20, z = 0},
            easing = "ease_out"
        },
        -- Strike position
        {
            time = 0.3,
            position = {x = 1, y = 11, z = 1},
            rotation = {x = 45, y = 10, z = 0},
            easing = "ease_in"
        }
    },
    speed = 60,  -- Faster animation for quick attack
    blend = 0.1,
    loop = false,
    hold_behavior = "none",
    
    on_start = function(entity)
        -- Could add attack sound here
        local parent = entity.object:get_attach()
        if parent and parent:is_player() then
            minetest.sound_play("default_dig_choppy", {
                object = parent,
                gain = 0.5,
                pitch = 1.2
            })
        end
    end
})

charged_models.create_model_config("sword_model", {
    position = {x = 3, y = 13, z = 3},
    rotation = {x = 0, y = 0, z = 0},
    scale = {x = 10, y = 10, z = 10},
    mesh = "hand.gltf",  -- Replace with sword model
    textures = {"hand_skin.png^[colorize:#C0C0C0:200"},  -- Silver color
    look_adjustment = {
        enabled = true,
        factor = 3.0,
        max_vertical_offset = 3.0,
        min_vertical_offset = -1.0,
        rotation_factor = 15.0
    },
    animations = {
        attack = "sword_attack"
    }
})

minetest.register_craftitem("charged_models:test_sword", {
    description = "Test Sword\nPress dig to attack!",
    inventory_image = "default_stick.png",
    stack_max = 1
})

charged_models.register_item_model("charged_models:test_sword", "sword_model")

-- Example 3: Bow with draw and hold animation
charged_models.create_animation("bow_draw", {
    trigger = "hold_dig",
    transform_keyframes = {
        -- Rest position
        {
            time = 0,
            position = {x = 2, y = 12, z = 4},
            rotation = {x = 0, y = -45, z = 0},
            easing = "linear"
        },
        -- Drawing back
        {
            time = 0.8,
            position = {x = 4, y = 13, z = 2},
            rotation = {x = -10, y = -30, z = 5},
            easing = "ease_out"
        },
        -- Fully drawn - hold this position
        {
            time = 1.2,
            position = {x = 5, y = 14, z = 1},
            rotation = {x = -15, y = -20, z = 10},
            easing = "ease_in"
        }
    },
    speed = 30,
    blend = 0.2,
    loop = false,
    hold_behavior = "hold_last_frame",
    
    on_hold = function(entity)
        -- Add tension effect while holding
        local parent = entity.object:get_attach()
        if parent and parent:is_player() then
            -- Small vibration effect could be added here
        end
    end,
    
    on_release = function(entity)
        -- Fire the arrow!
        local parent = entity.object:get_attach()
        if parent and parent:is_player() then
            minetest.chat_send_player(parent:get_player_name(), "Arrow fired!")
            minetest.sound_play("default_dig_snappy", {
                object = parent,
                gain = 0.7,
                pitch = 1.5
            })
        end
    end
})

charged_models.create_model_config("bow_model", {
    position = {x = 2, y = 12, z = 4},
    rotation = {x = 0, y = -45, z = 0},
    scale = {x = 8, y = 8, z = 8},
    mesh = "hand.gltf",  -- Replace with bow model
    textures = {"hand_skin.png^[colorize:#8B4513:150"},  -- Wood color
    animations = {
        draw = "bow_draw"
    }
})

minetest.register_craftitem("charged_models:test_bow", {
    description = "Test Bow\nHold dig to draw, release to fire!",
    inventory_image = "default_stick.png^[colorize:#8B4513:150",
    stack_max = 1
})

charged_models.register_item_model("charged_models:test_bow", "bow_model")

minetest.log("action", "[Charged Models] Example items loaded: charge_staff, test_sword, test_bow")

-- Give the examples to players for testing (optional - remove in production)
minetest.register_chatcommand("get_charged_examples", {
    description = "Get example charged model items for testing",
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if not player then
            return false, "Player not found"
        end
        
        local inv = player:get_inventory()
        inv:add_item("main", "charged_models:charge_staff")
        inv:add_item("main", "charged_models:test_sword") 
        inv:add_item("main", "charged_models:test_bow")
        
        return true, "Added charged model example items to your inventory!"
    end
})


