-- Pickaxe Example: Register existing minetest_game default:pick_wood with a pickaxe.gltf model
-- This example demonstrates how to register an existing item from another module with a 3D model

minetest.register_on_mods_loaded(function()
    
    -- Create realistic digging animation with natural pickaxe swing motion
    charged_models.create_animation("pickaxe_dig", {
        trigger = "hold_dig",  -- Triggered when holding left click (dig key)
        hold_behavior = "loop_until_release",  -- Loop the animation while holding
        loop_speed = 1.5,  -- Slightly faster for more responsive feel
        transform_keyframes = {
            -- Start position (ready to thrust)
            {
                time = 0.0,
                position = {x = 2, y = 13, z = 2},
                rotation = {x = 0, y = 75, z = 0},
                easing = "linear"
            },
            -- Thrust forward rapidly
            {
                time = 0.3,
                position = {x = 0, y = 10, z = 15},
                rotation = {x = 45, y = 90, z = 15},
                easing = "ease_out"
            },
            -- Maximum extension (impact point)
            {
                time = 0.4,
                position = {x = -1, y = 8, z = 18},
                rotation = {x = 60, y = 95, z = 20},
                easing = "ease_in_out"
            },
            -- Pull back toward player
            {
                time = 0.8,
                position = {x = 2, y = 13, z = 2},
                rotation = {x = 0, y = 75, z = 0},
                easing = "ease_in"
            }
        },
        speed = 80,  -- Faster animation speed for more dynamic feel
        blend = 0.15,  -- Smoother blending between keyframes
        loop = false,  -- Individual cycle doesn't loop, but hold_behavior handles the looping
        
        on_start = function(self)
            -- Initialize sound timing and play initial dig sound
            self.dig_cycle_count = 0
            self.last_impact_time = 0
            local parent = self.object:get_attach()
            if parent and parent:is_player() then
                minetest.sound_play("default_dig_cracky", {
                    object = parent,
                    gain = 0.4,
                    pitch = math.random(110, 130) / 100  -- Slight pitch variation
                })
            end
        end,
        
        on_hold = function(self)
            -- Play impact sound at the strike moment of each cycle
            if not self.last_impact_time then
                self.last_impact_time = 0
            end
            
            local current_time = minetest.get_us_time() / 1000000
            -- Calculate when impact should occur based on animation timing
            local cycle_duration = 1.0 / 0.8  -- 1.0 second base cycle / loop_speed
            local impact_time_in_cycle = 0.5  -- Impact occurs at 50% through animation
            
            -- Check if we've hit a new impact moment
            local time_since_start = current_time - (self.animation_start_time or current_time)
            local current_cycle = math.floor(time_since_start / cycle_duration)
            local time_in_cycle = (time_since_start % cycle_duration) / cycle_duration
            
            if current_cycle > (self.last_impact_cycle or -1) and
               time_in_cycle >= impact_time_in_cycle and
               time_in_cycle <= impact_time_in_cycle + 0.1 then
                
                self.last_impact_cycle = current_cycle
                local parent = self.object:get_attach()
                if parent and parent:is_player() then
                    minetest.sound_play("default_dig_cracky", {
                        object = parent,
                        gain = 0.3,
                        pitch = math.random(95, 115) / 100  -- Varied pitch for each strike
                    })
                end
            end
        end,
        
        on_release = function(self)
            -- Clean up timing variables
            self.dig_cycle_count = 0
            self.last_impact_time = 0
            self.last_impact_cycle = nil
            self.animation_start_time = nil
        end
    })
    
    -- Create idle animation for when not digging
    charged_models.create_animation("pickaxe_idle", {
        trigger = "none",  -- Default idle state
        transform_keyframes = {
            {
                time = 0.0,
                position = {x = 3, y = 13, z = 3},
                rotation = {x = 0, y = 80, z = 0},
                easing = "linear"
            }
        },
        speed = 30,
        blend = 0.2,
        loop = false,
        hold_behavior = "none"
    })
    
    -- Create a model configuration for the pickaxe
    charged_models.create_model_config("wooden_pickaxe", {
        position = {x = 3, y = 13, z = 3},
        rotation = {x = 0, y = 80, z = 0},
        scale = {x = 10, y = 10, z = 10},
        mesh = "pickaxe.gltf",
        textures = {"default_tool_woodpick.png"},  -- You can customize this texture
        look_adjustment = {
            enabled = true,
            factor = 5.0,
            max_vertical_offset = 5.0,
            min_vertical_offset = -2.0,
            rotation_factor = 20.0
        },
        physical = false,
        collide_with_objects = false,
        pointable = false,
        static_save = false,
        animations = {
            idle = "pickaxe_idle",
            dig = "pickaxe_dig"
        }
    })

    -- Register the default:pick_wood item with the pickaxe model configuration
    charged_models.register_item_model("default:pick_wood", "wooden_pickaxe")

    minetest.log("action", "[Pickaxe Example] Registered default:pick_wood with pickaxe.gltf model and digging animation")
    
end)
