-- Item Profile API Core
-- Provides the ability to define and apply modular item definition profiles

-- Storage for profiles and applied items
local profiles = {}
local applied_items = {}
local original_definitions = {}

---
-- Register a new item profile
-- @param profile_name string: Unique identifier for the profile
-- @param definition table: Profile definition containing callbacks and overrides
--   - on_apply: function(item_name) - Called when profile is applied to an item
--   - on_remove: function(item_name) - Called when profile is removed from an item
--   - overrides: table - Item definition overrides to apply
--   - filter: function(item_name, item_def) - Optional filter to check if item is compatible
---
function item_profiles.register_profile(profile_name, definition)
    if profiles[profile_name] then
        minetest.log("warning", "[item_profiles] Profile '" .. profile_name .. "' already exists, overwriting")
    end
    
    profiles[profile_name] = {
        name = profile_name,
        on_apply = definition.on_apply,
        on_remove = definition.on_remove,
        overrides = definition.overrides or {},
        filter = definition.filter
    }
    
    minetest.log("action", "[item_profiles] Registered profile: " .. profile_name)
    return true
end

---
-- Apply a profile to an item
-- @param item_name string: The technical name of the item
-- @param profile_name string: The name of the profile to apply
-- @param params table: Optional parameters to pass to the profile
---
function item_profiles.apply_profile(item_name, profile_name, params)
    local profile = profiles[profile_name]
    if not profile then
        minetest.log("warning", "[item_profiles] Profile '" .. profile_name .. "' does not exist")
        return false
    end
    
    local item_def = minetest.registered_items[item_name]
    if not item_def then
        minetest.log("warning", "[item_profiles] Item '" .. item_name .. "' does not exist")
        return false
    end
    
    -- Check filter if it exists
    if profile.filter and not profile.filter(item_name, item_def) then
        minetest.log("warning", "[item_profiles] Item '" .. item_name .. "' does not pass filter for profile '" .. profile_name .. "'")
        return false
    end
    
    -- Store original definition if not already stored
    if not original_definitions[item_name] then
        original_definitions[item_name] = table.copy(item_def)
    end
    
    -- Track applied profile
    if not applied_items[item_name] then
        applied_items[item_name] = {}
    end
    applied_items[item_name][profile_name] = params or {}
    
    -- Apply overrides
    if profile.overrides then
        minetest.override_item(item_name, profile.overrides)
    end
    
    -- Call on_apply callback
    if profile.on_apply then
        profile.on_apply(item_name, params)
    end
    
    minetest.log("action", "[item_profiles] Applied profile '" .. profile_name .. "' to item '" .. item_name .. "'")
    return true
end

---
-- Remove a profile from an item
-- @param item_name string: The technical name of the item
-- @param profile_name string: The name of the profile to remove
---
function item_profiles.remove_profile(item_name, profile_name)
    local profile = profiles[profile_name]
    if not profile then
        minetest.log("warning", "[item_profiles] Profile '" .. profile_name .. "' does not exist")
        return false
    end
    
    if not applied_items[item_name] or not applied_items[item_name][profile_name] then
        minetest.log("warning", "[item_profiles] Profile '" .. profile_name .. "' not applied to item '" .. item_name .. "'")
        return false
    end
    
    -- Call on_remove callback
    if profile.on_remove then
        profile.on_remove(item_name)
    end
    
    -- Remove from tracking
    applied_items[item_name][profile_name] = nil
    
    -- If no more profiles applied, restore original definition
    local has_profiles = false
    for _ in pairs(applied_items[item_name]) do
        has_profiles = true
        break
    end
    
    if not has_profiles and original_definitions[item_name] then
        minetest.override_item(item_name, original_definitions[item_name])
        applied_items[item_name] = nil
        original_definitions[item_name] = nil
    end
    
    minetest.log("action", "[item_profiles] Removed profile '" .. profile_name .. "' from item '" .. item_name .. "'")
    return true
end

---
-- Apply a profile to multiple items matching a pattern
-- @param pattern string: Lua pattern to match item names (e.g., "default:.*")
-- @param profile_name string: The name of the profile to apply
-- @param params table: Optional parameters to pass to the profile
---
function item_profiles.apply_profile_bulk(pattern, profile_name, params)
    local count = 0
    for item_name, _ in pairs(minetest.registered_items) do
        if string.match(item_name, pattern) then
            if item_profiles.apply_profile(item_name, profile_name, params) then
                count = count + 1
            end
        end
    end
    minetest.log("action", "[item_profiles] Applied profile '" .. profile_name .. "' to " .. count .. " items matching '" .. pattern .. "'")
    return count
end

---
-- Check if an item has a specific profile applied
---
function item_profiles.has_profile(item_name, profile_name)
    return applied_items[item_name] and applied_items[item_name][profile_name] ~= nil
end

---
-- Get all profiles applied to an item
---
function item_profiles.get_item_profiles(item_name)
    if not applied_items[item_name] then
        return {}
    end
    local profile_list = {}
    for profile_name, _ in pairs(applied_items[item_name]) do
        table.insert(profile_list, profile_name)
    end
    return profile_list
end

---
-- Get all registered profile names
---
function item_profiles.get_registered_profiles()
    local profile_list = {}
    for profile_name, _ in pairs(profiles) do
        table.insert(profile_list, profile_name)
    end
    return profile_list
end

minetest.log("action", "[item_profiles] API core loaded")
