-- Item Display System for Charged Models
-- Automatically displays held items with proper textures and node boxes
-- Integrates with charged_models animation system

-- ============================================================================
-- NAMESPACE
-- ============================================================================

item_display = {}

-- Storage for display configurations
local display_configs = {}
local active_displays = {}

-- ============================================================================
-- UTILITY FUNCTIONS
-- ============================================================================

local function get_item_texture(item_name)
    if not item_name or item_name == "" then
        return "unknown_item.png"
    end
    
    local def = minetest.registered_items[item_name]
    if not def then
        return "unknown_item.png"
    end
    
    local texture = def.inventory_image
    if not texture or texture == "" then
        texture = def.wield_image
    end
    if not texture or texture == "" then
        if def.tiles and #def.tiles > 0 then
            local tile = def.tiles[1]
            texture = type(tile) == "table" and (tile.name or "unknown_item.png") or tile
        else
            texture = "unknown_item.png"
        end
    end
    
    return texture
end

local function get_node_textures(node_name)
    if not node_name or node_name == "" then
        return {
            "unknown_node.png", "unknown_node.png", "unknown_node.png",
            "unknown_node.png", "unknown_node.png", "unknown_node.png"
        }
    end
    
    local node_def = minetest.registered_nodes[node_name]
    if not node_def then
        return {
            "unknown_node.png", "unknown_node.png", "unknown_node.png",
            "unknown_node.png", "unknown_node.png", "unknown_node.png"
        }
    end
    
    local tiles = node_def.tiles
    if not tiles or #tiles == 0 then
        return {
            "unknown_node.png", "unknown_node.png", "unknown_node.png",
            "unknown_node.png", "unknown_node.png", "unknown_node.png"
        }
    end
    
    local function get_texture_name(tile)
        if type(tile) == "string" then
            return tile
        elseif type(tile) == "table" then
            return tile.name or tile.image or "unknown_node.png"
        end
        return "unknown_node.png"
    end
    
    local textures = {}
    
    if #tiles == 1 then
        local tex = get_texture_name(tiles[1])
        textures = {tex, tex, tex, tex, tex, tex}
    elseif #tiles == 2 then
        local top_bottom = get_texture_name(tiles[1])
        local sides = get_texture_name(tiles[2])
        textures = {top_bottom, top_bottom, sides, sides, sides, sides}
    elseif #tiles == 3 then
        local top = get_texture_name(tiles[1])
        local bottom = get_texture_name(tiles[2])
        local sides = get_texture_name(tiles[3])
        textures = {top, bottom, sides, sides, sides, sides}
    elseif #tiles >= 6 then
        textures = {
            get_texture_name(tiles[1]),
            get_texture_name(tiles[2]),
            get_texture_name(tiles[3]),
            get_texture_name(tiles[4]),
            get_texture_name(tiles[5]),
            get_texture_name(tiles[6]),
        }
    else
        local first = get_texture_name(tiles[1])
        textures = {first, first, first, first, first, first}
    end
    
    return textures
end

local function is_node(item_name)
    if not item_name or item_name == "" then
        return false
    end
    local def = minetest.registered_items[item_name]
    return def and def.type == "node"
end

-- ============================================================================
-- DISPLAY ENTITIES
-- ============================================================================

-- Flat item display entity (for craft items)
minetest.register_entity("charged_models:item_display", {
    initial_properties = {
        visual = "cube",
        visual_size = {x = 0.02, y = 0.02, z = 0.00000000001},
        physical = false,
        collide_with_objects = false,
        pointable = false,
        static_save = false,
        textures = {"unknown_item.png", "unknown_item.png", "unknown_item.png",
                    "unknown_item.png", "unknown_item.png", "unknown_item.png"},
    },
    
    _item_name = "",
    _config = nil,
    
    on_activate = function(self, staticdata)
        self.object:set_armor_groups({immortal = 1})
        
        if staticdata and staticdata ~= "" then
            local data = minetest.deserialize(staticdata)
            if data then
                self._item_name = data.item_name or ""
                self._config = data.config
                
                if self._item_name ~= "" and self._config then
                    self:update_display(self._item_name, self._config)
                end
            end
        end
    end,
    
    get_staticdata = function(self)
        return minetest.serialize({
            item_name = self._item_name,
            config = self._config,
        })
    end,
    
    update_display = function(self, item_name, config)
        self._item_name = item_name
        self._config = config or {}
        
        local props = {
            visual = "cube",
            visual_size = config.scale or {x = 0.02, y = 0.02, z = 0.00000000001},
        }
        
        local texture = config.texture_override or get_item_texture(item_name)
        props.textures = {texture, texture, texture, texture, texture, texture}
        
        self.object:set_properties(props)
    end,
    
    on_step = function(self, dtime)
        local parent = self.object:get_attach()
        if not parent then
            self.object:remove()
            return
        end
    end,
})

-- Full cube display entity (for nodes)
minetest.register_entity("charged_models:node_display", {
    initial_properties = {
        visual = "cube",
        visual_size = {x = 0.01, y = 0.01, z = 0.01},
        physical = false,
        collide_with_objects = false,
        pointable = false,
        static_save = false,
        textures = {
            "unknown_node.png", "unknown_node.png", "unknown_node.png",
            "unknown_node.png", "unknown_node.png", "unknown_node.png"
        },
    },
    
    _node_name = "",
    _config = nil,
    
    on_activate = function(self, staticdata)
        self.object:set_armor_groups({immortal = 1})
        
        if staticdata and staticdata ~= "" then
            local data = minetest.deserialize(staticdata)
            if data then
                self._node_name = data.node_name or ""
                self._config = data.config
                
                if self._node_name ~= "" and self._config then
                    self:update_display(self._node_name, self._config)
                end
            end
        end
    end,
    
    get_staticdata = function(self)
        return minetest.serialize({
            node_name = self._node_name,
            config = self._config,
        })
    end,
    
    update_display = function(self, node_name, config)
        self._node_name = node_name
        self._config = config or {}
        
        local props = {
            visual = "cube",
            visual_size = config.scale or {x = 0.01, y = 0.01, z = 0.01},
        }
        
        local textures = config.texture_override and 
            {config.texture_override, config.texture_override, config.texture_override,
             config.texture_override, config.texture_override, config.texture_override} or
            get_node_textures(node_name)
        props.textures = textures
        
        self.object:set_properties(props)
    end,
    
    on_step = function(self, dtime)
        local parent = self.object:get_attach()
        if not parent then
            self.object:remove()
            return
        end
    end,
})

-- ============================================================================
-- CONFIGURATION API
-- ============================================================================

-- Default configuration
local DEFAULT_CONFIG = {
    display_type = "auto",  -- "auto", "node", "item", "none"
    
    item_config = {
        position = {x = -0.1, y = 0.05, z = 0.18},
        rotation = {x = 25, y = 45, z = 0},
        scale = {x = 0.02, y = 0.02, z = 0.00000000001},
        texture_override = nil,
    },
    
    node_config = {
        position = {x = -0.15, y = 0.05, z = 0.14},
        rotation = {x = 45, y = 45, z = 0},
        scale = {x = 0.01, y = 0.01, z = 0.01},
        texture_override = nil,
    },
}

---
-- Register a custom display configuration for a specific item
-- @param item_name string: The technical name of the item
-- @param config table: Configuration overrides
---
function item_display.register_config(item_name, config)
    if not item_name or item_name == "" then
        minetest.log("warning", "[item_display] Cannot register config with empty item name")
        return false
    end
    
    -- Deep merge with defaults
    local full_config = table.copy(DEFAULT_CONFIG)
    
    if config then
        for key, value in pairs(config) do
            if type(value) == "table" and type(full_config[key]) == "table" then
                full_config[key] = table.copy(full_config[key])
                for k, v in pairs(value) do
                    full_config[key][k] = v
                end
            else
                full_config[key] = value
            end
        end
    end
    
    display_configs[item_name] = full_config
    minetest.log("action", "[item_display] Registered custom config for: " .. item_name)
    return true
end

---
-- Get the display configuration for an item
---
function item_display.get_config(item_name)
    return display_configs[item_name] or DEFAULT_CONFIG
end

---
-- Determine the appropriate display type for an item
---
function item_display.get_display_type(item_name)
    if not item_name or item_name == "" then
        return "none"
    end
    
    local config = item_display.get_config(item_name)
    local display_type = config.display_type
    
    if display_type == "auto" then
        return is_node(item_name) and "node" or "item"
    end
    
    return display_type
end

-- ============================================================================
-- DISPLAY MANAGEMENT
-- ============================================================================

---
-- Spawn a display entity for an item
---
function item_display.spawn(parent, item_name)
    if not parent or not item_name or item_name == "" then
        return nil
    end
    
    local display_type = item_display.get_display_type(item_name)
    
    if display_type == "none" then
        return nil
    end
    
    local config = item_display.get_config(item_name)
    local entity_type, entity_config
    
    if display_type == "node" then
        entity_type = "charged_models:node_display"
        entity_config = config.node_config
    else
        entity_type = "charged_models:item_display"
        entity_config = config.item_config
    end
    
    local parent_pos = parent:get_pos()
    if not parent_pos then
        return nil
    end
    
    local display_entity = minetest.add_entity(
        parent_pos,
        entity_type,
        minetest.serialize({
            item_name = item_name,
            node_name = item_name,
            config = entity_config,
        })
    )
    
    if not display_entity then
        return nil
    end
    
    local entity = display_entity:get_luaentity()
    if entity then
        entity:update_display(item_name, entity_config)
    end
    
    display_entity:set_attach(
        parent,
        "",
        entity_config.position,
        entity_config.rotation,
        true
    )
    
    return display_entity
end

---
-- Update an existing display entity with a new item
---
function item_display.update(display_entity, item_name)
    if not display_entity then
        return false
    end
    
    local entity = display_entity:get_luaentity()
    if not entity then
        return false
    end
    
    local display_type = item_display.get_display_type(item_name)
    local current_type = entity.name
    
    local needs_node = display_type == "node" and current_type == "charged_models:item_display"
    local needs_item = display_type == "item" and current_type == "charged_models:node_display"
    
    if needs_node or needs_item then
        -- Need to recreate entity with different type
        local parent = display_entity:get_attach()
        if parent then
            display_entity:remove()
            return item_display.spawn(parent, item_name)
        end
        return nil
    end
    
    -- Same type, just update texture
    local config = item_display.get_config(item_name)
    local entity_config = display_type == "node" and config.node_config or config.item_config
    
    entity:update_display(item_name, entity_config)
    return display_entity
end

---
-- Remove a display entity
---
function item_display.remove(display_entity)
    if display_entity and display_entity:get_pos() then
        display_entity:remove()
        return true
    end
    return false
end

-- ============================================================================
-- CHARGED MODELS INTEGRATION
-- ============================================================================

-- Default hand model configuration
local DEFAULT_HAND_MODEL = {
    mesh = "hand.gltf",
    textures = {"hand_skin.png"},
    position = {x = 3, y = 13, z = 3},
    rotation = {x = 0, y = 0, z = 0},
    scale = {x = 10, y = 10, z = 10},
    
    look_adjustment = {
        enabled = true,
        factor = 5.0,
        max_vertical_offset = 5.0,
        min_vertical_offset = -2.0,
        rotation_factor = 20.0
    },
    
    animations = {
        idle = "hand_idle",
        swing = "hand_swing",
        place = "hand_place",
    },
}

-- Create default hand animations
charged_models.create_animation("hand_idle", {
    trigger = "none",
    keyframes = {start = 0, stop = 0},
    speed = 30,
    loop = true,
})

charged_models.create_animation("hand_swing", {
    trigger = "press_dig",
    keyframes = {start = 1, stop = 30},
    speed = 45,
    blend = 0.1,
    loop = false,
    transform_keyframes = {
        {time = 0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "linear"},
        {time = 0.3, position = {x = 3, y = 12, z = 5}, rotation = {x = -45, y = 0, z = 0}, easing = "ease_out"},
        {time = 0.6, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in"},
    },
})

charged_models.create_animation("hand_place", {
    trigger = "press_place",
    keyframes = {start = 1, stop = 20},
    speed = 40,
    blend = 0.1,
    loop = false,
    transform_keyframes = {
        {time = 0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "linear"},
        {time = 0.25, position = {x = 3, y = 11, z = 4}, rotation = {x = 15, y = 0, z = 0}, easing = "ease_out"},
        {time = 0.5, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in"},
    },
})

-- Create the default hand model
charged_models.create_model_config("default_hand", DEFAULT_HAND_MODEL)

-- Track display entities per model
local model_displays = {}

-- Extend charged_models dynamic model entity
local function extend_dynamic_model()
    local dynamic_model_def = minetest.registered_entities["charged_models:dynamic_model"]
    if not dynamic_model_def then
        minetest.log("warning", "[item_display] Could not find charged_models:dynamic_model")
        return
    end
    
    local original_on_step = dynamic_model_def.on_step
    
    dynamic_model_def.on_step = function(self, dtime)
        original_on_step(self, dtime)
        
        local parent = self.object:get_attach()
        if not parent or not parent:is_player() then
            return
        end
        
        local wielded = parent:get_wielded_item()
        local item_name = wielded:get_name()
        local model_key = tostring(self.object)
        local display = model_displays[model_key]
        
        -- Only show item displays for items using the default hand model
        -- Skip items that have custom 3D models (not using default_hand)
        local item_config = charged_models.get_item_model_config(item_name)
        local is_using_default_hand = item_config and
            item_config.models and
            #item_config.models == 1 and
            item_config.models[1] == charged_models.get_model_config("default_hand").models[1]
        
        if item_name and item_name ~= "" and is_using_default_hand then
            if display and display:get_pos() then
                local entity = display:get_luaentity()
                if entity then
                    local stored_item = entity._item_name or entity._node_name
                    if stored_item ~= item_name then
                        display = item_display.update(display, item_name)
                        model_displays[model_key] = display
                    end
                end
            else
                display = item_display.spawn(self.object, item_name)
                if display then
                    model_displays[model_key] = display
                end
            end
        else
            -- No item wielded or item has custom model, remove display
            if display and display:get_pos() then
                item_display.remove(display)
                model_displays[model_key] = nil
            end
        end
    end
    
    minetest.log("action", "[item_display] Extended charged_models:dynamic_model")
end

minetest.after(0, extend_dynamic_model)

-- ============================================================================
-- AUTOMATIC REGISTRATION
-- ============================================================================

---
-- Automatically apply display system to all items
---
function item_display.apply_to_all()
    local count = 0
    
    for item_name, item_def in pairs(minetest.registered_items) do
        -- Skip items that already have custom wield_image or are explicitly blank
        if not item_def.wield_image or item_def.wield_image == "" then
            -- Register with charged_models
            local success = pcall(function()
                charged_models.register_item_model(item_name, "default_hand")
            end)
            
            if success then
                count = count + 1
            end
        end
    end
    
    minetest.log("action", "[item_display] Applied display system to " .. count .. " items")
    return count
end

-- Apply to all items when mods are loaded
minetest.register_on_mods_loaded(function()
    item_display.apply_to_all()
end)

minetest.log("action", "[item_display] Item display system loaded")
