# Item Profiles API for Luanti

A modular API for defining and applying reusable profiles to items in Luanti. This mod allows you to create item behavior templates that can be easily applied to single items, groups of items, or all items at once.

## Features

- **Modular Profile System**: Define reusable item modification profiles
- **Bulk Operations**: Apply profiles to multiple items using pattern matching
- **Callbacks**: Execute custom code when profiles are applied or removed
- **Filters**: Control which items can receive specific profiles
- **Original State Preservation**: Automatically restores original item definitions when profiles are removed
- **Included Profile**: Ships with an "invisible items" profile that hides wielded items

## Installation

1. Download or clone this repository
2. Place the `item_profiles` folder in your Luanti `mods` directory
3. Enable the mod in your world settings

## API Reference

### Registering a Profile

```lua
item_profiles.register_profile(profile_name, definition)
```

**Parameters:**
- `profile_name` (string): Unique identifier for the profile
- `definition` (table): Profile definition with the following fields:
  - `overrides` (table): Item definition properties to override
  - `on_apply` (function, optional): Callback when profile is applied to an item
    - Parameters: `item_name`, `params`
  - `on_remove` (function, optional): Callback when profile is removed from an item
    - Parameters: `item_name`
  - `filter` (function, optional): Function to check if item is compatible
    - Parameters: `item_name`, `item_def`
    - Returns: `true` to allow, `false` to deny

**Example:**
```lua
item_profiles.register_profile("glowing", {
    overrides = {
        light_source = 14,
    },
    on_apply = function(item_name, params)
        minetest.log("action", "Made " .. item_name .. " glow")
    end,
    filter = function(item_name, item_def)
        -- Only apply to tools
        return item_def.type == "tool"
    end
})
```

### Applying a Profile

```lua
item_profiles.apply_profile(item_name, profile_name, params)
```

Applies a profile to a single item.

**Parameters:**
- `item_name` (string): Technical name of the item
- `profile_name` (string): Name of the profile to apply
- `params` (table, optional): Custom parameters passed to callbacks

**Returns:** `true` on success, `false` on failure

**Example:**
```lua
item_profiles.apply_profile("default:torch", "glowing")
```

### Applying Profiles in Bulk

```lua
item_profiles.apply_profile_bulk(pattern, profile_name, params)
```

Applies a profile to all items matching a Lua pattern.

**Parameters:**
- `pattern` (string): Lua pattern to match item names
- `profile_name` (string): Name of the profile to apply
- `params` (table, optional): Custom parameters passed to callbacks

**Returns:** Number of items modified

**Example:**
```lua
-- Make all default mod items invisible
item_profiles.apply_profile_bulk("^default:.*", "invisible")

-- Make all tools glow
item_profiles.apply_profile_bulk(".*pickaxe$", "glowing")
item_profiles.apply_profile_bulk(".*axe$", "glowing")
```

### Removing a Profile

```lua
item_profiles.remove_profile(item_name, profile_name)
```

Removes a profile from an item and restores its original definition if no other profiles remain.

**Returns:** `true` on success, `false` on failure

### Checking for Profiles

```lua
item_profiles.has_profile(item_name, profile_name)
```

Checks if an item has a specific profile applied.

**Returns:** `true` if profile is applied, `false` otherwise

### Getting Item Profiles

```lua
item_profiles.get_item_profiles(item_name)
```

Returns a list of all profile names applied to an item.

**Returns:** Table of profile names (empty table if none)

### Getting All Profiles

```lua
item_profiles.get_registered_profiles()
```

Returns a list of all registered profile names.

**Returns:** Table of profile names

## Included Profile: Invisible Items

The mod includes a ready-to-use "invisible" profile that makes items disappear when wielded. By default, it applies to **all items**.

## License

Source Code: MIT
Media: CC0

## Credits
Created By TomCon for the Luanti community.