minetest.register_on_mods_loaded(function()
    
    -- Idle animation with subtle flickering effect
    charged_models.create_animation("torch_idle", {
        trigger = "none",  -- No trigger, plays automatically when idle
        loop = true,
        loop_speed = 1.5,  -- Slightly faster for flame effect
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in_out"},
            {time = 0.3, position = {x = 3, y = 13.001, z = 3}, rotation = {x = .5, y = 0, z = .5}, easing = "ease_in_out"},
            {time = 0.6, position = {x = 3, y = 13, z = 3}, rotation = {x = .5, y = 0, z = .5}, easing = "ease_in_out"},
            {time = 0.9, position = {x = 3, y = 13.005, z = 3}, rotation = {x = .5, y = 0, z = .5}, easing = "ease_in_out"},
            {time = 1.2, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in_out"}
        }
    })
    
    -- Wave/swing animation
    charged_models.create_animation("torch_wave", {
        trigger = "press_dig",
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 0.15, position = {x = 5, y = 14, z = 2}, rotation = {x = -30, y = 45, z = 10}, easing = "ease_out"},
            {time = 0.35, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in"}
        }
    })
    
    -- Raise torch animation (for lighting)
    charged_models.create_animation("torch_raise", {
        trigger = "press_place",
        transform_keyframes = {
            {time = 0.0, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}},
            {time = 0.2, position = {x = 2, y = 16, z = 2}, rotation = {x = -20, y = 0, z = 0}, easing = "ease_out"},
            {time = 0.5, position = {x = 3, y = 13, z = 3}, rotation = {x = 0, y = 0, z = 0}, easing = "ease_in"}
        }
    })
    
    -- Torch model configuration
    charged_models.create_model_config("held_torch", {
        mesh = "torch.obj",
        textures = {"charged_models_torch_texture.png"},
        glow = 3,
        scale = {x = 10, y = 10, z = 10},
        position = {x = 3, y = 13, z = 3},
        rotation = {x = 0, y = 0, z = 0},
        animations = {
            idle = "torch_idle",
            wave = "torch_wave",
            raise = "torch_raise"
        }
    })
    
    -- Register the torch model to default torch item
    charged_models.register_item_model("default:torch", "held_torch")
    
end)

-- Particle spawner for fire effects
local torch_particle_spawners = {}

-- Function to create fire particles at torch position
local function spawn_torch_particles(player, model_pos)
    local player_name = player:get_player_name()
    
    -- Calculate world position of the torch flame (tip of the torch)
    local player_pos = player:get_pos()
    local player_yaw = player:get_look_horizontal()
    local player_pitch = player:get_look_vertical()
    
    -- Offset for the flame position (adjust based on your torch model)
    local flame_offset = {x = 0, y = 0.5, z = 0.3}  -- Slightly above and in front
    
    -- Rotate offset based on player's yaw
    local cos_yaw = math.cos(player_yaw)
    local sin_yaw = math.sin(player_yaw)
    local rotated_x = flame_offset.x * cos_yaw - flame_offset.z * sin_yaw
    local rotated_z = flame_offset.x * sin_yaw + flame_offset.z * cos_yaw
    
    local flame_pos = {
        x = player_pos.x + rotated_x,
        y = player_pos.y + flame_offset.y + 1.5,  -- Add player eye height
        z = player_pos.z + rotated_z
    }
    
    -- Main fire particles
    minetest.add_particlespawner({
        amount = 5,
        time = 0.1,
        minpos = {x = flame_pos.x - 0.05, y = flame_pos.y, z = flame_pos.z - 0.05},
        maxpos = {x = flame_pos.x + 0.05, y = flame_pos.y + 0.1, z = flame_pos.z + 0.05},
        minvel = {x = -0.1, y = 0.3, z = -0.1},
        maxvel = {x = 0.1, y = 0.8, z = 0.1},
        minacc = {x = 0, y = 0.1, z = 0},
        maxacc = {x = 0, y = 0.2, z = 0},
        minexptime = 0.3,
        maxexptime = 0.6,
        minsize = 1.5,
        maxsize = 2.5,
        collisiondetection = false,
        vertical = false,
        texture = "fire_basic_flame.png",
        glow = 14
    })
    
    -- Smoke particles
    minetest.add_particlespawner({
        amount = 2,
        time = 0.1,
        minpos = {x = flame_pos.x - 0.05, y = flame_pos.y + 0.1, z = flame_pos.z - 0.05},
        maxpos = {x = flame_pos.x + 0.05, y = flame_pos.y + 0.2, z = flame_pos.z + 0.05},
        minvel = {x = -0.05, y = 0.4, z = -0.05},
        maxvel = {x = 0.05, y = 0.7, z = 0.05},
        minacc = {x = 0, y = 0, z = 0},
        maxacc = {x = 0, y = 0, z = 0},
        minexptime = 0.5,
        maxexptime = 1.0,
        minsize = 2.0,
        maxsize = 3.5,
        collisiondetection = false,
        vertical = false,
        texture = "fire_basic_flame.png^[colorize:#000000:180",
        glow = 3
    })
    
    -- Ember particles (small sparks)
    if math.random() > 0.7 then  -- Only spawn occasionally
        minetest.add_particlespawner({
            amount = 1,
            time = 0.1,
            minpos = {x = flame_pos.x - 0.03, y = flame_pos.y, z = flame_pos.z - 0.03},
            maxpos = {x = flame_pos.x + 0.03, y = flame_pos.y + 0.05, z = flame_pos.z + 0.03},
            minvel = {x = -0.2, y = 0.2, z = -0.2},
            maxvel = {x = 0.2, y = 0.6, z = 0.2},
            minacc = {x = 0, y = -0.5, z = 0},
            maxacc = {x = 0, y = -0.3, z = 0},
            minexptime = 0.2,
            maxexptime = 0.4,
            minsize = 0.5,
            maxsize = 1.0,
            collisiondetection = true,
            vertical = false,
            texture = "fire_basic_flame.png",
            glow = 14
    })
    end
end

-- Track torch particle spawning
minetest.register_globalstep(function(dtime)
    for _, player in ipairs(minetest.get_connected_players()) do
        local player_name = player:get_player_name()
        local wielded = player:get_wielded_item()
        local item_name = wielded:get_name()
        
        -- Check if player is holding a torch
        if item_name == "default:torch" then
            -- Initialize timer if not exists
            if not torch_particle_spawners[player_name] then
                torch_particle_spawners[player_name] = {timer = 0}
            end
            
            -- Update timer
            torch_particle_spawners[player_name].timer = torch_particle_spawners[player_name].timer + dtime
            
            -- Spawn particles every 0.1 seconds
            if torch_particle_spawners[player_name].timer >= 0.1 then
                spawn_torch_particles(player, nil)
                torch_particle_spawners[player_name].timer = 0
            end
        else
            -- Clean up if no longer holding torch
            if torch_particle_spawners[player_name] then
                torch_particle_spawners[player_name] = nil
            end
        end
    end
end)

-- Clean up when player leaves
minetest.register_on_leaveplayer(function(player)
    local player_name = player:get_player_name()
    torch_particle_spawners[player_name] = nil
end)

minetest.log("action", "[Charged Models] Held torch with fire particles example loaded.")
