-- Charged Punch Hand using the charged_models API
-- Implements charging mechanic with wind-up and punch release animations

-- Configuration
local CHARGE_TIME = 0.5  -- seconds to fully charge
local CHARGE_MAX = 20    -- HUD bar segments
local PUNCH_RANGE = 5.0  -- Maximum reach distance
local MAX_CHARGE_THRESHOLD = 1.0

-- Position configurations
local HAND_FORWARD_POSITION = {x = 3, y = 13, z = 3}
local HAND_PULLBACK_POSITION = {x = 2, y = 14, z = 2}
local HAND_PUNCH_POSITION = {x = 3, y = 15, z = 7}

-- Rotation configurations
local HAND_BASE_ROTATION = {x = 0, y = 0, z = 0}
local HAND_PULLBACK_ROTATION = {x = 0, y = 0, z = -90}
local HAND_PUNCH_ROTATION = {x = 0, y = 0, z = 90}

-- Sound configuration
local IMPACT_SOUNDS = {
    nodes = {},
    entities = {},
    default_node = "impact",
    default_entity = "punch",
    miss = "woosh",
    max_charge_hit = "punch",
    clench = "clench",
}

-- Track player states
local player_states = {}

-- Initialize player state
local function init_player_state(player_name)
    if not player_states[player_name] then
        player_states[player_name] = {
            charging = false,
            charge_start = 0,
            charge_amount = 0,
            hud_bar = nil,
        }
    end
end

-- Create charge animation - winds up to pullback position and holds
charged_models.create_animation("charge_windup", {
    trigger = "hold_dig",  -- Hold left mouse button
    keyframes = {start = 0, stop = 12},
    speed = 24,
    blend = 0,
    loop = false,
    hold_behavior = "hold_last_frame",
    transform_keyframes = {
        {
            time = 0,
            position = HAND_FORWARD_POSITION,
            rotation = HAND_BASE_ROTATION,
            easing = "linear"
        },
        {
            time = CHARGE_TIME,
            position = HAND_PULLBACK_POSITION,
            rotation = HAND_PULLBACK_ROTATION,
            easing = "ease_out"
        }
    },
    on_start = function(self)
        -- Play clench sound when starting charge
        local parent = self.object:get_attach()
        if parent and parent:is_player() then
            local player_pos = parent:get_pos()
            minetest.sound_play(IMPACT_SOUNDS.clench, {
                pos = player_pos,
                max_hear_distance = 8,
                gain = 0.7,
            })
        end
    end,
    on_hold = function(self)
        -- Called when animation reaches end and is holding
        -- Could add visual effects here for fully charged state
    end
})

-- Create punch release animation
charged_models.create_animation("punch_release", {
    trigger = "release_dig",  -- Release left mouse button
    keyframes = {start = 12, stop = 0},
    speed = 48,
    blend = 0.05,
    loop = false,
    transform_keyframes = {
        {
            time = 0,
            position = HAND_PULLBACK_POSITION,
            rotation = HAND_PULLBACK_ROTATION,
            easing = "linear"
        },
        {
            time = 0.1,
            position = HAND_PUNCH_POSITION,
            rotation = HAND_PUNCH_ROTATION,
            easing = "ease_out"
        },
        {
            time = 0.4,
            position = HAND_PUNCH_POSITION,
            rotation = HAND_PUNCH_ROTATION,
            easing = "linear"
        },
        {
            time = 0.55,
            position = HAND_FORWARD_POSITION,
            rotation = HAND_BASE_ROTATION,
            easing = "ease_in"
        }
    },
    on_start = function(self)
        -- Perform punch logic when animation starts
        local parent = self.object:get_attach()
        if not parent or not parent:is_player() then return end
        
        local player = parent
        local player_name = player:get_player_name()
        local state = player_states[player_name]
        if not state then return end
        
        -- Get what player is pointing at
        local pointed = get_pointed_target(player)
        
        -- Deal damage
        local hit_something = false
        local hit_info = nil
        if pointed then
            hit_something, hit_info = damage_target(player, pointed, state.charge_amount)
        end
        
        -- Play sound
        local sound_name = IMPACT_SOUNDS.miss
        
        if hit_something and hit_info then
            if state.charge_amount >= MAX_CHARGE_THRESHOLD and IMPACT_SOUNDS.max_charge_hit then
                sound_name = IMPACT_SOUNDS.max_charge_hit
            else
                if hit_info.type == "entity" then
                    sound_name = IMPACT_SOUNDS.entities[hit_info.name] or IMPACT_SOUNDS.default_entity
                elseif hit_info.type == "node" then
                    sound_name = IMPACT_SOUNDS.nodes[hit_info.name] or IMPACT_SOUNDS.default_node
                end
            end
        end
        
        local player_pos = player:get_pos()
        minetest.sound_play(sound_name, {
            pos = player_pos,
            max_hear_distance = 16,
            gain = hit_something and 1.0 or 0.8,
        })
        
        -- Show message if not fully charged when hitting node
        if hit_info and hit_info.type == "node" and state.charge_amount < MAX_CHARGE_THRESHOLD then
            minetest.chat_send_player(player_name, 
                "Punch not fully charged! (" .. math.floor(state.charge_amount * 100) .. "%)")
        end
    end,
    on_complete = function(self)
        -- Reset charge state after punch completes
        local parent = self.object:get_attach()
        if parent and parent:is_player() then
            local player_name = parent:get_player_name()
            local state = player_states[player_name]
            if state then
                state.charging = false
                state.charge_amount = 0
            end
        end
    end
})

-- Create idle animation (rest position)
charged_models.create_animation("idle", {
    trigger = "none",
    keyframes = {start = 0, stop = 0},
    speed = 0,
    blend = 0.1,
    loop = true,
    transform_keyframes = {
        {
            time = 0,
            position = HAND_FORWARD_POSITION,
            rotation = HAND_BASE_ROTATION,
            easing = "linear"
        }
    }
})

-- Create the hand model configuration
charged_models.create_model_config("charged_hand", {
    position = HAND_FORWARD_POSITION,
    rotation = HAND_BASE_ROTATION,
    scale = {x = 10, y = 10, z = 10},
    mesh = "hand.gltf",
    textures = {"charged_models_hand_skin.png"},
    look_adjustment = {
        enabled = true,
        factor = 5.0,
        max_vertical_offset = 5.0,
        min_vertical_offset = -2.0,
        rotation_factor = 20.0
    },
    physical = false,
    collide_with_objects = false,
    pointable = false,
    static_save = false,
    animations = {
        idle = "idle",
        charge_windup = "charge_windup",
        punch_release = "punch_release"
    }
})

-- Register empty hand (air) with the charged hand model
charged_models.register_item_model("", "charged_hand")

-- Function to perform raycast and get what player is pointing at
function get_pointed_target(player)
    local pos = player:get_pos()
    local eye_height = player:get_properties().eye_height or 1.625
    local eye_pos = vector.add(pos, {x = 0, y = eye_height, z = 0})
    
    local look_dir = player:get_look_dir()
    local end_pos = vector.add(eye_pos, vector.multiply(look_dir, PUNCH_RANGE))
    
    local ray = minetest.raycast(eye_pos, end_pos, true, false)
    
    for pointed_thing in ray do
        if pointed_thing.type == "object" then
            local obj = pointed_thing.ref
            if obj ~= player and obj:get_luaentity() and 
               obj:get_luaentity().name ~= "charged_models:dynamic_model" then
                return pointed_thing
            end
        elseif pointed_thing.type == "node" then
            return pointed_thing
        end
    end
    
    return nil
end

-- Function to deal damage to target
function damage_target(player, pointed_thing, charge_amount)
    if not pointed_thing then return false, nil end
    
    local base_damage = 2
    local bonus_damage = 18 * charge_amount
    local total_damage = base_damage + bonus_damage
    local hit_something = false
    local hit_info = {type = nil, name = nil}
    
    if pointed_thing.type == "object" then
        local obj = pointed_thing.ref
        if obj and obj:get_luaentity() then
            local entity = obj:get_luaentity()
            
            obj:punch(player, 1.0, {
                full_punch_interval = 1.0,
                damage_groups = {fleshy = total_damage},
            }, nil)
            
            minetest.log("action", player:get_player_name() .. 
                " punched entity (damage: " .. total_damage .. ")")
            
            hit_something = true
            hit_info.type = "entity"
            hit_info.name = entity.name
        end
    elseif pointed_thing.type == "node" then
        local node_pos = pointed_thing.under
        local node = minetest.get_node(node_pos)
        
        local node_def = minetest.registered_nodes[node.name]
        if node_def and not node_def.groups.immortal then
            if charge_amount >= MAX_CHARGE_THRESHOLD then
                if not minetest.is_protected(node_pos, player:get_player_name()) then
                    minetest.node_dig(node_pos, node, player)
                    
                    minetest.log("action", player:get_player_name() .. 
                        " broke " .. node.name .. " with fully charged punch")
                    hit_something = true
                    hit_info.type = "node"
                    hit_info.name = node.name
                else
                    minetest.chat_send_player(player:get_player_name(), 
                        "This area is protected!")
                    hit_something = true
                    hit_info.type = "node"
                    hit_info.name = node.name
                end
            else
                hit_something = true
                hit_info.type = "node"
                hit_info.name = node.name
            end
        end
    end
    
    return hit_something, hit_info
end

-- Create charge HUD bar
local function create_charge_hud(player)
    return player:hud_add({
        hud_elem_type = "statbar",
        position = {x = 0.5, y = 0.85},
        text = "charged_models_rainbow.png",
        text2 = "charged_models_hand_skin.png",
        number = 0,
        item = CHARGE_MAX,
        direction = 0,
        size = {x = 24, y = 24},
        offset = {x = 0, y = 0},
    })
end

-- Remove charge HUD
local function remove_charge_hud(player)
    local player_name = player:get_player_name()
    local state = player_states[player_name]
    
    if state and state.hud_bar then
        player:hud_remove(state.hud_bar)
        state.hud_bar = nil
    end
end

-- Override default hand to disable mining
minetest.register_item(":", {
    type = "none",
    wield_image = "blank.png",
    wield_scale = {x = 1, y = 1, z = 1},
    tool_capabilities = {
        full_punch_interval = 1.0,
        max_drop_level = 0,
        groupcaps = {},
        damage_groups = {},
    },
    on_use = function(itemstack, user, pointed_thing)
        return nil
    end,
})

-- Prevent default digging behavior
minetest.register_on_punchnode(function(pos, node, puncher, pointed_thing)
    if not puncher or not puncher:is_player() then
        return false
    end
    
    local wielded = puncher:get_wielded_item()
    if wielded:is_empty() then
        return true
    end
    
    return false
end)

-- Global step to handle charge tracking and HUD
minetest.register_globalstep(function(dtime)
    for _, player in ipairs(minetest.get_connected_players()) do
        local wielded = player:get_wielded_item()
        local player_name = player:get_player_name()
        local ctrl = player:get_player_control()
        
        init_player_state(player_name)
        local state = player_states[player_name]
        
        -- Only handle empty hand
        if wielded:is_empty() then
            -- Handle charging
            if ctrl.LMB then
                if not state.charging then
                    -- Start charging
                    state.charging = true
                    state.charge_start = minetest.get_us_time() / 1000000
                    state.charge_amount = 0
                    
                    if not state.hud_bar then
                        state.hud_bar = create_charge_hud(player)
                    end
                end
                
                -- Update charge amount
                local current_time = minetest.get_us_time() / 1000000
                local elapsed = current_time - state.charge_start
                state.charge_amount = math.min(elapsed / CHARGE_TIME, 1.0)
                
                -- Update HUD
                if state.hud_bar then
                    player:hud_change(state.hud_bar, "number", 
                        math.floor(state.charge_amount * CHARGE_MAX))
                end
                
            elseif state.charging then
                -- Released - reset charge state immediately
                state.charging = false
                state.charge_amount = 0
                remove_charge_hud(player)
            end
        else
            -- Clean up if wielding something else
            if state.charging then
                state.charging = false
                state.charge_amount = 0
                remove_charge_hud(player)
            end
        end
    end
end)

-- Clean up on player leave
minetest.register_on_leaveplayer(function(player)
    local player_name = player:get_player_name()
    
    if player_states[player_name] then
        remove_charge_hud(player)
        player_states[player_name] = nil
    end
end)

minetest.log("action", "[Charged Hand] Loaded using charged_models API")
