-- invisible_item.lua
-- Invisible Items Profile Implementation
-- Uses the Item Profile API to make items invisible when wielded

-- Ensure the item_profiles API is available
if not item_profiles then
    error("Item Profile API is required but not loaded!")
end

---
-- Register the "invisible" profile
-- This profile makes items invisible when held in hand
---
item_profiles.register_profile("invisible", {
    -- Override settings to make the item invisible
    overrides = {
        wield_image = "",
        wield_scale = {x = 0, y = 0, z = 0},
    },
    
    -- Called when the profile is applied to an item
    on_apply = function(item_name, params)
        minetest.log("action", "[invisible_item] Made item invisible: " .. item_name)
    end,
    
    -- Called when the profile is removed from an item
    on_remove = function(item_name)
        minetest.log("action", "[invisible_item] Restored visibility for: " .. item_name)
    end,
    
    -- Optional: Filter to only apply to certain items
    -- Return true to allow, false to deny
    filter = function(item_name, item_def)
        -- Example: Don't make air invisible (it's already nothing)
        if item_name == "" or item_name == "air" or item_name == "ignore" then
            return false
        end
        
        -- Only apply to items that have a wield_image or inventory_image
        -- (i.e., items that can actually be held)
        if item_def.wield_image or item_def.inventory_image then
            return true
        end
        
        return true -- Allow by default
    end
})

---
-- USAGE
---

--INVISIBLE DEFAULT HAND PROFILE & APPLY
item_profiles.register_profile("invisible_default_hand", {
    overrides = {
    wield_image = "blank.png",
    },
})

minetest.register_on_mods_loaded(function()
    item_profiles.apply_profile("", "invisible_default_hand")

end)

-- Example 1: Make all items invisible
minetest.register_on_mods_loaded(function()
    -- Apply invisible profile to all registered items
    local count = item_profiles.apply_profile_bulk(".*", "invisible")
    minetest.log("action", "[invisible_item] Made " .. count .. " items invisible")
end)

--[[
-- Example 2: Make only specific mod's items invisible
minetest.register_on_mods_loaded(function()
    -- Make only default mod items invisible
    item_profiles.apply_profile_bulk("^default:.*", "invisible")
    
    -- Make only tools invisible
    item_profiles.apply_profile_bulk(".*_pickaxe$", "invisible")
    item_profiles.apply_profile_bulk(".*_shovel$", "invisible")
    item_profiles.apply_profile_bulk(".*_axe$", "invisible")
    item_profiles.apply_profile_bulk(".*_sword$", "invisible")
end)
]]




--[[
-- Example 4: Make items invisible with a toggle command
minetest.register_chatcommand("make_invisible", {
    params = "<item_name>",
    description = "Make an item invisible when wielded",
    privs = {server = true},
    func = function(name, param)
        if param == "" then
            return false, "Please specify an item name"
        end
        
        if item_profiles.apply_profile(param, "invisible") then
            return true, "Made " .. param .. " invisible"
        else
            return false, "Failed to make " .. param .. " invisible"
        end
    end
})

minetest.register_chatcommand("make_visible", {
    params = "<item_name>",
    description = "Restore visibility to an item",
    privs = {server = true},
    func = function(name, param)
        if param == "" then
            return false, "Please specify an item name"
        end
        
        if item_profiles.remove_profile(param, "invisible") then
            return true, "Restored visibility to " .. param
        else
            return false, "Failed to restore visibility to " .. param
        end
    end
})
]]

--[[
-- Example 5: Conditional invisibility based on player privilege
item_profiles.register_profile("invisible_for_admin", {
    overrides = {
        wield_image = "invisible_items_blank.png",
        wield_scale = {x = 0, y = 0, z = 0},
    },
    
    on_apply = function(item_name, params)
        -- Could add metadata or other tracking here
        minetest.log("action", "[invisible_item] Admin-only invisible item: " .. item_name)
    end,
    
    filter = function(item_name, item_def)
        -- Only apply to weapons
        return string.match(item_name, "sword") or string.match(item_name, "gun")
    end
})
]]

--[[
-- Example 6: Time-based invisibility
minetest.register_on_mods_loaded(function()
    -- Make items invisible only at night
    local function check_time()
        local time_of_day = minetest.get_timeofday()
        -- Night time is roughly 0.75 to 0.25
        if time_of_day > 0.75 or time_of_day < 0.25 then
            -- It's night, make items invisible
            if not item_profiles.has_profile("default:torch", "invisible") then
                item_profiles.apply_profile_bulk("^default:torch$", "invisible")
            end
        else
            -- It's day, make items visible
            if item_profiles.has_profile("default:torch", "invisible") then
                item_profiles.remove_profile("default:torch", "invisible")
            end
        end
    end
    
    -- Check every 10 seconds
    local timer = 0
    minetest.register_globalstep(function(dtime)
        timer = timer + dtime
        if timer >= 10 then
            check_time()
            timer = 0
        end
    end)
end)
]]

--[[
-- Example 7: List all invisible items
minetest.register_chatcommand("list_invisible", {
    params = "",
    description = "List all items with the invisible profile",
    privs = {server = true},
    func = function(name, param)
        local invisible_items = {}
        
        for item_name, _ in pairs(minetest.registered_items) do
            if item_profiles.has_profile(item_name, "invisible") then
                table.insert(invisible_items, item_name)
            end
        end
        
        if #invisible_items == 0 then
            return true, "No items are currently invisible"
        end
        
        return true, "Invisible items (" .. #invisible_items .. "): " .. table.concat(invisible_items, ", ")
    end
})
]]

minetest.log("action", "[invisible_item] Profile loaded and registered")
