-- Item Model Attachment Registry API for Luanti with Sequential Keyframe Animation System
-- Enhanced with full keybinding support and hold mechanics
-- Allows any held item to have multiple attached 3D models with configurable parameters and animations
-- Supports arbitrary sequential transform keyframes with interpolation

-- Initialize the global API namespace first
charged_models = charged_models or {}

-- Registry storage for item-model associations
local item_model_registry = {}

-- Model configuration templates
local model_configurations = {}

-- Animation configuration templates
local animation_configurations = {}

-- Track model entities per player (supports multiple models per player)
local player_models = {}

-- Track player input states for all keybindings
local player_input_states = {}

-- Available keybindings in the engine
local AVAILABLE_KEYS = {
    "up", "down", "left", "right",
    "jump", "aux1", "sneak",
    "dig", "place", "zoom"
}

-- Default configuration values for a single model
local DEFAULT_MODEL_CONFIG = {
    position = {x = 3, y = 13, z = 3},
    rotation = {x = 0, y = 0, z = 0},
    scale = {x = 10, y = 10, z = 10},
    mesh = "hand.gltf",
    textures = {"hand_skin.png"},
    look_adjustment = {
        enabled = true,
        factor = 5.0,
        max_vertical_offset = 5.0,
        min_vertical_offset = -2.0,
        rotation_factor = 20.0
    },
    physical = false,
    collide_with_objects = false,
    pointable = false,
    static_save = false,
    animations = {}  -- Animation configurations
}

-- Default animation configuration
local DEFAULT_ANIMATION = {
    name = "default",
    trigger = "none",  -- none, press_<key>, hold_<key>, release_<key> (e.g., press_dig, hold_place, release_jump)
    keyframes = {
        start = 0,
        stop = 0
    },
    speed = 30,
    blend = 0,
    loop = false,
    loop_speed = 1.0,  -- Speed multiplier for animation loops (1.0 = normal speed, 0.5 = half speed, 2.0 = double speed)
    hold_behavior = "none",  -- none, hold_last_frame, loop_until_release
    transform_keyframes = nil,  -- Array of sequential transform keyframes
    on_start = nil,  -- Optional callback function
    on_complete = nil,  -- Optional callback function
    on_hold = nil,  -- Optional callback for hold behavior
    on_release = nil  -- Optional callback for release
}

-- Default transform keyframe structure
local DEFAULT_TRANSFORM_KEYFRAME = {
    time = 0,  -- Time in seconds from animation start
    position = nil,  -- {x, y, z} - if nil, uses base position
    rotation = nil,  -- {x, y, z} - if nil, uses base rotation
    easing = "linear"  -- linear, ease_in, ease_out, ease_in_out
}

-- Helper function to deep copy tables
local function deep_copy(orig)
    local copy
    if type(orig) == 'table' then
        copy = {}
        for key, value in pairs(orig) do
            copy[key] = deep_copy(value)
        end
    else
        copy = orig
    end
    return copy
end

-- Helper function to apply easing function
local function apply_easing(progress, easing_type)
    if easing_type == "ease_out" then
        return 1 - (1 - progress) * (1 - progress)
    elseif easing_type == "ease_in" then
        return progress * progress
    elseif easing_type == "ease_in_out" then
        return progress < 0.5 and 2 * progress * progress or 1 - math.pow(-2 * progress + 2, 2) / 2
    else  -- linear
        return progress
    end
end

-- Helper function to interpolate between two keyframes
local function interpolate_keyframes(kf1, kf2, progress, base_pos, base_rot)
    local eased_progress = apply_easing(progress, kf2.easing or "linear")
    
    local result_pos = base_pos
    local result_rot = base_rot
    
    -- Interpolate position
    if kf1.position and kf2.position then
        result_pos = {
            x = kf1.position.x + (kf2.position.x - kf1.position.x) * eased_progress,
            y = kf1.position.y + (kf2.position.y - kf1.position.y) * eased_progress,
            z = kf1.position.z + (kf2.position.z - kf1.position.z) * eased_progress
        }
    elseif kf2.position then
        result_pos = {
            x = base_pos.x + (kf2.position.x - base_pos.x) * eased_progress,
            y = base_pos.y + (kf2.position.y - base_pos.y) * eased_progress,
            z = base_pos.z + (kf2.position.z - base_pos.z) * eased_progress
        }
    elseif kf1.position then
        result_pos = kf1.position
    end
    
    -- Interpolate rotation
    if kf1.rotation and kf2.rotation then
        result_rot = {
            x = kf1.rotation.x + (kf2.rotation.x - kf1.rotation.x) * eased_progress,
            y = kf1.rotation.y + (kf2.rotation.y - kf1.rotation.y) * eased_progress,
            z = kf1.rotation.z + (kf2.rotation.z - kf1.rotation.z) * eased_progress
        }
    elseif kf2.rotation then
        result_rot = {
            x = base_rot.x + (kf2.rotation.x - base_rot.x) * eased_progress,
            y = base_rot.y + (kf2.rotation.y - base_rot.y) * eased_progress,
            z = base_rot.z + (kf2.rotation.z - base_rot.z) * eased_progress
        }
    elseif kf1.rotation then
        result_rot = kf1.rotation
    end
    
    return result_pos, result_rot
end

-- Helper function to find current keyframe segment and progress
local function get_keyframe_interpolation(keyframes, current_time, base_pos, base_rot)
    if not keyframes or #keyframes == 0 then
        return base_pos, base_rot
    end
    
    -- Before first keyframe
    if current_time <= keyframes[1].time then
        if keyframes[1].position or keyframes[1].rotation then
            return keyframes[1].position or base_pos, keyframes[1].rotation or base_rot
        end
        return base_pos, base_rot
    end
    
    -- After last keyframe
    if current_time >= keyframes[#keyframes].time then
        local last_kf = keyframes[#keyframes]
        return last_kf.position or base_pos, last_kf.rotation or base_rot
    end
    
    -- Find the two keyframes we're between
    for i = 1, #keyframes - 1 do
        local kf1 = keyframes[i]
        local kf2 = keyframes[i + 1]
        
        if current_time >= kf1.time and current_time <= kf2.time then
            local segment_duration = kf2.time - kf1.time
            if segment_duration <= 0 then
                return kf2.position or base_pos, kf2.rotation or base_rot
            end
            
            local segment_progress = (current_time - kf1.time) / segment_duration
            return interpolate_keyframes(kf1, kf2, segment_progress, base_pos, base_rot)
        end
    end
    
    -- Fallback (shouldn't reach here)
    return base_pos, base_rot
end

-- API: Create an animation configuration
function charged_models.create_animation(name, config)
    if not name or type(name) ~= "string" then
        error("Animation name must be a string")
    end
    
    local merged_config = {}
    for key, value in pairs(DEFAULT_ANIMATION) do
        merged_config[key] = value
    end
    
    if config then
        for key, value in pairs(config) do
            merged_config[key] = value
        end
    end
    
    merged_config.name = name
    
    -- Process transform_keyframes if provided
    if merged_config.transform_keyframes and type(merged_config.transform_keyframes) == "table" then
        -- Ensure keyframes are sorted by time
        table.sort(merged_config.transform_keyframes, function(a, b)
            return (a.time or 0) < (b.time or 0)
        end)
        
        -- Validate and fill in defaults for each keyframe
        for i, kf in ipairs(merged_config.transform_keyframes) do
            for key, value in pairs(DEFAULT_TRANSFORM_KEYFRAME) do
                if kf[key] == nil then
                    kf[key] = value
                end
            end
        end
    end
    
    animation_configurations[name] = merged_config
    return merged_config
end

-- API: Get an animation configuration by name
function charged_models.get_animation(name)
    return animation_configurations[name]
end

-- Helper function to merge model config with defaults
local function merge_model_config(config)
    local merged = deep_copy(DEFAULT_MODEL_CONFIG)
    
    if config then
        for key, value in pairs(config) do
            if key == "animations" and type(value) == "table" then
                merged[key] = {}
                for anim_name, anim_config in pairs(value) do
                    if type(anim_config) == "string" then
                        -- Reference to named animation
                        merged[key][anim_name] = animation_configurations[anim_config]
                    else
                        -- Inline animation config
                        merged[key][anim_name] = charged_models.create_animation(anim_name, anim_config)
                    end
                end
            elseif type(value) == "table" and type(merged[key]) == "table" then
                for subkey, subvalue in pairs(value) do
                    merged[key][subkey] = subvalue
                end
            else
                merged[key] = value
            end
        end
    end
    
    return merged
end

-- API: Create a model configuration template (supports multiple models)
function charged_models.create_model_config(name, config)
    if not name or type(name) ~= "string" then
        error("Model configuration name must be a string")
    end
    
    -- Check if this is a multi-model config (array of models) or single model
    local is_multi = false
    if config and type(config) == "table" then
        -- If config has numeric indices, it's a multi-model setup
        if config[1] ~= nil then
            is_multi = true
        end
    end
    
    if is_multi then
        -- Multiple models configuration
        local models = {}
        for i, model_config in ipairs(config) do
            models[i] = merge_model_config(model_config)
        end
        model_configurations[name] = {
            multi_model = true,
            models = models
        }
    else
        -- Single model configuration (backward compatible)
        model_configurations[name] = {
            multi_model = false,
            models = {merge_model_config(config)}
        }
    end
    
    return model_configurations[name]
end

-- API: Get a model configuration by name
function charged_models.get_model_config(name)
    return model_configurations[name]
end

-- API: List all model configurations
function charged_models.list_model_configs()
    local configs = {}
    for name, _ in pairs(model_configurations) do
        table.insert(configs, name)
    end
    return configs
end

-- API: Register an item with a model configuration
function charged_models.register_item_model(item_name, config_name_or_config)
    if not item_name or type(item_name) ~= "string" then
        error("Item name must be a string")
    end
    
    local config
    if type(config_name_or_config) == "string" then
        config = model_configurations[config_name_or_config]
        if not config then
            error("Model configuration '" .. config_name_or_config .. "' not found")
        end
    elseif type(config_name_or_config) == "table" then
        config = charged_models.create_model_config("_inline_" .. item_name, config_name_or_config)
    else
        error("Configuration must be a string (config name) or table (inline config)")
    end
    
    item_model_registry[item_name] = config
end

-- API: Unregister an item from having a model
function charged_models.unregister_item_model(item_name)
    item_model_registry[item_name] = nil
end

-- API: Check if an item has a registered model
function charged_models.has_item_model(item_name)
    return item_model_registry[item_name] ~= nil
end

-- API: Get the model configuration for an item
function charged_models.get_item_model_config(item_name)
    return item_model_registry[item_name]
end

-- API: List all registered items
function charged_models.list_registered_items()
    local items = {}
    for item_name, _ in pairs(item_model_registry) do
        table.insert(items, item_name)
    end
    return items
end

-- Function to calculate position/rotation based on player's look direction
local function get_adjusted_transform(player, model_config, custom_pos, custom_rot)
    local base_pos = custom_pos or model_config.position
    local base_rot = custom_rot or model_config.rotation
    
    if not model_config.look_adjustment.enabled then
        return base_pos, base_rot
    end
    
    local look_vertical = player:get_look_vertical()
    local pitch_factor = -look_vertical / (math.pi / 2)
    
    local vertical_offset = pitch_factor * model_config.look_adjustment.factor
    vertical_offset = math.max(
        model_config.look_adjustment.min_vertical_offset, 
        math.min(model_config.look_adjustment.max_vertical_offset, vertical_offset)
    )
    
    local rotation_adjustment = pitch_factor * model_config.look_adjustment.rotation_factor
    
    local adjusted_position = {
        x = base_pos.x,
        y = base_pos.y + vertical_offset,
        z = base_pos.z
    }
    
    local adjusted_rotation = {
        x = base_rot.x + rotation_adjustment,
        y = base_rot.y,
        z = base_rot.z
    }
    
    return adjusted_position, adjusted_rotation
end

-- Register the dynamic model entity with animation support
minetest.register_entity("charged_models:dynamic_model", {
    initial_properties = {
        visual = "mesh",
        mesh = "hand.gltf",
        textures = {"hand_skin.png"},
        physical = false,
        collide_with_objects = false,
        pointable = false,
        static_save = false,
        visual_size = {x = 10, y = 10, z = 10},
    },
    
    on_activate = function(self, staticdata)
        self.object:set_armor_groups({immortal = 1})
        self.state = "idle"
        self.animation_timer = 0
        self.current_animation = nil
        self.model_index = 1
        self.is_holding = false
        self.hold_trigger = nil
        
        if staticdata and staticdata ~= "" then
            local data = minetest.deserialize(staticdata)
            if data and data.config then
                self.config = data.config
                if data.model_index then
                    self.model_index = data.model_index
                end
                
                self.object:set_properties({
                    mesh = self.config.mesh,
                    textures = self.config.textures,
                    visual_size = self.config.scale,
                    physical = self.config.physical,
                    collide_with_objects = self.config.collide_with_objects,
                    pointable = self.config.pointable,
                    static_save = self.config.static_save
                })
                
                -- Set initial idle animation if exists
                if self.config.animations and self.config.animations.idle then
                    self:play_animation("idle")
                end
            end
        end
    end,
    
    get_staticdata = function(self)
        if self.config then
            return minetest.serialize({
                config = self.config,
                model_index = self.model_index
            })
        end
        return ""
    end,
    
    play_animation = function(self, anim_name, is_hold)
        if not self.config or not self.config.animations then return false end
        
        local anim = self.config.animations[anim_name]
        if not anim then return false end
        
        self.current_animation = anim
        self.animation_timer = 0
        self.state = "animating"
        self.is_holding = is_hold or false
        self.hold_callback_called = false  -- Reset hold callback flag
        
        -- Call on_start callback if exists
        if anim.on_start then
            anim.on_start(self)
        end
        
        -- Set the animation on the object
        self.object:set_animation(
            {x = anim.keyframes.start, y = anim.keyframes.stop},
            anim.speed or 30,
            anim.blend or 0,
            anim.loop or false
        )
        
        return true
    end,
    
    update_animation = function(self, dtime)
        if not self.current_animation then return end
        
        -- Apply loop speed multiplier to delta time for transform keyframes
        local loop_speed = self.current_animation.loop_speed or 1.0
        local adjusted_dtime = dtime * loop_speed
        self.animation_timer = self.animation_timer + adjusted_dtime
        
        local parent = self.object:get_attach()
        if not parent or not parent:is_player() then return end
        
        -- Handle transform keyframes during animation
        if self.current_animation.transform_keyframes and #self.current_animation.transform_keyframes > 0 then
            local keyframes = self.current_animation.transform_keyframes
            local total_duration = keyframes[#keyframes].time
            
            -- Handle hold behavior
            if self.is_holding and self.animation_timer >= total_duration then
                if self.current_animation.hold_behavior == "hold_last_frame" then
                    -- Stay at last frame
                    local last_kf = keyframes[#keyframes]
                    local custom_pos = last_kf.position or self.config.position
                    local custom_rot = last_kf.rotation or self.config.rotation
                    local adjusted_pos, adjusted_rot = get_adjusted_transform(parent, self.config, custom_pos, custom_rot)
                    self.object:set_attach(parent, "", adjusted_pos, adjusted_rot, true)
                    
                    -- Call on_hold callback if exists (only once per hold, not every frame)
                    if self.current_animation.on_hold and not self.hold_callback_called then
                        self.current_animation.on_hold(self)
                        self.hold_callback_called = true
                    end
                    return
                elseif self.current_animation.hold_behavior == "loop_until_release" then
                    -- Loop the animation - reset timer to loop
                    self.animation_timer = self.animation_timer - total_duration
                    
                    -- Call on_hold callback if exists (only once per loop cycle)
                    if self.current_animation.on_hold then
                        self.current_animation.on_hold(self)
                    end
                end
            end
            
            -- Get interpolated position and rotation at current time
            local custom_pos, custom_rot = get_keyframe_interpolation(
                keyframes,
                self.animation_timer,
                self.config.position,
                self.config.rotation
            )
            
            -- Apply look adjustment to interpolated transform
            local adjusted_pos, adjusted_rot = get_adjusted_transform(parent, self.config, custom_pos, custom_rot)
            self.object:set_attach(parent, "", adjusted_pos, adjusted_rot, true)
            
            -- Check if animation is complete (and not holding)
            if not self.is_holding and not self.current_animation.loop and self.animation_timer >= total_duration then
                self:on_animation_complete()
            elseif not self.is_holding and self.current_animation.loop and self.animation_timer >= total_duration then
                -- Loop the animation
                self.animation_timer = self.animation_timer - total_duration
            end
        else
            -- For animations without transform keyframes, still apply look adjustment to base position
            local adjusted_pos, adjusted_rot = get_adjusted_transform(parent, self.config)
            self.object:set_attach(parent, "", adjusted_pos, adjusted_rot, true)
            
            -- Estimate completion
            local total_frames = self.current_animation.keyframes.stop - self.current_animation.keyframes.start
            local anim_duration = math.abs(total_frames) / (self.current_animation.speed or 30)
            
            if not self.is_holding and not self.current_animation.loop and self.animation_timer >= anim_duration then
                self:on_animation_complete()
            end
        end
    end,
    
    on_animation_complete = function(self)
        if not self.current_animation then return end
        
        -- Call on_complete callback if exists
        if self.current_animation.on_complete then
            self.current_animation.on_complete(self)
        end
        
        self.current_animation = nil
        self.state = "idle"
        self.animation_timer = 0
        self.is_holding = false
        self.hold_callback_called = false  -- Reset hold callback flag
        
        -- Return to idle animation or default position
        local parent = self.object:get_attach()
        if parent then
            if self.config.animations and self.config.animations.idle then
                self:play_animation("idle")
            else
                local adjusted_pos, adjusted_rot = get_adjusted_transform(parent, self.config)
                self.object:set_attach(parent, "", adjusted_pos, adjusted_rot, true)
            end
        end
    end,
    
    on_release = function(self)
        if not self.current_animation then return end
        
        -- Call on_release callback if exists
        if self.current_animation.on_release then
            self.current_animation.on_release(self)
        end
        
        self.is_holding = false
        self.hold_callback_called = false  -- Reset hold callback flag
        
        -- If hold behavior was active, complete the animation
        if self.current_animation.hold_behavior ~= "none" then
            self:on_animation_complete()
        end
    end,
    
    on_step = function(self, dtime)
        local parent = self.object:get_attach()
        if not parent then
            self.object:remove()
            return
        end
        
        if not parent:is_player() then
            self.object:remove()
            return
        end
        
        -- Update animation if playing
        if self.state == "animating" then
            self:update_animation(dtime)
        elseif self.state == "idle" and self.config then
            -- Update position based on look direction when idle
            local adjusted_pos, adjusted_rot = get_adjusted_transform(parent, self.config)
            self.object:set_attach(parent, "", adjusted_pos, adjusted_rot, true)
        end
    end,
})

-- Function to attach models to player based on item configuration
local function attach_models_to_player(player, item_name)
    local player_name = player:get_player_name()
    local item_config = item_model_registry[item_name]
    
    if not item_config then
        return false
    end
    
    -- Remove existing models if any
    if player_models[player_name] then
        for _, model in ipairs(player_models[player_name]) do
            if model and model:get_pos() then
                model:remove()
            end
        end
        player_models[player_name] = nil
    end
    
    -- Initialize model storage for this player
    player_models[player_name] = {}
    
    local pos = player:get_pos()
    
    -- Attach all models defined in the configuration
    for i, model_config in ipairs(item_config.models) do
        local model = minetest.add_entity(
            pos, 
            "charged_models:dynamic_model", 
            minetest.serialize({
                config = model_config,
                model_index = i
            })
        )
        
        if model then
            local entity = model:get_luaentity()
            if entity then
                entity.config = model_config
                entity.model_index = i
            end
            
            model:set_properties({
                mesh = model_config.mesh,
                textures = model_config.textures,
                visual_size = model_config.scale,
                physical = model_config.physical,
                collide_with_objects = model_config.collide_with_objects,
                pointable = model_config.pointable,
                static_save = model_config.static_save
            })
            
            local adjusted_pos, adjusted_rot = get_adjusted_transform(player, model_config)
            model:set_attach(player, "", adjusted_pos, adjusted_rot, true)
            
            table.insert(player_models[player_name], model)
        end
    end
    
    return #player_models[player_name] > 0
end

-- Function to remove models from player
local function remove_models_from_player(player)
    local player_name = player:get_player_name()
    
    if player_models[player_name] then
        for _, model in ipairs(player_models[player_name]) do
            if model and model:get_pos() then
                model:remove()
            end
        end
        player_models[player_name] = nil
    end
end

-- Function to trigger animation based on input event type
local function trigger_animation(player, event_type, key)
    local player_name = player:get_player_name()
    local models = player_models[player_name]
    
    if not models then return end
    
    local trigger_string = event_type .. "_" .. key
    
    -- Trigger animation on all models that have a matching trigger
    for _, model in ipairs(models) do
        if model and model:get_pos() then
            local entity = model:get_luaentity()
            if entity and entity.config and entity.config.animations then
                -- Find animation with matching trigger
                for anim_name, anim in pairs(entity.config.animations) do
                    if anim.trigger == trigger_string then
                        local is_hold = event_type == "hold"
                        
                        -- For hold events, only start if not already holding this key
                        if is_hold and entity.is_holding and entity.hold_trigger == key then
                            -- Already holding this key, don't restart animation
                            return
                        end
                        
                        entity:play_animation(anim_name, is_hold)
                        
                        if is_hold then
                            entity.hold_trigger = key
                        end
                    end
                end
            end
        end
    end
end

-- Function to handle key release for held animations
local function handle_key_release(player, key)
    local player_name = player:get_player_name()
    local models = player_models[player_name]
    
    if not models then return end
    
    -- Check if any model was holding this key
    for _, model in ipairs(models) do
        if model and model:get_pos() then
            local entity = model:get_luaentity()
            if entity and entity.is_holding and entity.hold_trigger == key then
                entity:on_release()
            end
        end
    end
    
    -- Also trigger release animations
    trigger_animation(player, "release", key)
end

-- Initialize player input state
local function init_player_input_state(player_name)
    player_input_states[player_name] = {}
    for _, key in ipairs(AVAILABLE_KEYS) do
        player_input_states[player_name][key] = false
    end
end

-- Global step to check wielded item and handle input
minetest.register_globalstep(function(dtime)
    for _, player in ipairs(minetest.get_connected_players()) do
        local wielded = player:get_wielded_item()
        local player_name = player:get_player_name()
        local item_name = wielded:get_name()
        
        -- Initialize input state tracking
        if not player_input_states[player_name] then
            init_player_input_state(player_name)
        end
        
        -- Check if player is holding a registered item
        if item_model_registry[item_name] then
            local current_models = player_models[player_name]
            local needs_new_models = false
            
            if not current_models or #current_models == 0 then
                needs_new_models = true
            else
                -- Verify all models are still valid and match the current item
                local item_config = item_model_registry[item_name]
                if #current_models ~= #item_config.models then
                    needs_new_models = true
                else
                    for i, model in ipairs(current_models) do
                        if not model or not model:get_pos() then
                            needs_new_models = true
                            break
                        end
                        local entity = model:get_luaentity()
                        if not entity or not entity.config or entity.config ~= item_config.models[i] then
                            needs_new_models = true
                            break
                        end
                    end
                end
            end
            
            if needs_new_models then
                attach_models_to_player(player, item_name)
            end
            
            -- Check for input changes to trigger animations
            local controls = player:get_player_control()
            local last_state = player_input_states[player_name]
            
            -- Process all available keys
            for _, key in ipairs(AVAILABLE_KEYS) do
                local current_state = controls[key]
                local last_key_state = last_state[key]
                
                if current_state and not last_key_state then
                    -- Key pressed
                    trigger_animation(player, "press", key)
                    -- Also trigger hold immediately
                    trigger_animation(player, "hold", key)
                elseif not current_state and last_key_state then
                    -- Key released
                    handle_key_release(player, key)
                end
                -- Remove the continuous hold trigger - it was causing animation restarts
                
                last_state[key] = current_state
            end
        else
            if player_models[player_name] then
                remove_models_from_player(player)
            end
        end
    end
end)

-- Clean up when player leaves
minetest.register_on_leaveplayer(function(player)
    local player_name = player:get_player_name()
    remove_models_from_player(player)
    last_input_state[player_name] = nil
end)

minetest.log("action", "[Charged Models] Multi-keyframe sequential animation system loaded.")

local modpath = minetest.get_modpath(minetest.get_current_modname())
dofile(modpath .. "/cube-entities.lua")
--dofile(modpath .. "/examples.lua")
dofile(modpath .. "/mtg_wooden_pickaxe_example.lua")
dofile(modpath .. "/dual_swords_example.lua")
dofile(modpath .. "/charged_greatsword_overhead_particles_example.lua")
dofile(modpath .. "/two_handed_fist_example.lua")
dofile(modpath .. "/charged_hand.lua")
dofile(modpath .. "/torch_example.lua")
