-- SPDX-FileCopyrightText: 2024 Tuxilio <mail@tuxil.io>
--
-- SPDX-License-Identifier: GPL-3.0-or-later

local modname = core.get_current_modname()
local modpath = core.get_modpath(modname)

mqtt = {}
mqtt.isConnected = {}

-- Handle mod security if needed
local ie, req_ie = _G, core.request_insecure_environment
if req_ie then ie = req_ie() end
if not ie then
    error("The MQTT mod requires access to insecure functions in order "..
        "to work.  Please add the mqtt mod to your secure.trusted_mods "..
        "setting or disable the mqtt mod.")
end

ie.package.path =
    -- To find LuaMQTT's init.lua
    "/usr/local/share/lua/5.1/?/init.lua;"
    -- For LuaMQTT to find its files
    .."/usr/local/share/lua/5.1/?/?.lua;"
    -- For local installations
    .."~/.luarocks/share/lua/5.1/?.lua;"
    .."~/.luarocks/share/lua/5.1/?/init.lua"
    ..ie.package.path

-- The build of Lua that Minetest comes with only looks for libraries under
-- /usr/local/share and /usr/local/lib but LuaSocket is often installed under
-- /usr/share and /usr/lib.
if not rawget(_G, "jit") and package.config:sub(1, 1) == "/" then
    ie.package.path = ie.package.path..
            ";/usr/share/lua/5.1/?.lua"..
            ";/usr/share/lua/5.1/?/init.lua"

    ie.package.cpath = ie.package.cpath..
            ";/usr/lib/lua/5.1/?.so"..
            ";/usr/lib64/lua/5.1/?.so"

    ie.package.cpath = "/usr/lib/x86_64-linux-gnu/lua/5.1/?.so;"..ie.package.cpath
end

-- Temporarily set require so that LuaMQTT can access it
local old_require = require
require = ie.require

mqtt.lib = ie.require("mqtt")
mqtt.ioloop = ie.require("mqtt.ioloop")

-- Reset require for mod security
-- TODO: This is broken, so **this should be fixed**.
-- require = old_require

--- Function to create and connect an MQTT client
-- Parameters:
--     uri: The URI of the MQTT broker to connect to
--     client_id: A unique identifier for the MQTT client
--     username: The username for authentication (optional)
--     password: The password for authentication (optional)
-- Returns:
--     client: The connected MQTT client instance
--     If the connection fails, it may return nil or an error message (depending on the implementation of mqtt.client)
function mqtt.connect(uri, clientId, username, password)
    local client = mqtt.lib.client{
        uri = uri,
        id = clientId,
        username = username,
        password = password,
        clean = true,
    }

    core.log("action", "[mqtt] created MQTT client")

    client:on{
        connect = function(connack)
            if connack.rc ~= 0 then
                core.log("error", "[mqtt] connection to MQTT broker failed: " .. connack:reason_string() .. ", " .. connack)
            else
                core.log("action", "[mqtt] connected to MQTT broker: " .. uri)
                mqtt.isConnected[client] = true
                mqtt.ioloop.get():remove(client)
            end
        end,
        error = function(err)
            core.log("error", "[mqtt] connection error: " .. err)
        end,
    }

    core.log("action", "[mqtt] attempting to connect to MQTT broker: " .. uri)

    local ioloop = mqtt.ioloop.get()
    ioloop:add(client)
    ioloop:run_until_clients()

    return client
end

--- Function to publish a message to an MQTT broker
-- Parameters:
--     client: The MQTT client instance
--     topic: The topic to publish the message to
--     payload: The message payload to be published
--     options: A table containing optional parameters
--         qos: The QoS level for message publication (default is 1)
--         retain: A boolean flag indicating whether to retain the message (default is false)
--         dup: A boolean flag indicating whether the message is a duplicate (default is false)
--         properties: A table for additional properties for publishing the message (optional)
--         user_properties: A table for user-defined properties for publishing the message (optional)
--         callback: A function to call when the published message is acknowledged (optional)
-- Returns:
--     success: true or packet id on success, or false and error message on failure
function mqtt.publish(client, topic, payload, options)
    if not options then options = {} end
    local qos = options.qos or 1
    local retain = options.retain or false
    local dup = options.dup or false
    local properties = options.properties or {}
    local user_properties = options.user_properties or {}
    local callback = options.callback or nil

    local args = {
        topic = topic,
        payload = payload,
        qos = qos,
        retain = retain,
        dup = dup,
        properties = properties,
        user_properties = user_properties,
        callback = callback,
    }

    local success, packet_id_or_error = client:publish(args)

    if callback then
        callback(success, packet_id_or_error)
    end

    return success, packet_id_or_error
end
