local floor = math.floor
local ceil = math.ceil
local pow = math.pow

-- Corresponds to wallmounted
NORTH = minetest.dir_to_wallmounted({x=0, y=0, z=1}, true)
EAST = minetest.dir_to_wallmounted({x=1, y=0, z=0}, true)
SOUTH = minetest.dir_to_wallmounted({x=0, y=0, z=-1}, true)
WEST = minetest.dir_to_wallmounted({x=-1, y=0, z=0}, true)
--UP = minetest.dir_to_wallmounted({x=0, y=1, z=0}, true)
--DOWN = minetest.dir_to_wallmounted({x=0, y=-1, z=0}, true)

rooms = {}
portals = {}

function swap_remove(the_table, pos)
	if pos == #the_table then
		return table.remove(the_table)
	end
	local temp = table.remove(the_table)
	temp, the_table[pos] = the_table[pos], temp
	return temp
end

setmetatable(portals, {
	__index = function(portal_table, def)
		local def_str = string.format("%s,%s,%s", def.top, def.sides, def.bot)
		local portal = rawget(portal_table, def_str)
		if portal then return portal end
		portal = Portal.new()
		rawset(portal_table, def_str, portal)
		return portal
	end,
	__newindex = function(portal_table, def, value)
		assert(false)
	end,
})

Portal = {}
Portal.__index = Portal

function Portal.new(def)
	local portal = {}
	setmetatable(portal, Portal)
 	return portal
end

-- Returns true if you opened a portal (maybe? I forget)
function Portal:disconnect(room, room_direction)
	for i, v in pairs(self) do
		if v.index == room.index and v.direction == room_direction then
			swap_remove(self, i)
			break
		end
	end
	room.neighbors[room_direction] = nil
	local result
	if #self == 2 then
		local a_index = self[1].index
		local b_index = self[2].index
		if a_index ~= b_index or true then
			local a_direction = self[1].direction
			local b_direction = self[2].direction
			local a = rooms[a_index]
			local b = rooms[b_index]
			a.neighbors[a_direction] =
				{ portal = self, index = b_index, rotation = b_direction }
			a:open_portal(a_direction)
			b.neighbors[b_direction] =
				{ portal = self, index = a_index, rotation = a_direction }
			b:open_portal(b_direction)
			result = other
		end
	elseif #self == 1 then
		local other_index = self[1].index
		local other_direction = self[1].direction
		local other = rooms[other_index]
		other:close_portal(other_direction)
		other:set_portal_barrier(other_direction, "rooms:barrier")
	end
	rooms:save()
	return result
end

-- Returns true if you opened a portal
function Portal:connect(room, room_direction)
	assert(room.index)
	assert(room_direction)
	table.insert(self, {index = room.index, direction = room_direction})
	local result
	room.neighbors[room_direction] = { portal = self }
	if #self == 2 then
		local other_index = self[1].index
		if room.index ~= other_index or true then
			local other_direction = self[1].direction
			local other = rooms[other_index]
			room.neighbors[room_direction] =
				{ portal = self, index = other.index, rotation = other_direction }
			room:open_portal(room_direction)
			other.neighbors[other_direction] =
				{ portal = self, index = room.index, rotation = room_direction }
			other:open_portal(other_direction)
			result = other
		end
	elseif #self == 3 then
		local room1 = rooms[self[1].index]
		room1:close_portal(self[1].direction)
		room1:set_portal_barrier(self[1].direction, "rooms:barrier")
		local room2 = rooms[self[2].index]
		room2:close_portal(self[2].direction)
		room2:set_portal_barrier(self[2].direction, "rooms:barrier")
	end
	rooms:save()
	return result
end

function room_pos(world_pos)
	world_pos = vector.divide(world_pos, 12)
	return {
		x=floor(world_pos.x) * 12,
		y=floor(world_pos.y) * 12,
		z=floor(world_pos.z) * 12,
	}
end

function index_to_pos(index)
	local x = 0
	local y = 0
	local z = 0
	local i = 1
	while index > 0 do
		if index % 2 == 1 then x = x + i end
		if index % 4 >= 2 then y = y + i end
		if index % 8 >= 4 then z = z + i end
		index = floor(index / 8)
		i = i * 2
	end
	return skew {x=x, y=y, z=z}
end
function pos_to_index(room_pos)
	local pos = unskew(room_pos)
	local index = 0
	local i = 1
	while pos.x >= i or pos.y >= i or pos.z >= i do
		if pos.x % (i * 2) >= i then index = index + i * i * i end
		if pos.y % (i * 2) >= i then index = index + i * i * i * 2 end
		if pos.z % (i * 2) >= i then index = index + i * i * i * 4 end
		i = i * 2
	end
	return index
end
function skew(p)
	return {
		x = (p.x + 3*p.y - p.z) * 12,
		y = p.x * 12,
		z = (p.x + p.y + 2*p.z) * 12,
	}
end
function unskew(p)
	return {
		x = p.y / 12,
		y = (2*p.x - 3*p.y + p.z) / 7 / 12,
		z = (-p.x - 2*p.y + 3*p.z) / 7 / 12,
	}
end

Room = {}
Room.__index = Room

function Room.allocate_new()
	local room = Room.new(#rooms + 1)
 	table.insert(rooms, room)
 	rooms:save()
 	return room
end

function Room.new(i)
	local room = { neighbors = {} }
	setmetatable(room, Room)
 	room.index = i
 	room.pos = index_to_pos(i)
 	return room
end

function Room:centroid() return vector.add(self.pos, 6) end

function Room:get_portal_blocks(direction)
	local result = {
		top = {},
		bot = {},
		sides = {},
	}
	for i = 4,7 do
		if direction == NORTH then
			table.insert(result.top, vector.add(self.pos, {x=i, y=8, z=11}))
			table.insert(result.sides, vector.add(self.pos, {x=3, y=i, z=11}))
			table.insert(result.sides, vector.add(self.pos, {x=8, y=i, z=11}))
			table.insert(result.bot, vector.add(self.pos, {x=i, y=3, z=11}))
		elseif direction == EAST then
			table.insert(result.top, vector.add(self.pos, {x=11, y=8, z=i}))
			table.insert(result.sides, vector.add(self.pos, {x=11, y=i, z=3}))
			table.insert(result.sides, vector.add(self.pos, {x=11, y=i, z=8}))
			table.insert(result.bot, vector.add(self.pos, {x=11, y=3, z=i}))
		elseif direction == SOUTH then
			table.insert(result.top, vector.add(self.pos, {x=i, y=8, z=0}))
			table.insert(result.sides, vector.add(self.pos, {x=3, y=i, z=0}))
			table.insert(result.sides, vector.add(self.pos, {x=8, y=i, z=0}))
			table.insert(result.bot, vector.add(self.pos, {x=i, y=3, z=0}))
		elseif direction == WEST then
			table.insert(result.top, vector.add(self.pos, {x=0, y=8, z=i}))
			table.insert(result.sides, vector.add(self.pos, {x=0, y=i, z=3}))
			table.insert(result.sides, vector.add(self.pos, {x=0, y=i, z=8}))
			table.insert(result.bot, vector.add(self.pos, {x=0, y=3, z=i}))
		end
	end
	return result
end

function Room:portal_frame_dir(world_pos)
	local offset = vector.subtract(world_pos, self.pos)
	if offset.x == 0 then
		return WEST
	elseif offset.x == 11 then
		return EAST
	elseif offset.z == 0 then
		return SOUTH
	elseif offset.z == 11 then
		return NORTH
	end
	return nil
end

function Room:portal_modified(direction)
	if not minetest.get_node_or_nil(self.pos) then
		minetest.emerge_area(
			self.pos,
			vector.add(self.pos, 12),
			function (blockpos, action, calls_remaining, param)
				if calls_remaining ~= 0 then return end
				self:portal_modified(direction)
			end
		)
		return
	end
	local neighbor = self.neighbors[direction]
	if neighbor and neighbor.portal then
		neighbor.portal:disconnect(self, direction)
	end
	local blocks = self:get_portal_blocks(direction)
	local def = calc_portal_def(blocks)
	if def then
		local is_valid = true
		for _, name in pairs(def) do
			if name == "rooms:frame" then
				is_valid = false
				break
			end
		end
		if is_valid then
			local other = portals[def]:connect(self, direction)
			if not other then
				self:set_portal_barrier(direction, "rooms:barrier")
			end
		else
			local neighbor = self.neighbors[direction]
			if neighbor and neighbor.portal then
				neighbor.portal:disconnect(self, direction)
			end
			self:close_portal(direction)
			self:set_portal_barrier(direction, "rooms:barrier_pink")
		end
	else
		local neighbor = self.neighbors[direction]
		if neighbor and neighbor.portal then
			neighbor.portal:disconnect(self, direction)
		end
		self:close_portal(direction)
		self:set_portal_barrier(direction, "rooms:barrier_red")
	end
	rooms:save()
end

function Room:fill_frames(direction)
	local blocks = self:get_portal_blocks(direction)
	for _, side in pairs(blocks) do
		for _, pos in ipairs(side) do
			if minetest.get_node(pos).name == "air" then
				-- Set then place so that the grass's
				-- on_place callback sees the frame and not air
				-- TODO is this a bug with minetest?
				minetest.set_node(pos, {name = "rooms:frame"})
				minetest.place_node(pos, {name = "rooms:frame"})
			end
		end
	end
end

--- returns a `function(x,y,z) -> {x=..., y=..., z=...}` that maps portal rotations
function get_rotation_map(dir, other_dir)
	local function flip(p) return {x=11-p.x, y=p.y, z=11-p.z} end
	local function right(p) return {x=p.z, y=p.y, z=11-p.x} end
	local function left(p) return {x=11-p.z, y=p.y, z=p.x} end
	local function pass(p) return {x=p.x, y=p.y, z=p.z} end
	return ({
		[NORTH]={[NORTH]=flip, [EAST]=left, [SOUTH]=pass, [WEST]=right},
		[EAST] ={[NORTH]=right, [EAST]=flip, [SOUTH]=left, [WEST]=pass},
		[SOUTH]={[NORTH]=pass, [EAST]=right, [SOUTH]=flip, [WEST]=left},
		[WEST] ={[NORTH]=left, [EAST]=pass, [SOUTH]=right, [WEST]=flip},
	})[dir][other_dir]
end

-- TODO the current order is portal:connect -> room:open_portal, room:close_portal -> portal:disconnect
-- this feels unintuitive
-- The portal should no longer be accessible to teleports
function Room:close_portal(dir)
	local neighbor = self.neighbors[dir]
	if neighbor and neighbor.index then
		neighbor.index = nil
		neighbor.rotation = nil
	end
end

-- Draw mirror of other on the other portal
function Room:open_portal(dir)
	local other = rooms[self.neighbors[dir].index]
	local other_dir = self.neighbors[dir].rotation
	local offset = vector.add(self.pos,
		vector.multiply(minetest.wallmounted_to_dir(dir), 12))
	local map = get_rotation_map(dir, other_dir)
	for x = 1, 10 do
		for y = 1, 10 do
			for z = 1, 10 do
				local node = minetest.get_node(vector.add(other.pos, map{x=x, y=y, z=z}))
				local world_pos = vector.add(offset, {x=x, y=y, z=z})
				minetest.set_node(world_pos, node)
			end
		end
	end
	for _, wall_dir in ipairs{NORTH, EAST, SOUTH, WEST} do
		for i = 3,8 do
			for j = 3,8 do
				local node_name
				if wall_dir == dir then
					node_name = "rooms:air"
				else
					node_name = "rooms:barrier_gray"
				end
				local from_pos
				local to_pos
				if (i == 3 or i == 8) and (j == 3 or j == 8) then
				elseif wall_dir == SOUTH then
					from_pos = vector.add(other.pos, map{x=i, y=j, z=11})
					to_pos = vector.add(offset, {x=i, y=j, z=11})
				elseif wall_dir == WEST then
					from_pos = vector.add(other.pos, map{x=11, y=j, z=i})
					to_pos = vector.add(offset, {x=11, y=j, z=i})
				elseif wall_dir == NORTH then
					from_pos = vector.add(other.pos, map{x=i, y=j, z=0})
					to_pos = vector.add(offset, {x=i, y=j, z=0})
				elseif wall_dir == EAST then
					from_pos = vector.add(other.pos, map{x=0, y=j, z=i})
					to_pos = vector.add(offset, {x=0, y=j, z=i})
				end
				if from_pos then
					local node = minetest.get_node(from_pos)
					if node.name == "rooms:air" then
						node = { name = node_name }
					end
					minetest.swap_node(to_pos, node)
				end
			end
		end
	end
end

function Room:set_portal_barrier(direction, node_name)
	local nodes = {}
	for i = 3,8 do
		for j = 3,8 do
			if (i == 3 or i == 8) and (j == 3 or j == 8) then
			elseif direction == NORTH then
				table.insert(nodes, vector.add(self.pos, {x=i, y=j, z=12}))
			elseif direction == EAST then
				table.insert(nodes, vector.add(self.pos, {x=12, y=j, z=i}))
			elseif direction == SOUTH then
				table.insert(nodes, vector.add(self.pos, {x=i, y=j, z=-1}))
			elseif direction == WEST then
				table.insert(nodes, vector.add(self.pos, {x=-1, y=j, z=i}))
			end
		end
	end
	minetest.bulk_set_node(nodes, {name=node_name})
end

function Room:update_mirrors(world_pos)
	for dir, neighbor in pairs(self.neighbors) do
		if neighbor.rotation then
			local map = get_rotation_map(dir, neighbor.rotation)
			local other = rooms[neighbor.index]
			local offset = vector.subtract(world_pos, self.pos)

			local node = minetest.get_node(world_pos)
			local other_pos = vector.add(vector.add(map(offset), other.pos),
				vector.multiply(minetest.wallmounted_to_dir(neighbor.rotation), 12))
			minetest.set_node(other_pos, node) -- place the node
		end
	end
end

function calc_portal_def(block_set)
	local result = {}
	for i, side in pairs(block_set) do
		local match = true
		for _, pos in ipairs(side) do
			local name = minetest.get_node(pos).name
			if not result[i] then
				result[i] = name
			elseif result[i] ~= name then
				return nil
			end
		end
	end
	return result
end

-- Load files
local modpath = minetest.get_modpath(minetest.get_current_modname())
dofile(modpath .. "/nodes.lua")
dofile(modpath .. "/save.lua")
dofile(modpath .. "/worldgen.lua")
dofile(modpath .. "/player.lua")
