# Using arclib

This is a guide for downstream developers intending to use arclib to create game-specific integration mods.

## Quick Start

### Installation

Add `arclib` to your mod's dependencies in `mod.conf`:

```toml
depends = arclib
```

### Basic Usage

```lua
-- Create a manager instance
local manager = arclib.manager({
	url_target = "wss://archipelago.gg:38281",
	login_slotname = "PlayerName",
	login_gamename = "Your Game Name",
	modstore = core.get_mod_storage()  -- For persistence
})

core.register_on_shutdown(function()
		manager:close()
	end)

-- Standard configuration chat command
arclib.ux.configure_command({
		modstore = modstore,
		manager = manager
	})

-- Standard chat bridging
arclib.ux.chat_bridge(manager)
```

## API Reference

### State object

Both `arclib.manager` and `arclib.connection` expose a `state` table.

- The connection's `state` is the live, authoritative view of data received from the Archipelago server, maintained and updated internally by `arclib.connection`.
- The manager's `state` is a cached snapshot table that is updated on every connection `update` event by copying keys from the underlying connection's `state`. This snapshot persists across disconnects so your mod can continue to inspect the latest known state even when the socket is down.

Downstream consumers should treat both of these as read-only and use the manager's methods/events to send messages or change server-side state.

#### Contents and derived fields

The connection builds up `state` incrementally from server commands. The exact structure is defined by the Archipelago protocol, but the library adds some convenience/derived fields on top. You do not need to depend on every field; focus on the parts your game logic actually uses.

Common fields include:

- `room_info` – Last `RoomInfo` command payload, expanded into top-level keys.
- `data_package` – Full data package (`games`, etc.), with additional lookup tables per game:
  - `item_id_to_name` – Inverse of `item_name_to_id`.
  - `location_id_to_name` – Inverse of `location_name_to_id`.
- `connected` / `room_update` – Latest connection/session metadata from `Connected` / `RoomUpdate` commands, expanded into top-level keys.
- `items_received` – Ordered array of all items received so far; kept in sync across incremental `ReceivedItems` updates, with resync on desync detection.
- `checked_locations` / `missing_locations` – Arrays of known checked and missing location IDs.
- `checked_location_index` – Map `location_id -> status` built from `checked_locations` and `missing_locations`:
  - `true` – location is known checked
  - `false` – location is known missing
  - `nil` – location is currently unknown
- `hintdata` / `hints` – Aggregated hint data built from `Retrieved` and `SetReply` notifications. `hintdata` keeps the raw key→list mapping; `hints` is a flat array convenient for scanning.
- `location_info` – Mapping from location ID to latest `LocationInfo` scout data for that location, when available.
- `players_index` – Mapping from slot ID to player data (built from `players`).
- Other protocol-provided fields such as `slot`, `slot_info`, `players`, etc. that are passed through from the server without additional transformation.

The manager simply mirrors and accumulates these fields from the underlying connection, making them available as a long-lived snapshot via `manager.state`.

#### Name and ID lookup helpers

Rather than walking the `state` structure manually, prefer the lookup helpers in `arclib.util` when you need to translate between IDs and human-readable names:

- `arclib.util.get_slot_game(state, slot_id)` – Return the game name for a slot ID.
- `arclib.util.get_location_name(state, game, location_id)` – Look up a location name.
- `arclib.util.get_item_name(state, game, item_id)` – Look up an item name.
- `arclib.util.get_player_name(state, player_id)` – Look up a player/slot name.

These helpers assume that `state.data_package`, `state.slot_info`, and related structures have already been populated by the server; call them from within or after `update` handlers.

### Manager (`arclib.manager`)

The main interface for Archipelago connections.

Handles automatic reconnects and message resends, including persisting queued reliable messages across restarts. Maintains a cached replicated state from the server, including across disconnects, and provides helpers for common Archipelago workflows.

#### Constructor
```lua
local manager = arclib.manager(options)
```

##### Options

- `modstore`: Luanti mod storage for persistence (recommended)
- `storeprefix`: Optional prefix for storage keys
- `url_target`: Archipelago WebSocket URL
- `url_proxy`: HTTP proxy URL (default: `http://localhost:9839`)
- `login_slotname`: Player slot name
- `login_gamename`: Game name
- `login_password`: Optional room password

#### Methods

- `manager:open()` – Start connection.
- `manager:close()` – Stop connection.
- `manager:check_location(locations)` – Check location(s) by name or ID (reliable, persisted).
- `manager:goal_complete()` – Send goal completion (reliable, persisted).
- `manager:say(text)` – Send chat message.
- `manager:flush_send_queue()` – Manually flush pending messages.
- `manager:location_info(location_id)` – Return combined information about a location and its item, using both scouting and hint data (see below).

#### Events

- `update` – State updated from server (any data update received, including partial state during startup).
- `printtext` – Chat message received (from PrintJSON) and parsed.
- `disconnect` – Connection lost.
- `rejected` – Connection rejected by server.
- `cmd:*` – Specific Archipelago commands, following Archipelago protocol's TitleCase naming.

#### Multiplayer patterns

- The `arclib.manager` API can technically support multiple managers in one world (for example, one per Archipelago slot), but this pattern is not yet used in any known public mods and requires careful design on your side.
- The most straightforward approach is to treat the world itself as the "player": run a single manager for the world, connect it to one Archipelago slot, and have all in-world players contribute to the same pool of location checks and objectives.
- Luanti encourages multiplayer-capable games by default, but some games intentionally restrict themselves to singleplayer; arclib does not enforce either choice.
- It is up to your game-specific mod to decide how multiplayer should behave (shared world slot vs per-player slots, how to present progress, etc.). arclib only provides the connection and state; it does not prescribe multiplayer policy.

#### Location scouting and hints

`manager:location_info(location_id)` returns a table that merges the best available information from `LocationScouts` and team hints. This is intended as the primary entry point for game logic that needs to answer questions like "what item is here?" or "who receives this item?" without having to manually traverse protocol fields.

The returned table always includes:

- `location_id` – Numeric location ID.
- `location_name` – Human-readable name for this location in your game.

If data is available, it may also include:

- `item_id` / `item_name` – The item placed at this location.
- `receiving_player_id` / `receiving_player_name` – The slot that receives the item.
- `receiving_game` – Game name for the receiving slot.
- `hint_status` – A string such as `"priority"`, `"avoid"`, `"found"`, or `"no_priority"` when a hint applies; `nil` when there is no hint.
- `item_flags` – A table of booleans derived from scout flags (if scouted): `progression`, `useful`, `trap`.

Scouted data, when present, takes precedence over hints. For how to turn `location_info` into visible flags and what the defaults look like, see “Flags and spoiler policy” below.

#### Flags and spoiler policy

`manager:location_info(location_id)` exposes flag-style information about a location (for example `item_flags.progression/useful/trap` and any hint status). arclib does not decide which of these flags should be shown to the player; that is entirely up to your game-specific mod and its spoiler policy.

To help with consistent behavior, `arclib.itemvis.standard_flags(location_info)` implements a recommended baseline mapping from a `location_info` table to a flag set suitable for visualization:

- It takes the table returned by `manager:location_info(location_id)`.
- It returns a flag set — a table where keys are flag names (such as `progression`, `useful`, `trap`, `priority`, `avoid`) and values are ignored; only the presence of a key matters.

You are encouraged to start from `itemvis.standard_flags(location_info)` as a default and adjust if needed for your game (for example, to hide traps, or to suppress progression flags until certain conditions are met).

### Connection (`arclib.connection`)

Low-level protocol handler (used internally by manager).  Not intended for downstream use, exported as-is.

### WebSocket (`arclib.websocket`)

HTTP-to-WebSocket bridge client.  Not intended for downstream use, exported as-is.

### Utilities (`arclib.util`)

- `create_emitter()` - Create event emitter functions
- `deepcopy(src)` - Deep copy tables with cycle support
- `generate_uuid()` - Generate UUID v4
- `log(level, ...)` - Structured logging
- `print_json_convert(state, data)` - Convert PrintJSON to text
- `structstore(modstore, prefix)` - Persistent structured storage

### UX (`arclib.ux`)

- `configure(options)` - Load stored configuration / show configuration UI

### Item Visualization (`arclib.itemvis`)

Standard pseudo-3D visualization for Archipelago items, creating ephemeral visual models that aren't stored in the map.

- `arclib.itemvis.standard_flags(location_info)` – Helper that converts a `location_info` table into a standard flag set; see “Flags and spoiler policy” for details.

#### Constructor
```lua
local vis = arclib.itemvis.add(options)
```

##### Options
- `pos` (required): Center position for the visual
- `check` (optional): Function returning truthy to keep visual, auto-destroy if falsy
- `applied_to_node` (optional): Node name this visual is attached to
- `get_flags` (optional): Function returning a set of flags to display for this instance. It should return a table where keys are flag names (for example `progression`, `useful`, `trap`, `priority`, `avoid`) and values are ignored; only the presence of a key matters. See “Flags and spoiler policy” for how to derive this.

#### Methods
- `vis:remove()` - Destroy the visual model
- `vis:explode()` - Destroy with particle explosion effect
- `vis:move_to(pos)` - Move the visual to a new position

#### Node Integration
- `arclib.itemvis.node_apply(pos, node)` - Apply visualization to a specific node
- `arclib.itemvis.node_register(nodenames)` - Register visualization for node types via ABM/LBM
- `arclib.itemvis.check_all()` - Manually trigger validation check

When using `node_apply` / `node_register`, item visualization can also show per-location flags automatically. If a node definition provides a function `_arclib_get_location_flags(pos, node)`, `itemvis` will call it when creating the visual for that node. This function should return the same kind of flag set that `get_flags` (above) does: a table where keys are flag names and values are ignored. See “Flags and spoiler policy” for how to derive this.

#### Utility Functions
- `arclib.itemvis.inside_radius(pos, radius)` - Find all itemvis instances within radius, returns array

#### Usage Example
```lua
-- Create a floating item visualization at a position
local vis = arclib.itemvis.add({
    pos = {x = 100, y = 10, z = 200},
    check = function()
        -- Keep visual only while condition is true
        return some_condition
    end
})

-- Create a node using itemvis for display.
core.register_node(modname .. ":apitem", {
		drawtype = "airlike",
		paramtype = "light",
		sunlight_propagates = true,
		walkable = false,
		on_construct = arclib.itemvis.node_apply,
		after_destruct = arclib.itemvis.node_apply,
		on_punch = function(pos)
			-- (Replace with your method of accessing manager and getting Location ID)
			api.manager:check_location(core.get_meta(pos):get_float("location_id"))
			for _, v in ipairs(arclib.itemvis.inside_radius(pos, 0.5)) do
				v:explode()
			end
			core.remove_node(pos)
		end
	})
arclib.itemvis.node_register({modname .. ":apitem"})
```

## Architecture

```
Your Mod
	↕
arclib.manager
arclib.connection
arclib.websocket
	↕ (HTTP)
External websockproxy
	↕ (WebSocket)
Archipelago Server
```