-- LUALOCALS < ---------------------------------------------------------
local minetest, pairs, pcall, string, tonumber, vector
    = minetest, pairs, pcall, string, tonumber, vector
local string_format, string_gsub, string_match
    = string.format, string.gsub, string.match
-- LUALOCALS > ---------------------------------------------------------

local include = ...
local exportdb, savedb = include("exportdb")
local exportall = include("exportall")

local modname = minetest.get_current_modname()

local function save_export_report()
	savedb()
	local defs, media = exportall()
	return true, string_format("exported %d defs and %d media", defs, media)
end

minetest.register_chatcommand(modname, {
		description = "Rip all items in already marked for export",
		privs = {server = true},
		func = save_export_report
	})

minetest.register_chatcommand(modname .. "_clear", {
		description = "Clear all saved exports",
		privs = {server = true},
		func = function()
			for k in pairs(exportdb) do exportdb[k] = nil end
			return save_export_report()
		end
	})

minetest.register_chatcommand(modname .. "_inv", {
		description = "Rip all items in inventory",
		privs = {server = true},
		func = function(name)
			local player = minetest.get_player_by_name(name)
			if not player then return false, "invalid player" end
			for _, list in pairs(player:get_inventory():get_lists()) do
				for _, stack in pairs(list) do
					if not stack:is_empty() then
						exportdb[stack:get_name()] = true
					end
				end
			end
			return save_export_report()
		end
	})

minetest.register_chatcommand(modname .. "_here", {
		description = "Rip all nodes within a radius",
		params = "[radius = 100]",
		privs = {server = true},
		func = function(name, param)
			local player = minetest.get_player_by_name(name)
			if not player then return false, "invalid player" end

			local radius = tonumber(param) or 100
			local pos = player:get_pos()

			local foundids = {}
			do
				local vm = minetest.get_voxel_manip(
					vector.subtract(pos, radius),
					vector.add(pos, radius))
				local data = vm:get_data()
				for i = 1, #data do foundids[data[i]] = true end
			end

			for k in pairs(foundids) do
				local n = minetest.get_name_from_content_id(k)
				if n then exportdb[n] = true end
			end

			return save_export_report()
		end
	})

local function patternfunc(setto)
	return function(_, param)
		if param == "" then return false, "must supply pattern" end
		local ok, err = pcall(function()
				for k in pairs(minetest.registered_items) do
					if string_match(k, param) then
						exportdb[k] = setto
					end
				end
			end)
		if not ok then return false, string_gsub(err, ".*:%d+:%s*", "") end
		return save_export_report()
	end
end
minetest.register_chatcommand(modname .. "_add", {
		description = "Rip all defs with technical names matching a pattern",
		params = "<lua pattern>",
		privs = {server = true},
		func = patternfunc(true)
	})
minetest.register_chatcommand(modname .. "_rm", {
		description = "Un-rip all defs with technical names matching a pattern",
		params = "<lua pattern>",
		privs = {server = true},
		func = patternfunc(nil)
	})
