-- LUALOCALS < ---------------------------------------------------------
local io, ipairs, minetest, os, pairs, string, type
    = io, ipairs, minetest, os, pairs, string, type
local io_open, os_remove, string_gsub, string_sub
    = io.open, os.remove, string.gsub, string.sub
-- LUALOCALS > ---------------------------------------------------------

local include = ...
local dumptable, sortedpairs = include("dumptable")
local getdir, ripmedia = include("fileops")
local exportdb = include("exportdb")

local modname = minetest.get_current_modname()

local nodekeys = {
	climbable = true,
	collision_box = true,
	description = true,
	drawtype = true,
	inventory_image = true,
	legacy_facedir_simple = true,
	light_source = true,
	liquid_move_physics = function(def)
		return def.liquid_move_physics
		or def.liquidtype ~= nil and def.liquidtype ~= "none"
		or nil
	end,
	mesh = true,
	node_box = true,
	paramtype = true,
	paramtype2 = true,
	place_param2 = true,
	selection_box = true,
	sounds = true,
	special_tiles = true,
	sunlight_propagates = true,
	tiles = true,
	type = true,
	use_texture_alpha = true,
	visual_scale = true,
	walkable = true,
	wield_image = true,
}

local blocked = {
	air = true,
	ignore = true
}

local function exportall()
	local count = 0

	local outdir = getdir(minetest.get_worldpath() .. "/" .. modname)

	local outf = io_open(outdir .. "/exported.lua", "w")
	outf:write("-- AUTOMATICALLY GENERATED by <https://gitlab.com/szest/defripper>"
		.. "\n\nlocal reg = ...\n\n")

	local filtered = {}
	for k, v in pairs(minetest.registered_items) do
		if not blocked[k] and exportdb[k] then
			local t = {}
			for k2, v2 in pairs(v) do
				local keydef = nodekeys[k2]
				if type(keydef) == "function" then
					t[k2] = keydef(v)
				elseif keydef then
					t[k2] = v2
				end
			end
			t._raw_name = k
			filtered[k] = t
		end
	end

	local localized = {}
	do
		local candidates = {}
		for k, v in pairs(filtered) do
			dumptable(v, {
					path = string_gsub(k, "%W", "_"),
					locals = function(item, ctx, obj)
						local cand = candidates[item]
						if not cand then
							cand = {
								qty = 0,
								name = ctx.path,
								obj = obj
							}
							candidates[item] = cand
						end
						cand.qty = cand.qty + 1
						if #ctx.path < #cand.name
						or (#ctx.path == #cand.name
							and ctx.path < cand.name) then
							cand.name = ctx.path
						end
					end
				})
		end
		for k, v in pairs(candidates) do
			dumptable(v.obj, {
					path = v.name,
					locals = function(item)
						if item ~= k then
							local cand = candidates[item]
							cand.rawqty = cand.rawqty or cand.qty
							cand.useby = cand.useby or {}
							cand.useby[v.name] = true
							cand.qty = cand.qty - v.qty + 1
						end
					end
				})
		end
		for k, v in sortedpairs(candidates) do
			if #k + #v.name * (v.qty + 1) < #k * v.qty then
				outf:write("local " .. v.name .. " = " .. k .. "\n\n")
				localized[k] = v.name
			end
		end
	end

	local mymedia = {}
	for _, v in sortedpairs(filtered) do
		ripmedia(mymedia, v.sounds, outdir, "sounds", function(spec, fn)
				return string_sub(fn, 1, #spec + 1) == spec .. "."
				and string_sub(fn, -4) == ".ogg"
			end)
		ripmedia(mymedia, v.tiles, outdir, "textures")
		ripmedia(mymedia, v.special_tiles, outdir, "textures")
		ripmedia(mymedia, v.inventory_image, outdir, "textures")
		ripmedia(mymedia, v.wield_image, outdir, "textures")
		ripmedia(mymedia, v.mesh, outdir, "models")
		outf:write("reg(" .. dumptable(v, {
					locals = function(item)
						return localized[item]
					end
				}) .. ")\n\n")
		count = count + 1
	end
	outf:close()

	for _, mdir in ipairs({"sounds", "textures", "models"}) do
		for _, fn in pairs(minetest.get_dir_list(outdir .. "/" .. mdir, false) or {}) do
			local fulldest = outdir .. "/" .. mdir .. "/" .. fn
			if not mymedia[fulldest] then
				os_remove(fulldest)
			end
		end
	end

	local mediaqty = 0
	for _ in pairs(mymedia) do mediaqty = mediaqty + 1 end

	return count, mediaqty
end

return exportall
