-- LUALOCALS < ---------------------------------------------------------
local minetest, pairs, rawget, rawset, vector
    = minetest, pairs, rawget, rawset, vector
-- LUALOCALS > ---------------------------------------------------------

local modname = minetest.get_current_modname()

local api = rawget(_G, modname) or {}
rawset(_G, modname, api)

------------------------------------------------------------------------
-- Immediate Check Utility

-- Perform an immediate check on a node for validation, and apply
-- the good/bad alternative change if needed.

-- node parameter is optional; if provided, it's trusted.

function api.fdm_validate_check(pos, node)
	node = node or minetest.get_node(pos)
	local def = minetest.registered_nodes[node.name]
	local val = def and def.fdm_validate
	if not val then return end
	local valid = val(pos, node)
	if valid and def.fdm_alternative_good then
		node.name = def.fdm_alternative_good
		return minetest.set_node(pos, node)
	elseif (not valid) and def.fdm_alternative_bad then
		node.name = def.fdm_alternative_bad
		return minetest.set_node(pos, node)
	end
end

------------------------------------------------------------------------
-- Immediate Check on Node Change

-- If a node is updated, check it immediately, as well as all of its
-- immediate neighbors (recursively if necessary).

local function nodeupdated(pos, node, ...)
	local check = api.fdm_validate_check
	check(pos, node)
	check(vector.add(pos, vector.new(0, 1, 0)))
	check(vector.add(pos, vector.new(1, 0, 0)))
	check(vector.add(pos, vector.new(-1, 0, 0)))
	check(vector.add(pos, vector.new(0, 0, 1)))
	check(vector.add(pos, vector.new(0, 0, -1)))
	check(vector.add(pos, vector.new(0, -1, 0)))
	return ...
end

-- Hook all MT API methods that change the node grid (except for liquid
-- transform, which is not applicable to this game) to perform update
-- checks automatically.

for fn, param in pairs({
		set_node = true,
		add_node = true,
		remove_node = false,
		swap_node = true,
		dig_node = false,
		place_node = true,
		add_node_level = false
	}) do
	local func = minetest[fn]
	minetest[fn] = function(pos, pn, ...)
		return nodeupdated(pos, param and pn, func(pos, pn, ...))
	end
end

------------------------------------------------------------------------
-- Background Validation

-- LBM to recheck immediately upon reloading areas.

minetest.register_lbm({
		name = modname .. ":check",
		run_at_every_load = true,
		nodenames = {"group:fdm_validate"},
		action = function(...) return api.fdm_validate_check(...) end
	})

-- ABM to "patrol" for nodes that need to be checked in background
-- automatically, to ensure nothing can slip past our hooks indefinitely
-- while loaded. Even if this turns out to be "expensive" and exceeds
-- the ABM budget, as it's the only ABM we anticipate needing in the
-- entire game mode, we should be fine if some mapblocks are cut off
-- and deferred until some random future check.

minetest.register_abm({
		nodenames = {"group:fdm_validate"},
		interval = 1,
		chance = 1,
		action = function(...) return api.fdm_validate_check(...) end
	})
