-- LUALOCALS < ---------------------------------------------------------
local addgroups, error, ipairs, math, minetest, pairs, piredo_map,
      piredo_player, string, type, vector
    = addgroups, error, ipairs, math, minetest, pairs, piredo_map,
      piredo_player, string, type, vector
local math_random, string_format
    = math.random, string.format
-- LUALOCALS > ---------------------------------------------------------

local modname = minetest.get_current_modname()
local hash = minetest.hash_node_position
local ambgroup = "ambiance"

------------------------------------------------------------------------
-- Discover known sources of ambiance

local known_sources = {}

local function checkamb(pos, node)
	node = node or minetest.get_node(pos)
	local def = minetest.registered_nodes[node.name]
	local amb = def and def[ambgroup]
	if not amb then return end
	if amb.surface then
		local apos = vector.offset(pos, 0, 1, 0)
		local anode = minetest.get_node(apos)
		local adef = minetest.registered_nodes[anode.name]
		if not (adef and adef.air_equivalent) then return end
	end
	return amb
end

local function notify(pos, node)
	local amb = checkamb(pos, node)
	if not amb then return end
	local key = hash(pos)
	if known_sources[key] then return end
	local c = {
		pos = pos,
		amb = amb,
		cycle = math_random() * 2 * amb.interval,
	}
	c.time = math_random() * c.cycle
	known_sources[key] = c
end

piredo_map.register_on_putschem(function(pos, _, roomdata)
		for _, p in ipairs(minetest.find_nodes_in_area(pos,
				vector.add(pos, roomdata.size),
				"group:" .. ambgroup)) do
			notify(p)
		end
	end)
minetest.register_abm({
		nodenames = {"group:" .. ambgroup},
		interval = 1,
		chance = 1,
		action = notify
	})
minetest.register_lbm({
		name = modname .. ":ambience",
		nodenames = {"group:" .. ambgroup},
		run_at_every_load = true,
		action = notify
	})

local oldtotal
minetest.register_globalstep(function(dtime)
		local total = 0
		local rm = {}
		for k, v in pairs(known_sources) do
			if checkamb(v.pos) then
				v.time = (v.time or 0) + dtime
				if (not v.cycle) or (v.time > v.cycle) then
					v.time = 0
					v.cycle = math_random() * 2 * v.amb.interval
				end
				total = total + 1
			else
				rm[#rm + 1] = k
			end
		end
		for i = 1, #rm do known_sources[rm[i]] = nil end
		if total ~= oldtotal then
			oldtotal = total
			minetest.log("info", string_format("%s sources: %d",
					ambgroup, total))
		end
	end)

------------------------------------------------------------------------
-- Manage ambiance for players

local pdata = piredo_player.create_player_data()

piredo_player.register_playerstep(function(player)
		local ppos = player:get_pos()
		ppos.y = ppos.y + player:get_properties().eye_height
		local data = pdata(player)
		for k, v in pairs(known_sources) do
			local d = data[k]
			if d and v.time == 0 then
				minetest.sound_fade(d.id, 0.1, 0)
				d = nil
			end
			if (not d) or (v.time == 0 and not v.amb.loop) then
				d = {near = vector.distance(v.pos, ppos) < 8}
				local param = {}
				for sk, sv in pairs(v.amb.spec) do param[sk] = sv end
				param.pos = v.pos
				param.gain = d.near and (param.gain or 1) / 5 or 0.0001
				param.max_hear_distance = 128
				param.loop = v.amb.loop
				param.start_time = v.time
				d.id = minetest.sound_play(param.name, param)
				data[k] = d
			else
				local near = vector.distance(v.pos, ppos) < 8
				if near ~= d.near then
					local gain = near and (v.amb.gain or 1) / 5 or 0.0001
					minetest.sound_fade(d.id, 0.1, gain)
					d.near = near
				end
			end
		end
		local rm = {}
		for k, v in pairs(data) do
			if not known_sources[k] then
				minetest.sound_stop(v.id)
				rm[#rm + 1] = k
			end
		end
		for i = 1, #rm do
			data[rm[i]] = nil
		end
	end)

local function setambiance(nodename, data)
	local def = minetest.registered_nodes[nodename]
	if not def then error(nodename .. " not found") end
	data.spec = data.spec or def.sounds and (def.sounds.footstep
		or def.sounds.dig or def.sounds.dug)
	if type(data.spec) == "string" then
		data.spec = {name = data.spec}
	end
	if not data.spec then error("no ambient sound") end
	addgroups(nodename, ambgroup)
	return minetest.override_item(nodename, {[ambgroup] = data})
end

setambiance("piredo_terrain:default__water_source", {
		interval = 25,
		gain = 0.25,
		pitch = 2,
		surface = true,
	})
setambiance("piredo_terrain:default__water_flowing", {
		interval = 10,
		gain = 0.15,
		loop = true,
	})
setambiance("piredo_terrain:default__lava_source", {
		interval = 10,
		surface = true,
		spec = "xdecor_boiling_water",
	})
setambiance("piredo_terrain:default__lava_flowing", {
		interval = 10,
		gain = 0.25,
		loop = true,
		spec = "xdecor_boiling_water",
	})

setambiance("piredo_terrain:piranesi__cauldron", {
		interval = 1,
		spec = "xdecor_boiling_water",
	})
setambiance("piredo_terrain:piranesi__cauldron_purple", {
		interval = 1,
		spec = "xdecor_boiling_water",
	})
setambiance("piredo_terrain:piranesi__cauldron_black", {
		interval = 1,
		spec = "xdecor_boiling_water",
	})
setambiance("piredo_terrain:piranesi__machine_gear_1_a", {
		interval = 5,
		spec = "piredo_puzzles_sztest_gears",
	})
setambiance("piredo_terrain:piranesi__machine_gear_2_a", {
		interval = 5,
		spec = "piredo_puzzles_sztest_gears",
	})
setambiance("piredo_terrain:fire__basic_flame", {
		interval = 10,
		spec = "fire_small",
		gain = 0.25,
		loop = true,
	})
setambiance("piredo_terrain:fire__permanent_flame", {
		interval = 10,
		spec = "fire_small",
		gain = 0.25,
		loop = true,
	})
