-- LUALOCALS < ---------------------------------------------------------
local ipairs, loadstring, math, minetest, next, os, pairs, string,
      table, tonumber
    = ipairs, loadstring, math, minetest, next, os, pairs, string,
      table, tonumber
local math_ceil, math_floor, os_time, string_format, string_gsub,
      string_rep, string_sub, table_insert
    = math.ceil, math.floor, os.time, string.format, string.gsub,
      string.rep, string.sub, table.insert
-- LUALOCALS > ---------------------------------------------------------

local modname = minetest.get_current_modname()

------------------------------------------------------------------------
-- SETTINGS

local function getconf(suff)
	return tonumber(minetest.settings:get(modname .. "_" .. suff))
end

-- How often to publish updates to players. Too infrequent and the meter
-- is no longer as "real-time", but too frequent and they'll get
-- bombarded with HUD change packets.
local interval = getconf("interval") or 2

-- The amount of time in each measurement period. This is also the
-- amount of time between each period expiration.
local period_length = getconf("period_length") or 2

-- The number of time periods across which to accumualate statistics.
local period_count = getconf("period_count") or 31

-- Size of buckets into which dtime values are sorted in weighted
-- histogram.
local bucket_step = getconf("bucket_step") or 0.05

-- Maximum number of buckets. All step times too large for any other
-- bucket will go into the highest bucket.
local bucket_max = getconf("bucket_max") or 20

-- Maximum number of characters to use for ascii bar graph
local graphbar_width = getconf("graphbar_width") or 40

-- Constructor function to pre-initialize a new period table.
local newperiod = loadstring("return {" .. string_rep("0,", bucket_max) .. "}")

-- Padding off screen corner.
local padx = getconf("padx") or 8
local pady = getconf("pady") or 24

-- Separation between graph and summary
local separate = getconf("separate") or 8

------------------------------------------------------------------------
-- MEASUREMENT

-- Queue of accounting periods.
local periods = {}

-- Precise game runtime clock.
local clock = 0

-- Collect statistics at each step.
minetest.register_globalstep(function(dtime)
		-- Update clock.
		clock = clock + dtime

		-- Find current accounting period, initialize
		-- if not already present.
		local key = math_floor(clock / period_length)
		local cur = periods[key]
		if not cur then
			cur = newperiod()
			periods[key] = cur
		end

		-- Find correct histogram bucket.
		local bucket = math_floor(dtime / bucket_step)
		if bucket > bucket_max then bucket = bucket_max
		elseif bucket < 1 then bucket = 1 end

		-- Add weight to bucket.
		cur[bucket] = cur[bucket] + dtime
	end)

------------------------------------------------------------------------
-- USER TOGGLE

-- Create a separate privilege for players to see the lagometer. This
-- feature is too "internal" to show to all players unconditionally,
-- but not so "internal" that it should depend on the "server" priv.
minetest.register_privilege("lagometer", "Can see the lagometer")

local helptext = string_format(string_gsub([[
		The lagometer is a weighted histogram of the probability distribution of
		server step times over a sliding window of the past ~%d seconds. The
		vertical axis is the step time, with labels on the right indicating the
		upper bound of each bucket. The horizontal axis is the total amount of
		time that was spent doing steps of that size, with value labels along
		the left side. Each server step will add its size to the largest bucket
		that fits it. The largest bucket (%0.2f) also includes all lag spikes
		too large to fit in any bucket. Old step time is removed from each bucket
		once it is older than the sliding window size.]], "%s+", " "),
	period_length * (period_count - 1),
	bucket_max * bucket_step)

-- Command to manually toggle the lagometer.
minetest.register_chatcommand("lagometer", {
		description = "Toggle the lagometer\n\n" .. helptext,
		privs = {lagometer = true},
		func = function(name)
			local player = minetest.get_player_by_name(name)
			if not player then return end
			local old = player:get_meta():get_string("lagometer") or ""
			local v = (old == "") and "1" or ""
			player:get_meta():set_string("lagometer", v)
			minetest.chat_send_player(name, "Lagometer: "
				.. (v ~= "" and "ON" or "OFF"))
		end,
	})

------------------------------------------------------------------------
-- REPORTING

-- Keep track of connected players and their meters.
local meters = {}

-- Pre-allocated bar graph.
local graphbar = string_rep("|", graphbar_width)

-- Function to publish current HUD to a specific player.
local hudlines = {}
local hud_elem_type = minetest.features.hud_def_type_field and "type" or "hud_elem_type"
local function updatehud(player)
	local pname = player:get_player_name()
	for line = 1, #hudlines do
		local meter = meters[pname]
		if not meter then
			meter = {}
			meters[pname] = meter
		end
		local mline = meter[line]

		-- Players with privilege will see the meter, players without
		-- will get an empty string. The meters are always left in place
		-- rather than added/removed for simplicity, and to make it easier
		-- to handle when the priv is granted/revoked while the player
		-- is connected.
		local text = ""
		if minetest.get_player_privs(pname).lagometer
		and (player:get_meta():get_string("lagometer") or "") ~= ""
		then text = hudlines[line] .. string_rep("\n", #hudlines - line) end

		-- Only apply the text if it's changed, to minimize the risk of
		-- generating useless unnecessary packets.
		if text ~= "" and not mline then
			meter[line] = {
				text = text,
				hud = player:hud_add({
						[hud_elem_type] = "text",
						position = {x = 1, y = 1},
						text = text,
						alignment = {x = -1, y = -1},
						number = 0xC0C0C0,
						offset = {x = -padx, y = -pady + (line <= bucket_max
								and -separate or 0)}
					})
			}
		elseif mline and text == "" then
			player:hud_remove(mline.hud)
			meter[line] = nil
		elseif mline and mline.text ~= text then
			player:hud_change(mline.hud, "text", text)
			mline.text = text
		end
	end
end

local function timefmt(s)
	s = math_floor(s)
	if s < 60 then return string_format("%d", s) end
	local m = math_floor(s / 60)
	s = s - m * 60
	if m < 60 then return string_format("%d:%02d", m, s) end
	local h = math_floor(m / 60)
	m = m - h * 60
	if h < 24 then return string_format("%d:%02d:%02d", h, m, s) end
	local d = math_floor(h / 24)
	h = h - d * 24
	return string_format("%d.%02d:%02d:%02d", d, h, m, s)
end

local function countkeys(t)
	local n = 0
	local k = next(t)
	while k ~= nil do
		n = n + 1
		k = next(t, k)
	end
	return n
end

-- Function to update HUD lines and publish everywhere applicable.
local meterfmt = "%2.2f %s %2.2f"
local mtversion = minetest.get_version()
local function publish()
	local max_lag = minetest.get_server_max_lag()
	local uptime = minetest.get_server_uptime()
	local connected = minetest.get_connected_players()
	local connqty = countkeys(connected)
	local entqty = countkeys(minetest.luaentities)

	-- Expire old periods, and accumulate current ones.
	local accum = newperiod()
	do
		local curkey = math_floor(clock / period_length)
		for pk, pv in pairs(periods) do
			if pk <= curkey - period_count then
				periods[pk] = nil
			else
				for ik, iv in ipairs(pv) do
					accum[ik] = accum[ik] + iv
				end
			end
		end
	end

	-- Construct the HUD text, including weighted histogram visualization.
	local total = 0
	local samples = 0
	local lines = {}
	for bucket = 1, bucket_max do
		local qty = accum[bucket]
		local size = bucket * bucket_step
		total = total + qty
		samples = samples + qty / size
		table_insert(lines, 1, qty <= 0 and "" or string_format(meterfmt, qty,
				-- Maximum width of a graph bar corresponds to 50% of the total
				-- time in the window, so that there will never be 2 bars of the
				-- same length that don't have the same amount of time, even if
				-- there is one bar that's longer than all others and is cut off.
				string_sub(graphbar, 1, math_ceil(qty * 2 * graphbar_width
						/ period_length / period_count)), size))
	end
	-- Compute quantiles.
	local quants
	if total > 0 then
		local tally = 0
		for bucket = 1, bucket_max do
			local qty = accum[bucket]
			if qty > 0 then
				local size = bucket * bucket_step
				if not quants then
					quants = {size, size, size, size, size}
				end
				quants[math_ceil(tally / total * 4) + 1] = bucket * bucket_step
				tally = tally + accum[bucket]
			end
		end
		for i = 2, 5 do
			if quants[i] < quants[i - 1] then quants[i] = quants[i - 1] end
		end
	end
	local quantstr = quants and string_format("%0.2f %0.2f %0.2f %0.2f %0.2f",
		quants[1], quants[2], quants[3], quants[4], quants[5]) or "?"
	-- Add the final statistic lines to the HUD text.
	lines[#lines + 1] = string_format("mean %0.3f quants %s", total / samples, quantstr)
	lines[#lines + 1] = string_format("v%s con %d ent %d max_lag %0.3f uptime %s",
		mtversion.string, connqty, entqty, max_lag, timefmt(uptime))
	hudlines = lines

	-- Apply the appropriate text to each meter.
	for _, player in ipairs(connected) do
		updatehud(player)
	end

	-- Publish a JSON dump for external use.
	if minetest.settings:get_bool(modname .. "_publish_json") then
		minetest.safe_file_write(
			minetest.get_worldpath() .. "/lagometer.json",
			minetest.write_json({
					timestamp = os_time(),
					mtversion = mtversion,
					uptime = uptime,
					connections = connqty,
					entities = entqty,
					max_lag = max_lag,
					mean = total / samples,
					quants = quants,
					interval = interval,
					period_length = period_length,
					period_count = period_count,
					bucket_step = bucket_step,
					bucket_max = bucket_max,
					buckets = accum
				}))
	end

end

-- Run the publish method on a timer, so that player displays
-- are updated while lag is falling off.
local function update()
	publish()
	minetest.after(interval, update)
end
minetest.after(0, update)

-- Remove meter registrations when players leave.
minetest.register_on_leaveplayer(function(player)
		meters[player:get_player_name()] = nil
	end)

-- Display meter as soon as players join.
minetest.register_on_joinplayer(updatehud)
