-- LUALOCALS < ---------------------------------------------------------
local ipairs, loadstring, math, minetest, next, os, string, table,
      tonumber
    = ipairs, loadstring, math, minetest, next, os, string, table,
      tonumber
local math_ceil, math_floor, os_time, string_format, string_rep,
      string_sub, table_insert, table_sort
    = math.ceil, math.floor, os.time, string.format, string.rep,
      string.sub, table.insert, table.sort
-- LUALOCALS > ---------------------------------------------------------

local modname = minetest.get_current_modname()

------------------------------------------------------------------------
-- SETTINGS

local function getconf(suff)
	return tonumber(minetest.settings:get(modname .. "_" .. suff))
end

-- How often to publish updates to players. Too infrequent and the meter
-- is no longer as "real-time", but too frequent and they'll get
-- bombarded with HUD change packets.
local interval = getconf("interval") or 2

-- The total amount of time to include in the reporting period, in seconds.
local report_period = getconf("report_period") or 60

-- Size of buckets into which dtime values are sorted in weighted
-- histogram.
local bucket_step = getconf("bucket_step") or 0.05

-- Maximum number of buckets. All step times too large for any other
-- bucket will go into the highest bucket.
local bucket_max = getconf("bucket_max") or 20

-- Maximum number of characters to use for ascii bar graph
local graphbar_width = getconf("graphbar_width") or 40

-- Padding off screen corner.
local padx = getconf("padx") or 8
local pady = getconf("pady") or 24

-- Separation between graph and summary
local separate = getconf("separate") or 8

------------------------------------------------------------------------
-- MEASUREMENT

-- Queue of samples.
local samples = {}

-- Precise game runtime clock.
local clock = 0

-- Collect statistics at each step.
minetest.register_globalstep(function(dtime)
		clock = clock + dtime
		samples[#samples + 1] = dtime
	end)

------------------------------------------------------------------------
-- PREFERENCES

-- Get lagometer settings per player
local function mode_get(player, meta)
	meta = meta or player:get_meta()
	local mget = function(k) return (meta:get_string(k) or "") ~= "" end
	local tbl = {
		prefer = {
			enable = mget("lagometer"),
			basic = mget("lagometer_basic"),
		}
	}
	local privs = minetest.get_player_privs(player:get_player_name())
	tbl.allow = {
		enable = tbl.prefer.enable and (privs.lagometer or privs.lagometer_basic),
		basic = tbl.prefer.basic or (not privs.lagometer)
	}
	return tbl
end

------------------------------------------------------------------------
-- HUD UPDATE

-- Keep track of connected players and their meters.
local meters = {}

-- Pre-allocated bar graph.
local graphbar = string_rep("|", graphbar_width)

-- Function to publish current HUD to a specific player.
local graphlines = {}
local summarylines = {}
local hud_elem_type = minetest.features.hud_def_type_field and "type" or "hud_elem_type"
local function updatehud(player)
	local pname = player:get_player_name()
	local hudlines = {}
	for i = 1, #graphlines do
		hudlines[#hudlines + 1] = {false, graphlines[i]}
	end
	for i = 1, #summarylines do
		hudlines[#hudlines + 1] = {true, summarylines[i]}
	end
	local mode = mode_get(player).allow
	for idx = 1, #hudlines do
		local line = hudlines[idx]

		local meter = meters[pname]
		if not meter then
			meter = {}
			meters[pname] = meter
		end
		local mline = meter[idx]

		-- Players with privilege will see the meter, players without
		-- will get an empty string. The meters are always left in place
		-- rather than added/removed for simplicity, and to make it easier
		-- to handle when the priv is granted/revoked while the player
		-- is connected.
		local text = ""
		if mode.enable and (line[1] or not mode.basic)
		then text = line[2] .. string_rep("\n", #hudlines - idx) end

		-- Only apply the text if it's changed, to minimize the risk of
		-- generating useless unnecessary packets.
		if text ~= "" and not mline then
			meter[idx] = {
				text = text,
				hud = player:hud_add({
						[hud_elem_type] = "text",
						position = {x = 1, y = 1},
						text = text,
						alignment = {x = -1, y = -1},
						number = 0xC0C0C0,
						offset = {x = -padx, y = -pady
							+ (line[1] and 0 or -separate)},
						z_index = 2000,
					})
			}
		elseif mline and text == "" then
			player:hud_remove(mline.hud)
			meter[idx] = nil
		elseif mline and mline.text ~= text then
			player:hud_change(mline.hud, "text", text)
			mline.text = text
		end
	end
end

------------------------------------------------------------------------
-- REGISTER PRIVILEGES

local function privupdated(name)
	local player = minetest.get_player_by_name(name)
	if player then return updatehud(player) end
end

-- Create a separate privilege for players to see the lagometer. This
-- feature is too "internal" to show to all players unconditionally,
-- but not so "internal" that it should depend on the "server" priv.
minetest.register_privilege("lagometer", {
		description = "Can see the full lagometer",
		on_grant = privupdated,
		on_revoke = privupdated,
	})

-- Create a separate privilege for players to see just the summary lines
-- and not the full histogram. Hiding the histogram may have some utility
-- in preventing players from overreporting lag due to psychological effects.
minetest.register_privilege("lagometer_basic", {
		description = "Can see the lagometer summary",
		on_grant = privupdated,
		on_revoke = privupdated,
	})

------------------------------------------------------------------------
-- USER TOGGLE

-- Change lagometer preferences and return correct message.
local function mode_set(metakey)
	local function modestr(t)
		return (t.enable and "on" or "off") .. ":" .. (t.basic and "basic" or "full")
	end
	return function(pname)
		local player = minetest.get_player_by_name(pname)
		if not player then return end
		local meta = player:get_meta()
		local old = meta:get_string(metakey) or ""
		meta:set_string(metakey, (old == "") and "1" or "")
		updatehud(player)
		local mode = mode_get(player, meta)
		local prefer = modestr(mode.prefer)
		local allow = modestr(mode.allow)
		return true, "Lagometer mode set to " .. prefer
		.. (prefer == allow and "" or (", but limited to "
				.. allow .. " by privileges"))
	end
end

-- Command to manually toggle the lagometer.
minetest.register_chatcommand("lagometer", {
		description = "Toggle the lagometer",
		func = mode_set("lagometer"),
	})

-- Command to toggle between full and basic lagometer mode
minetest.register_chatcommand("lagometer_mode", {
		description = "Toggle lagometer detail mode",
		func = mode_set("lagometer_basic"),
	})

------------------------------------------------------------------------
-- STATISTICS CYCLE

local function timefmt(s)
	s = math_floor(s)
	if s < 60 then return string_format("%d", s) end
	local m = math_floor(s / 60)
	s = s - m * 60
	if m < 60 then return string_format("%d:%02d", m, s) end
	local h = math_floor(m / 60)
	m = m - h * 60
	if h < 24 then return string_format("%d:%02d:%02d", h, m, s) end
	local d = math_floor(h / 24)
	h = h - d * 24
	return string_format("%d.%02d:%02d:%02d", d, h, m, s)
end

local function countkeys(t)
	local n = 0
	local s = 0
	local k = next(t)
	while k ~= nil do
		n = n + 1
		if t[k].on_step then s = s + 1 end
		k = next(t, k)
	end
	return n, s
end

-- Prune old samples.
local function prune()
	local total = 0
	for i = #samples, 2, -1 do -- i=1 would be no op
		total = total + samples[i]
		if total >= report_period then
			local size = #samples - i + 1
			for j = 1, size do
				samples[j] = samples[j + i - 1]
			end
			for j = #samples, size, -1 do
				samples[j] = nil
			end
			return
		end
	end
end

-- Function to update HUD lines and publish everywhere applicable.
local meterfmt = "%2.2f %s %2.2f"
local mtversion = minetest.get_version()
local newhistogram = loadstring("return {" .. string_rep("0,", bucket_max) .. "}")
local function publish()
	local max_lag = minetest.get_server_max_lag()
	local uptime = minetest.get_server_uptime()
	local connected = minetest.get_connected_players()
	local connqty = countkeys(connected)
	local entqty, entstep = countkeys(minetest.luaentities)

	prune()

	-- Compute histogram and some basic statistics.
	local sampleqty = #samples
	local total = 0
	local tsqr = 0
	local max = 0
	local histogram = newhistogram()
	for i = 1, sampleqty do
		local dtime = samples[i]
		total = total + dtime
		tsqr = tsqr + (dtime * dtime)
		if dtime > max then max = dtime end
		local bucket = math_floor(dtime / bucket_step) + 1
		if bucket > bucket_max then bucket = bucket_max end
		histogram[bucket] = histogram[bucket] + dtime
	end

	-- Construct the weighted histogram visualization.
	graphlines = {}
	for bucket = 1, bucket_max do
		local qty = histogram[bucket]
		local size = bucket * bucket_step
		table_insert(graphlines, 1, qty <= 0 and "" or string_format(meterfmt, qty,
				-- Maximum width of a graph bar corresponds to 50% of the total
				-- time in the window, so that there will never be 2 bars of the
				-- same length that don't have the same amount of time, even if
				-- there is one bar that's longer than all others and is cut off.
				string_sub(graphbar, 1, math_ceil(qty * 2 * graphbar_width
						/ report_period)), size))
	end

	-- Compute quantiles.
	local quants
	if sampleqty >= 1 then
		local t = {}
		for i = 1, sampleqty do t[i] = samples[i] end
		table_sort(t)
		quants = {0, 0, 0, 0, 0}
		local accum = 0
		for i = 1, sampleqty do
			local ti = t[i]
			local idx = math_ceil(accum / total * 4 + 1)
			quants[idx] = ti
			accum = accum + ti
		end
	end

	-- Construct final summary HUD lines.
	summarylines = {}
	local quantstr = quants and string_format("%0.2f %0.2f %0.2f %0.2f %0.2f",
		quants[1], quants[2], quants[3], quants[4], quants[5]) or "?"
	summarylines[#summarylines + 1] = string_format("avg %0.3f mtbs %0.3f quants %s",
		total / sampleqty, tsqr / total, quantstr)
	summarylines[#summarylines + 1] = string_format("v%s con %d ent %d es %d max_lag %0.3f uptime %s",
		mtversion.string, connqty, entqty, entstep, max_lag, timefmt(uptime))

	-- Apply the appropriate text to each meter.
	for _, player in ipairs(connected) do
		updatehud(player)
	end

	-- Publish a JSON dump for external use.
	if minetest.settings:get_bool(modname .. "_publish_json") then
		minetest.safe_file_write(
			minetest.get_worldpath() .. "/lagometer.json",
			minetest.write_json({
					timestamp = os_time(),
					mtversion = mtversion,
					uptime = uptime,
					connections = connqty,
					entities = entqty,
					max_lag = max_lag,
					total = total,
					sampleqty = sampleqty,
					mean = total / sampleqty,
					quants = quants,
					interval = interval,
					histogram = histogram,
				}))
	end

end

-- Run the publish method on a timer, so that player displays
-- are updated while lag is falling off.
local function update()
	publish()
	minetest.after(interval, update)
end
minetest.after(0, update)

-- Remove meter registrations when players leave.
minetest.register_on_leaveplayer(function(player)
		meters[player:get_player_name()] = nil
	end)

-- Display meter as soon as players join.
minetest.register_on_joinplayer(updatehud)
