--[[ This mod is basically a compromise solution
for a long-known bug.

It checks if Luanti is running with the recommended
player movement settings. If the player has set any of the
relevant movement settings to a non-default value, that will
take preference over the game settings, which will break
this game’s vision somewhat, since the levels are carefully
designed to work with the game-provided movement settings.

Since there apparently is no way for C.O.W. to force-set
the movement settings without some arcane hackery, this mod
will just check the values and display a warning if a
deviation has been detected.
(This is important because it is very hard for a player
to realize otherwise they're playing C.O.W. with the
"wrong" physics.)

The player is then given a choice to either return to
Luanti and fix the problem or to ignore it and
continue playing.
Both a dialog message and a chat message will be displayed.

TODO: Find a non-hacky (!) way for C.O.W. to force player-movement
settings, then remove this mod.
]]

cow_check_movement_settings = {}

local S = core.get_translator("cow_check_movement_settings")
local FS = function(...) return core.formspec_escape(S(...)) end
local F = core.formspec_escape

-- Threshold for floating-point equality check
local EPSILON = 0.0001

-- This is the list of settings that needed to be checked, and their
-- expected value.
-- IMPORTANT: This MUST be equal to core.conf in this game's root directory!
local settings_to_check = {
	movement_speed_walk = 5,
	movement_speed_crouch = 5,
	movement_speed_jump = 15,
	movement_gravity = 7.5,
	movement_acceleration_air = 1,
	movement_acceleration_default = 4,
}

local check_settings = function()
	local invalid_settings = {}
	for key, expected_value in pairs(settings_to_check) do
		local actual_value = tonumber(core.settings:get(key))
		if actual_value then
			-- Floating-point equality check with tolerance
			if math.abs(expected_value - actual_value) > EPSILON then
				table.insert(invalid_settings, key)
			end
		end
	end
	return invalid_settings
end

local invalid_settings = check_settings()

local registered_on_dismiss_warnings = {}
cow_check_movement_settings.register_on_dismiss_warning = function(callback)
	table.insert(registered_on_dismiss_warnings, callback)
end

local STR_WARNING_1 = S("WARNING: The player movement settings are not at the recommended values for C.O.W. The physics might not work as intended!")
local STR_WARNING_2 = S("Please exit the game and reset the following Luanti settings to their default value:")
--~ list separator for list of invalid player movement settings
local STR_INVALID_SETTINGS = table.concat(invalid_settings, S(", "))

local disconnect = function(player)
	core.log("action", "[cow_check_movement_settings] "..player:get_player_name().." chose to leave the game")
	core.disconnect_player(player:get_player_name(), S("You quit. Remember, C.O.W. expects the following Luanti settings to be reset to the default value: @1", STR_INVALID_SETTINGS))
end

local show_dialog = function(player_name)
	local form = "formspec_version[7]size[10,5]"..
		"allow_close[false]"..
		"style_type[textarea;textcolor=#ebecdc]"..
		"image[0.8,0.6;8.4,2.8;cow_gui_text_border.png;8]"..
		"textarea[1,0.8;8,2.5;;;"..F(STR_WARNING_1).."\n"..F(STR_WARNING_2).."\n\n"..STR_INVALID_SETTINGS.."]"..
		"button_exit[1,3.7;3.5,0.8;exit;"..FS("Exit game").."]"..
		"button_exit[5.5,3.7;3.5,0.8;play;"..FS("Continue playing anyway").."]"
	core.show_formspec(player_name, "cow_check_movement_settings:movement_settings_warning", form)
end

core.register_on_player_receive_fields(function(player, formname, fields)
	if formname ~= "cow_check_movement_settings:movement_settings_warning" then
		return
	end
	if fields.play then
		core.log("action", "[cow_check_movement_settings] "..player:get_player_name().." chose to play game despite the non-default movement settings")
		-- Call 'dismiss warning' callbacks
		for i=1, #registered_on_dismiss_warnings do
			registered_on_dismiss_warnings[i](player)
		end
	elseif fields.exit or fields.quit then
		disconnect(player)
	end
end)

core.register_on_joinplayer(function(player)
	local pname = player:get_player_name()
	if #invalid_settings == 0 then
		return
	end

	core.log("action", "[cow_check_movement_settings] Non-default movement settings detected, game physics might be degraded")
	-- Chat messages
	local function msg(text)
		core.chat_send_player(pname, core.colorize("#e3c054", text))

	end
	msg(STR_WARNING_1)
	msg(STR_WARNING_2)
	msg(STR_INVALID_SETTINGS)

	-- Additional, a dialog
	show_dialog(pname)
end)

-- Expose checker function publicly
cow_check_movement_settings.check_movement_settings = function()
	local t = check_settings()
	if #t == 0 then
		return true
	else
		return false
	end
end
