local S = core.get_translator("cow_game")

cow_game = {}

local countdown = nil
local countdown_job
local game_message_job

local camera_gui_active = false

local game_timer = 0
local game_timer_active = false
local game_timer_timer = 0
local timed_out = false

cow_game.STATE_MENU = 0
cow_game.STATE_LOADING_LEVEL = 1
cow_game.STATE_PREPLAY = 2
cow_game.STATE_PLAY = 3
cow_game.STATE_TIMEOUT = 4
cow_game.STATE_GAME_OVER = 5
cow_game.STATE_RESULTS = 6

local START_LIVES = 3
local STATE_TIME_TIMEOUT = 3.0
local STATE_TIME_GAME_OVER = 3.0
local IS_PROTECTED_FOR = 3.0

local current_lives = 3
local is_protected = false
local is_protected_for = 0

local cheat_godmode = false
local cheat_superboost = false
local cheat_superlaser = false

local game_state = cow_game.STATE_MENU

local score = 0

local MAX_SCORE = 2147483647

function cow_game.set_state(state)
	game_state = state
	core.log("action", "[cow_game] Game state set to: "..state)
end
function cow_game.get_state()
	return game_state
end

function cow_game.add_score(player, add_score, particle_position)
	if add_score <= 0 then
		core.log("error", "[cow_game] add_score called with non-positive score!")
		return
	end
	if particle_position then
		cow_gui.show_score_in_world(particle_position, add_score)
	end

	cow_game.set_score(player, cow_game.get_score(player)+add_score, true)
end

function cow_game.set_score(player, new_score, show)
	score = math.floor(math.min(MAX_SCORE, math.max(0, new_score)))
	if show then
		cow_gui.show_score(player, score)
	end
end
function cow_game.get_score(player)
	return score
end

function cow_game.timed_out()
	return timed_out
end

function cow_game.start_timer(player, seconds)
	game_timer_active = true
	game_timer = seconds
	game_timer_timer = 0
	cow_gui.show_time(player, seconds)
end

function cow_game.stop_timer(player)
	game_timer_active = false
	cow_gui.hide_time(player)
end

function cow_game.start_countdown(player, callback_start)
	countdown = 4
	local function update_countdown(player)
		if not player or not player:is_player() then
			return
		end
		if countdown then
			countdown = countdown - 1
			if countdown == 0 then
				core.sound_play({name="cow_game_countdown_go", gain=0.8}, {to_player=player:get_player_name()}, true)
				if callback_start then
					callback_start()
				end
			elseif countdown < 0 then
				cow_gui.hide_countdown(player)
				return
			end
			if countdown > 0 then
				core.sound_play({name="cow_game_countdown", gain=0.8}, {to_player=player:get_player_name()}, true)
			end
			cow_gui.show_countdown(player, countdown)
			countdown_job = core.after(1, update_countdown, player)
		end
	end
	countdown_job = core.after(1, update_countdown, player)
end

function cow_game.cancel_countdown(player)
	if countdown_job then
		countdown_job:cancel()
		countdown_job = nil
	end
	cow_holder.unhold_player(player)
	cow_gui.hide_countdown(player)
end

function cow_game.stop_play(player)
	cow_game.set_state(cow_game.STATE_MENU)
	cow_game.stop_timer(player)
	cow_game.cancel_countdown(player)
	cow_gui.hide_score(player)
	cow_gui.hide_lives(player)
	cow_gui.hide_boost(player)
	timed_out = false
end

function cow_game.start_play(player, settings)
	if not core.global_exists("cow_objects") then
		core.log("error", "[cow_game] cow_game.start_play must not be called before mod init!")
		return
	end
	cow_player.set_texture_mode(player, "normal")
	player:set_camera({mode="third"})
	player:hud_set_flags({crosshair=true})
	cow_game.stop_timer(player)
	cow_game.cancel_countdown(player)
	cow_gui.hide_highscore(player)
	cow_sky.init_timeofday()

	cow_game.set_state(cow_game.STATE_PREPLAY)
	cow_game.set_score(player, 0, true)

	timed_out = false
	game_timer = settings.time
	cow_gui.show_time(player, game_timer)
	cow_game.set_lives(player, START_LIVES)
	cow_game.set_protected(player, false)

	cow_levels.init_level(settings.level)
	cow_levels.go_to_level(player, settings.level)
	local spawn_pos = cow_levels.get_level_spawn_pos(settings.level)

	local ok = cow_holder.hold_player(player)
	if not ok then
		core.log("error", "[cow_game] Failed to spawn player holder for "..player:get_player_name())
	end

	local after_countdown = function()
		if player and player:is_player() then
			cow_game.start_timer(player, game_timer)
			local detached = cow_holder.unhold_player(player)
			if detached then
				core.after(0, function()
					if player and player:is_player() then
						cow_levels.go_to_level(player, settings.level, false)
						cow_gui.show_boost(player, "recharging")
					end
				end)
			end
		end
		cow_game.set_state(cow_game.STATE_PLAY)
	end
	cow_game.start_countdown(player, after_countdown)
end

function cow_game.set_lives(player, lives)
	current_lives = math.max(lives, 0)
	cow_gui.show_lives(player, current_lives)
end
function cow_game.get_lives(player)
	return lives
end
function cow_game.is_protected(player)
	return is_protected or cheat_godmode
end
function cow_game.set_protected(player, state)
	if state == true then
		is_protected = true
		is_protected_for = IS_PROTECTED_FOR
	else
		is_protected = false
		is_protected_for = 0
	end
end

function cow_game.die(player, lives)
	if not lives then
		lives = 1
	end
	lives = math.floor(lives)
	if lives <= 0 then
		return
	end
	current_lives = math.max(current_lives - lives, 0)
	cow_gui.show_lives(player, current_lives)
	core.sound_play({name="cow_game_lose_life", gain=0.9}, {to_player=player:get_player_name()}, true)
	if current_lives <= 0 then
		cow_game.game_over(player)
		cow_player.set_texture_mode(player, "dead")
		local head = cow_player.spawn_dead_body_parts(player)
		if head then
			player:set_attach(head)
		end
	else
		cow_player.set_texture_mode(player, "damaged")
		cow_game.set_protected(player, true)
	end
end
function cow_game.die_if_unprotected(player)
	if not cow_game.is_protected(player) then
		cow_game.die(player)
		return
	end
end

function cow_game.game_over(player)
	cow_gui.hide_boost(player)
	cow_game.set_state(cow_game.STATE_GAME_OVER)
	game_timer = -STATE_TIME_TIMEOUT

	if game_message_job then
		game_message_job:cancel()
	end
	cow_gui.show_game_message(player, "game_over")
	core.sound_play({name="cow_game_game_over", gain=0.5}, {to_player=player:get_player_name()}, true)
	local game_message_job = core.after(STATE_TIME_GAME_OVER, function(player)
		cow_gui.hide_game_message(player)
		game_message_job = nil
	end, player)
end

function cow_game.get_godmode()
	return cheat_godmode
end
function cow_game.get_superboost()
	return cheat_superboost
end
function cow_game.get_superlaser()
	return cheat_superlaser
end

local timer = 0
core.register_globalstep(function(dtime)
	timer = timer + dtime
	if game_timer_active then
		game_timer_timer = game_timer_timer + dtime
	end
	-- Reduce globalstep load
	if timer < 0.1 and game_timer_timer < 1 then
		return
	end

	local state = cow_game.get_state()
	local players = core.get_connected_players()
	for p=1, #players do
		if state == cow_game.STATE_PLAY and is_protected then
			is_protected_for = is_protected_for - (dtime + timer)
			if is_protected_for <= 0 then
				is_protected = false
				is_protected_for = 0
				cow_player.set_texture_mode(players[p], "normal")
			end
		end
	end

	timer = 0
	local update_game_timer = false
	if game_timer_active and game_timer_timer >= 1 then
		game_timer_timer = 0
		update_game_timer = true
	end

	if update_game_timer then
		game_timer = math.max(- (STATE_TIME_TIMEOUT + STATE_TIME_GAME_OVER), game_timer - 1)
		local entering_results = false
		if state == cow_game.STATE_PLAY and game_timer == 0 then
			timed_out = true
			cow_game.set_state(cow_game.STATE_TIMEOUT)
		elseif state == cow_game.STATE_TIMEOUT and game_timer == -STATE_TIME_TIMEOUT then
			cow_game.set_state(cow_game.STATE_GAME_OVER)
		elseif state == cow_game.STATE_GAME_OVER and game_timer == -(STATE_TIME_TIMEOUT+STATE_TIME_GAME_OVER) then
			cow_game.set_state(cow_game.STATE_RESULTS)
			entering_results = true
		end
		state = cow_game.get_state()

		for p=1, #players do
			local player = players[p]

			if state == cow_game.STATE_PLAY and game_timer >= 1 and game_timer <= 10 then
				core.sound_play({name="cow_game_countdown_final", gain=0.4}, {to_player=player:get_player_name()}, true)
			elseif state == cow_game.STATE_TIMEOUT and game_timer == 0 then
				if game_message_job then
					game_message_job:cancel()
				end
				cow_holder.hold_player(player)
				cow_gui.show_game_message(player, "timeout")
				core.sound_play({name="cow_game_timeout", gain=0.5}, {to_player=player:get_player_name()}, true)
				local game_message_job = core.after(STATE_TIME_TIMEOUT, function(player)
					cow_gui.hide_game_message(player)
					game_message_job = nil
				end, player)
			elseif state == cow_game.STATE_GAME_OVER and game_timer == -STATE_TIME_TIMEOUT then
				if game_message_job then
					game_message_job:cancel()
				end
				cow_gui.show_game_message(player, "game_over")
				core.sound_play({name="cow_game_game_over", gain=0.15}, {to_player=player:get_player_name()}, true)
				local game_message_job = core.after(STATE_TIME_GAME_OVER, function(player)
					cow_gui.hide_game_message(player)
					game_message_job = nil
				end, player)
			elseif entering_results == true then
				local is_new_highscore = cow_highscore.send_new_score(cow_game.get_score(player))
				local current_highscore = cow_highscore.get_highscore()
				cow_results.show_results(player, { score = cow_game.get_score(player), highscore = current_highscore, new_highscore = is_new_highscore })
				cow_game.stop_timer(player)
				cow_gui.hide_score(player)
				cow_gui.hide_lives(player)
			end
			if game_timer_active and cow_game.get_state() ~= cow_game.STATE_GAME_OVER then
				cow_gui.show_time(player, game_timer)
			end
		end
	end
end)

core.register_chatcommand("cancel", {
	description = S("Cancel the active game and return to the menu"),
	privs = {},
	params = "",
	func = function(name, param)
		local state = cow_game.get_state()
		if state == cow_game.STATE_MENU or state == cow_game.STATE_RESULTS then
			return false, S("There’s no game to cancel!")
		end
		local player = core.get_player_by_name(name)
		if not player then
			return false, S("No player.")
		end
		cow_game.stop_play(player)
		core.after(0, function()
			if player and player:is_player() then
				cow_menu.open_menu(player, true)
			end
		end)
		return true
	end,
})

if core.settings:get_bool("cow_cheats", false) then
	core.register_chatcommand("kill", {
		description = S("Lose a life, or multiple lives"),
		privs = { server = true },
		params = S("[<lives>]"),
		func = function(name, param)
			local state = cow_game.get_state()
			if state == cow_game.STATE_PREPLAY then
				return false, S("Can’t die yet!")
			elseif state ~= cow_game.STATE_PLAY then
				return false, S("There’s no active game!")
			end
			local lives
			if param == "" then
				lives = 1
			else
				lives = tonumber(param)
				if not lives then
					return false
				end
				lives = math.max(math.floor(lives))
			end
			local player = core.get_player_by_name(name)
			if not player then
				return false, S("No player.")
			end
			cow_game.die(player, lives)
			return true
		end,
	})

	core.register_chatcommand("gameover", {
		description = S("Force the game the end"),
		privs = { server = true },
		params = "",
		func = function(name, param)
			local state = cow_game.get_state()
			if state == cow_game.STATE_PREPLAY then
				return false, S("Can’t go game over yet!")
			elseif state ~= cow_game.STATE_PLAY then
				return false, S("There’s no active game!")
			end
			local player = core.get_player_by_name(name)
			if not player then
				return false, S("No player.")
			end
			current_lives = 1
			cow_game.die(player)
			cow_game.game_over(player)
			cow_player.set_texture_mode(player, "dead")
			return true
		end,
	})

	core.register_chatcommand("add_time", {
		description = S("Add time to the clock"),
		privs = { server = true },
		--~ seconds, as in time
		params = S("<seconds>"),
		func = function(name, param)
			local state = cow_game.get_state()
			if state ~= cow_game.STATE_PREPLAY and state ~= cow_game.STATE_PLAY then
				return false, S("There’s no active game!")
			end
			local player = core.get_player_by_name(name)
			if not player then
				return false, S("No player.")
			end
			local add_time = tonumber(param)
			if not add_time then
				return false
			end
			add_time = math.max(0, math.floor(add_time))
			game_timer = game_timer + add_time
			if game_timer < 0 then
				game_timer = 0
			end
			game_timer_timer = 0
			cow_gui.show_time(player, game_timer)
			return true
		end,
	})

	core.register_chatcommand("extra_life", {
		description = S("Add one or multiple extra lives"),
		privs = { server = true },
		params = S("[<lives>]"),
		func = function(name, param)
			local state = cow_game.get_state()
			if state ~= cow_game.STATE_PREPLAY and state ~= cow_game.STATE_PLAY then
				return false, S("There’s no active game!")
			end
			local player = core.get_player_by_name(name)
			if not player then
				return false, S("No player.")
			end
			local extra_lives
			if param == "" then
				extra_lives = 1
			else
				extra_lives = tonumber(param)
			end
			if not extra_lives then
				return false
			end
			extra_lives = math.max(0, math.floor(extra_lives))
			current_lives = current_lives + extra_lives
			cow_gui.show_lives(player, current_lives)
			return true
		end,
	})

	core.register_chatcommand("godmode", {
		description = S("Toggle God Mode (be immortal)"),
		privs = { server = true },
		params = "",
		func = function(name, param)
			local state = cow_game.get_state()
			cheat_godmode = not cheat_godmode
			if cheat_godmode then
				return true, S("God Mode enabled.")
			else
				return true, S("God Mode disabled.")
			end
		end,
	})

	core.register_chatcommand("superboost", {
		description = S("Toggle Super Boost (low boost cooldown)"),
		privs = { server = true },
		params = "",
		func = function(name, param)
			local state = cow_game.get_state()
			cheat_superboost = not cheat_superboost
			if cheat_superboost then
				return true, S("Super Boost enabled.")
			else
				return true, S("Super Boost disabled.")
			end
		end,
	})

	core.register_chatcommand("superlaser", {
		description = S("Toggle Super Laser (insta-kill)"),
		privs = { server = true },
		params = "",
		func = function(name, param)
			local state = cow_game.get_state()
			cheat_superlaser = not cheat_superlaser
			if cheat_superlaser then
				return true, S("Super Laser enabled.")
			else
				return true, S("Super Laser disabled.")
			end
		end,
	})
end
