local SPAWN_POS = vector.new(0, 3.5, 0)
local DEFAULT_FOV = 92
local PITCH_MIN_VISIBLE = -0.6154
local PITCH_MAX_VISIBLE = 0.8570

local OPACITY = "95"
local TEXTURES = {"cow_player_cow.png"}
local TEXTURES_TRANSLUCENT = {"cow_player_cow.png^[opacity:"..OPACITY}

local TEXTURES_DAMAGED = {"cow_player_cow_damaged.png"}
local TEXTURES_DAMAGED_TRANSLUCENT = {"cow_player_cow_damaged.png^[opacity:"..OPACITY}

local TEXTURES_DEAD = {"blank.png"}
local TEXTURES_DEAD_TRANSLUCENT = {"blank.png"}

cow_player = {}

local texture_mode = "normal"

local update_textures = function(player)
	local pitch = player:get_look_vertical()
	local props = player:get_properties()
	local old_use_texture_alpha = props.use_texture_alpha == true
	local old_textures = props.textures

	local new_use_texture_alpha = false
	local new_textures = TEXTURES

	-- No translucency in third-person front mode
	if player:get_camera().mode == "third_front" then
		new_use_texture_alpha = false
	-- Make model translucent at certain view angles so we can still see where we're aiming
	elseif pitch > PITCH_MAX_VISIBLE or pitch < PITCH_MIN_VISIBLE then
		new_use_texture_alpha = true
	else
		new_use_texture_alpha = false
	end

	if texture_mode == "normal" then
		if new_use_texture_alpha then
			new_textures = TEXTURES_TRANSLUCENT
		else
			new_textures = TEXTURES
		end
	elseif texture_mode == "damaged" then
		if new_use_texture_alpha then
			new_textures = TEXTURES_DAMAGED_TRANSLUCENT
		else
			new_textures = TEXTURES_DAMAGED
		end
	elseif texture_mode == "dead" then
		if new_use_texture_alpha then
			new_textures = TEXTURES_DEAD_TRANSLUCENT
		else
			new_textures = TEXTURES_DEAD
		end
	else
		core.log("error", "[cow_player] Unsupported texture mode: "..tostring(mode))
		return
	end
	if old_use_texture_alpha ~= new_use_texture_alpha or old_textures[1] ~= new_textures[1] then
		player:set_properties({textures = new_textures, use_texture_alpha = new_use_texture_alpha})
	end
end

cow_player.set_texture_mode = function(player, mode)
	if mode == "damaged" or mode == "normal" or mode == "dead" then
		texture_mode = mode
		update_textures(player)
	else
		core.log("error", "[cow_player] Unsupported texture mode: "..tostring(mode))
	end
end

core.register_on_newplayer(function(player)
	player:set_pos(SPAWN_POS)
end)
core.register_on_joinplayer(function(player)
	player:set_fov(DEFAULT_FOV)
	player:set_properties({
		collisionbox = { -0.9, 0, -0.9, 0.9, 1.6, 0.9 } ,
		selectionbox = { -0.9, 0, -0.9, 0.9, 1.6, 0.9 } ,
		visual_size = { x = 10, y = 10 },
		visual = "mesh",
		mesh = "cow_player_cow.glb",
		textures = TEXTURES,
		backface_culling = true,
		use_texture_alpha = false,
		eye_height = 1.40,
	})
	player:set_eye_offset(vector.new(0, 0, 0), vector.new(0, 5, 5), vector.new(0, 9.5, -5))
	local inv = player:get_inventory()
	inv:set_size("craft", 0)
	inv:set_size("craftpreview", 0)
	inv:set_size("craftresult", 0)
	local wieldhand, hotbar
	if not core.is_creative_enabled(player:get_player_name()) then
		player:set_pos(SPAWN_POS)
		wieldhand = false
		hotbar = false
		player:set_inventory_formspec("")
		player:hud_set_hotbar_itemcount(1)
		inv:set_size("main", 0)
		player:set_camera({mode="third"})
	else
		player:set_inventory_formspec("formspec_version[6]size[10.75,5.75]list[current_player;main;0.5,0.5;8,4]")
	end
	player:hud_set_flags({
		minimap = false,
		minimap_radar = false,
		wielditem = wieldhand,
		hotbar = hotbar,
	})
	player:set_physics_override({
		sneak = false,
	})
end)

core.register_globalstep(function(dtime)
	local players = core.get_connected_players()
	for p=1, #players do
		local player = players[p]
		update_textures(player)
	end
end)


local body_parts = {
	{ "head", { -0.75, 0, -0.75, 0.5, 1, 0.75 }, { vector.new(0.0, 0.5, 0.6) } },
	{ "body", { -0.75, 0, -0.75, 0.5, 0.8, 0.75 }, { vector.new(0.0, 0.2, -0.2) }},
	{ "udder", { -0.25, 0, -0.25, 0.25, 0.2, 0.25 }, { vector.new(0.0, 0.05, -0.2) } },
	{ "leg", { -0.12, 0, -0.12, 0.12, 0.47, 0.12 }, {
		vector.new(0.25, -0.05, 0.2),
		vector.new(-0.25, -0.05, 0.2),
		vector.new(0.25, -0.05, -0.5),
		vector.new(-0.25, -0.05, -0.5)
	}},
	{ "tail", { -0.25, 0, -0.25, 0.25, 0.62, 0.25 }, { vector.new(0.0, 0.2, -0.9) } },
}

for b=1, #body_parts do
	local id = body_parts[b][1]
	local box = body_parts[b][2]
	core.register_entity("cow_player:cow_"..id, {
		initial_properties = {
			visual = "mesh",
			textures = {"cow_player_cow_dead.png"},
			pointable = false,
			physical = true,
			collide_with_objects = false,
			collisionbox = box,
			selectionbox = box,
			visual_size = { x = 10, y = 10 },
			visual = "mesh",
			mesh = "cow_player_cow_"..id..".glb",
			backface_culling = true,
			use_texture_alpha = false,
			static_save = false,
			stepheight = 2/16,
		},
		on_activate = function(self)
			self.object:set_armor_groups({immortal=1})
			self.object:set_acceleration(vector.new(0, -7.5, 0))
		end,
		on_step = function(self, dtime, moveresult)
			if not self.object:get_properties().physical then
				return
			end
			local vel = self.object:get_velocity()
			local epsilon = 0.001
			if moveresult.collides then
				-- If speed after collision is very low, stop object entirely
				if vector.length(vel) < epsilon then
					self.object:set_velocity(vector.zero())
					self.object:set_acceleration(vector.zero())
					core.log("verbose", "[cow_objects] body part '"..id.."' halted at "..core.pos_to_string(self.object:get_pos(), 1))
					return
				end
				for c=1, #moveresult.collisions do
					local collision = moveresult.collisions[c]
					if collision.axis == "y" then
						vel.y = vel.y * 0.2
						break
					end
				end
			end
			local delta = 5 * dtime

			local hvel = table.copy(vel)
			hvel.y = 0
			local speed = vector.length(hvel)
			local hdir = vector.normalize(hvel)

			if speed > epsilon then
				speed = math.max(0, speed - delta)
			elseif speed < epsilon then
				speed = math.min(0, speed + delta)
			else
				speed = 0
			end
			local svel = vector.multiply(hdir, speed)
			vel = vector.new(svel.x, vel.y, svel.z)

			self.object:set_velocity(vel)
		end,
	})
end

function cow_player.spawn_dead_body_parts(player)
	local pos = player:get_pos()
	local player_vel = player:get_velocity()
	local player_yaw = player:get_look_horizontal()
	local head
	local body_xoff = math.random(-10, 10) * 0.1
	local body_zoff = math.random(-10, 10) * 0.1
	local body_vel = vector.offset(player_vel, body_xoff, 0, body_zoff)
	for b=1, #body_parts do
		local part = body_parts[b][1]
		local id = "cow_player:cow_"..part
		local count = #body_parts[b][3]
		for c=1, count do
			local boff = body_parts[b][3][c]
			boff = vector.rotate_around_axis(boff, vector.new(0, 1, 0), player_yaw)
			local bpos = vector.add(pos, boff)
			local ent = core.add_entity(bpos, id)
			if ent then
				local vel
				if part == "body" then
					vel = body_vel
				else
					local part_xoff = math.random(-10, 10) * 0.065
					local part_zoff = math.random(-10, 10) * 0.065
					vel = vector.offset(body_vel, part_xoff, 0, part_zoff)
				end
				ent:set_velocity(vel)
				ent:set_yaw(player_yaw)
				if part == "head" then
					head = ent
				end
			end
		end
	end
	return head
end
