cow_gui = {}

local DEFAULT_FOV = 72

local huds = {}

local hud_state = {
	score = 0,
	lives = 0,
	time = 0,
	countdown = -1,
}

local FOV_MIN = 12
local FOV_MAX = 120

function cow_gui.is_small_hud_enabled(player)
	-- If the setting explicitly specifies a HUD size mode,
	-- we enforce that one.
	local hud_size_mode = core.settings:get("cow_hud_size_mode")
	if hud_size_mode == "small" then
		return true
	elseif hud_size_mode == "normal" then
		return false
	end
	-- Any other value for that setting above counts as "auto".

	-- Automatic HUD size mode ("auto"): Determine HUD size mod depending on
	-- window size and HUD scale
	local pinfo = core.get_player_window_information(player:get_player_name())
	if not pinfo then
		return false
	end
	local width = pinfo.size.x * pinfo.real_hud_scaling
	local height = pinfo.size.y * pinfo.real_hud_scaling
	if width < 1600 or height < 1200 then
		return true
	else
		return false
	end
end

local function get_hud(player, hud_name)
	for h=1, #huds do
		local def = player:hud_get(huds[h])
		if def.name == hud_name then
			-- returns HUD id, list in huds table, HUD definition
			return huds[h], h, def
		end
	end
end

-- Pixel width and height of a single score digit image
local DIGIT_SIZE = 64
local DIGIT_SIZE_SMALL = 32
-- Width of colon image in pixels
local COLON_WIDTH = 20
local COLON_WIDTH_SMALL = 10

-- Maximum score that can be displayed.
-- Not neccessarily the maximum score
-- that is technically supported.
-- Tested on 800×600 resulution.
local MAX_DISPLAYABLE_SCORE = 9999999999

local render_integer_combine_string = function(num, start_x, leading_zeroes, y_coord, small)
	local digits = {}
	local m = 1
	if not leading_zeroes then
		leading_zeroes = 1
	end
	repeat
		local digit = math.floor((num % (10^m)) / (10^(m-1)))
		table.insert(digits, digit)
		m = m+1
	until 10^(m-1) > num and m > leading_zeroes
	local tex = ""
	local i = 0
	if not start_x then
		start_x = 0
	end
	if not y_coord then
		y_coord = 0
	end
	local texture = "cow_gui_digits.png"
	local dh = DIGIT_SIZE
	if small then
		texture = "cow_gui_digits_small.png"
		dh = DIGIT_SIZE_SMALL
	end
	for d=#digits, 1, -1 do
		local digit = digits[d]
		local x = start_x + i*dh
		tex = tex .. x..","..y_coord.."="..texture.."\\^[sheet\\:10x1\\:"..digit..",0"
		if d > 1 then
			tex = tex .. ":"
		end
		i = i + 1
	end
	return tex, start_x + #digits * dh, dh
end

function cow_gui.render_integer(num, leading_zeroes, square, small)
	num = math.min(MAX_DISPLAYABLE_SCORE, math.max(0, num))
	num = math.floor(num)
	local tex, width
	local dh = DIGIT_SIZE
	if small then
		dh = DIGIT_SIZE_SMALL
	end
	if square then
		local _
		_, width = render_integer_combine_string(num, nil, leading_zeroes, nil, small)
		local sw = math.max(width, dh)
		local center_y = math.floor(sw/2)
		tex, width = render_integer_combine_string(num, nil, leading_zeroes, center_y, small)
		tex = "[combine:"..sw.."x"..sw..":"..tex
	else
		tex, width = render_integer_combine_string(num, nil, leading_zeroes, nil, small)
		tex = "[combine:"..width.."x"..dh..":"..tex
	end
	return tex, { x = width, y = dh }
end

function cow_gui.show_game_message(player, message_type)
	local texture
	if message_type == "timeout" then
		texture = cow_locale.get_localized_image(player, "cow_gui_times_up.png")
	elseif message_type == "game_over" then
		texture = cow_locale.get_localized_image(player, "cow_gui_game_over.png")
	else
		core.log("error", "[cow_gui] Invalid message_type in show_game_message: "..tostring(message_type))
		return
	end
	local small = cow_gui.is_small_hud_enabled(player)
	local hud, list_id = get_hud(player, "game_message")

	local scale = { x = 1, y = 1 }
	local offset = { x = 0, y = 100 }
	if small then
		scale = { x = 0.5, y = 0.5 }
		offset = { x = 0, y = 50 }
	end
	if hud then
		player:hud_change(hud, "text", texture)
		player:hud_change(hud, "scale", scale)
		player:hud_change(hud, "offset", offset)
	else
		hud = player:hud_add({
			type = "image",
			name = "game_message",
			text = texture,
			position = { x = 0.5, y = 0 },
			scale = scale,
			alignment = { x = 0, y = 1 },
			offset = offset,
			z_index = 71,
		})
		table.insert(huds, hud)
	end
end
function cow_gui.hide_game_message(player)
	local hud, list_id = get_hud(player, "game_message")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

function cow_gui.show_countdown(player, new_countdown)
	local texture
	if new_countdown == hud_state.countdown then
		return
	end
	if new_countdown > 0 then
		texture = cow_locale.get_localized_image(player, "cow_gui_countdown_"..new_countdown..".png")
	else
		texture = cow_locale.get_localized_image(player, "cow_gui_countdown_go.png")
	end
	local hud, list_id = get_hud(player, "countdown")
	local scale = { x = 1, y = 1 }
	local offset = { x = 0, y = 96 }
	if cow_gui.is_small_hud_enabled(player) then
		scale = { x = 0.5, y = 0.5 }
		offset = { x = 0, y = 48 }
	end
	if hud then
		player:hud_change(hud, "text", texture)
		player:hud_change(hud, "scale", scale)
		player:hud_change(hud, "offset", offset)
	else
		hud = player:hud_add({
			type = "image",
			name = "countdown",
			text = texture,
			position = { x = 0.5, y = 0 },
			scale = scale,
			alignment = { x = 0, y = 1 },
			offset = offset,
			z_index = 70,
		})
		table.insert(huds, hud)
	end
	hud_state.countdown = new_countdown
end
function cow_gui.hide_countdown(player)
	local hud, list_id = get_hud(player, "countdown")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

function cow_gui.show_highscore(player)
	local small = cow_gui.is_small_hud_enabled(player)
	local hud_text = get_hud(player, "highscore_text")

	local scale = { x = 1, y = 1 }
	local offset = { x = -16, y = -96 }
	if small == true then
		scale = { x = 0.5, y = 0.5 }
		offset = { x = -8, y = -48 }
	end
	if hud_text then
		player:hud_change(hud_text, "scale", scale)
		player:hud_change(hud_text, "offset", offset)
	else
		hud_text = player:hud_add({
			type = "image",
			name = "highscore_text",
			text = "cow_highscore_highscore.png",
			position = { x = 1, y = 1 },
			scale = scale,
			size = { x = 300, y = 100 },
			alignment = { x = -1, y = -1 },
			offset = offset,
			z_index = 80,
		})
		table.insert(huds, hud_text)
	end

	local hud_highscore = get_hud(player, "highscore")
	local highscore = cow_highscore.get_highscore()
	local tex, size = cow_gui.render_integer(highscore, nil, nil, small)

	offset = { x = -16, y = -16 }
	if small then
		offset = { x = -8, y = -8 }
	end
	if hud_highscore then
		player:hud_change(hud_highscore, "text", tex)
		player:hud_change(hud_highscore, "size", size)
		player:hud_change(hud_highscore, "offset", offset)
	else
		hud_highscore = player:hud_add({
			type = "image",
			name = "highscore",
			text = tex,
			position = { x = 1, y = 1 },
			scale = { x = 1, y = 1 },
			size = size,
			alignment = { x = -1, y = -1 },
			offset = offset,
			z_index = 80,
		})
		table.insert(huds, hud_highscore)
	end
end
function cow_gui.hide_highscore(player)
	local hud, list_id = get_hud(player, "highscore_text")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
	hud, list_id = get_hud(player, "highscore")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

function cow_gui.show_score(player, score)
	if score then
		hud_state.score = score
	else
		score = hud_state.score
	end
	local hud = get_hud(player, "score")
	local small = cow_gui.is_small_hud_enabled(player)
	local tex, size = cow_gui.render_integer(score, nil, nil, small)
	local offset = { x = -32, y = 32 }
	if small then
		offset = { x = -16, y = 16 }
	end
	if hud then
		player:hud_change(hud, "text", tex)
		player:hud_change(hud, "size", size)
		player:hud_change(hud, "offset", offset)
	else
		hud = player:hud_add({
			type = "image",
			name = "score",
			text = tex,
			position = { x = 1, y = 0 },
			scale = { x = 1, y = 1 },
			size = size,
			alignment = { x = -1, y = 1 },
			offset = offset,
			z_index = 80,
		})
		table.insert(huds, hud)
	end
end
function cow_gui.hide_score(player, score)
	local hud, list_id = get_hud(player, "score")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

function cow_gui.show_score_in_world(pos, score)
	local size
	if score < 10 then
		size = 5
	elseif score < 100 then
		size = 10
	elseif score < 1000 then
		size = 24
	else
		size = 36
	end
	core.add_particlespawner({
		amount = 1,
		time = 0.001,
		pos = pos,
		vel = vector.new(0, 1, 0),
		texture = {
			name = cow_gui.render_integer(score, nil, true),
			alpha_tween = { start = 0.8, 1, 0 },
		},
		exptime = 3,
		size = size,
	})
end

local render_time = function(seconds, small)
	seconds = math.floor(seconds)
	seconds = math.max(0, seconds)
	local minutes = math.floor(seconds / 60)
	local seconds_mod = seconds % 60

	-- Limit maximum displayable minutes and seconds
	if minutes >= 999 then
		minutes = 999
		seconds_mod = 59
	end

	local dh = DIGIT_SIZE
	local colon_w = COLON_WIDTH
	local colon_tex = "cow_gui_colon.png"
	if small then
		dh = DIGIT_SIZE_SMALL
		colon_w = COLON_WIDTH_SMALL
		colon_tex = "cow_gui_colon_small.png"
	end
	local tex_minutes, m_w = render_integer_combine_string(minutes, nil, nil, nil, small)
	local tex_seconds, s_w = render_integer_combine_string(seconds_mod, m_w + colon_w, 2, nil, small)
	local w = m_w + colon_w + s_w
	local tex = "[combine:"..w.."x"..dh..":"..tex_minutes..":"..(m_w)..",0="..colon_tex..":"..tex_seconds
	return tex, { x = w, y = dh }
end

function cow_gui.show_time(player, seconds)
	if seconds then
		hud_state.time = seconds
	else
		seconds = hud_state.time
	end
	local hud = get_hud(player, "time")
	local small = cow_gui.is_small_hud_enabled(player)
	local tex, size = render_time(seconds, small)
	local offset = { x = 0, y = 32 }
	if small then
		offset = { x = 0, y = 16 }
	end
	if hud then
		player:hud_change(hud, "text", tex)
		player:hud_change(hud, "size", size)
		player:hud_change(hud, "offset", offset)
	else
		hud = player:hud_add({
			type = "image",
			name = "time",
			text = tex,
			position = { x = 0.5, y = 0 },
			scale = { x = 1, y = 1 },
			size = size,
			alignment = { x = 0, y = 1 },
			offset = offset,
			z_index = 60,
		})
		table.insert(huds, hud)
	end
end
function cow_gui.hide_time(player)
	local hud, list_id = get_hud(player, "time")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

function cow_gui.show_lives(player, lives)
	if lives then
		hud_state.lives = lives
	else
		lives = hud_state.lives
	end
	local hud_statbar = get_hud(player, "lives")
	local overlives = false
	local statbar_num = lives*2
	if lives > 5 then
		statbar_num = 2
		overlives = true
	end
	local small = cow_gui.is_small_hud_enabled(player)
	local offset = { x = 32, y = 32 }
	local size = { x = 70, y = 64 }
	local texture = "cow_gui_life.png"
	if small then
		offset = { x = 16, y = 16 }
		size = { x = 38, y = 32 }
		texture = "cow_gui_life_small.png"
	end
	if hud_statbar then
		player:hud_change(hud_statbar, "number", statbar_num)
		player:hud_change(hud_statbar, "offset", offset)
		player:hud_change(hud_statbar, "size", size)
		player:hud_change(hud_statbar, "text", texture)
	else
		hud_statbar = player:hud_add({
			type = "statbar",
			name = "lives",
			number = statbar_num,
			text = texture,
			position = { x = 0, y = 0 },
			scale = { x = 1, y = 1 },
			size = size,
			alignment = { x = 1, y = 1 },
			offset = offset,
			z_index = 59,
		})
		table.insert(huds, hud_statbar)
	end
	local hud_number = get_hud(player, "lives_number")
	local lives_number_shown = math.min(lives, 99)
	local lives_number_texture
	if overlives then
		lives_number_texture = cow_gui.render_integer(lives_number_shown, nil, nil, small)
	else
		lives_number_texture = "blank.png"
	end
	offset = { x = 98, y = 32 }
	if small then
		offset = { x = 54, y = 16 }
	end
	if hud_number then
		player:hud_change(hud_number, "text", lives_number_texture)
		player:hud_change(hud_number, "offset", offset)
	else
		hud_number = player:hud_add({
			type = "image",
			name = "lives_number",
			text = lives_number_texture,
			position = { x = 0, y = 0 },
			scale = { x = 1, y = 1 },
			size = { x = 64, y = 64 },
			alignment = { x = 1, y = 1 },
			offset = offset,
			z_index = 59,
		})
		table.insert(huds, hud_number)
	end
end
function cow_gui.hide_lives(player)
	local hud, list_id = get_hud(player, "lives")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
	hud, list_id = get_hud(player, "lives_number")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

function cow_gui.show_boost(player, state)
	local hud = get_hud(player, "boost")
	local texture
	if state == "recharging" then
		texture = "cow_gui_boost_recharging.png"
	elseif state == "ready" then
		texture = "cow_gui_boost_ready.png"
	elseif state == "arming" then
		texture = "cow_gui_boost_arming.png"
	end

	if hud then
		player:hud_change(hud, "text", texture)
	else
		local scale = { x = 1, y = 1 }
		local offset = { x = 0, y = -24 }
		if cow_gui.is_small_hud_enabled(player) then
			scale = { x = 0.5, y = 0.5 }
			offset = { x = 0, y = -12 }
		end
		hud = player:hud_add({
			type = "image",
			name = "boost",
			text = texture,
			position = { x = 0.5, y = 1 },
			scale = scale,
			size = { x = 64, y = 64 },
			alignment = { x = 0, y = -1 },
			offset = offset,
			z_index = 81,
		})
		table.insert(huds, hud)
	end
end
function cow_gui.hide_boost(player)
	local hud, list_id = get_hud(player, "boost")
	if hud then
		player:hud_remove(hud)
		table.remove(huds, list_id)
	end
end

core.register_on_joinplayer(function(player)
	player:set_formspec_prepend("bgcolor[#a2c04700]"..
		"background9[0,0;10,10;cow_gui_formspec_border.png;true;12]"..
		"listcolors[#1e7453;#55a058;#232228;#1e7453;#ebecdc]"..
		"style_type[image_button;border=false;content_offset=0]"..
		"style_type[button,button_exit,image_button;sound=cow_gui_select]"..
		"style_type[label,textarea;textcolor=#232228]"..
		"style_type[box;colors=#1e7453;bordercolors=#FFFFFF;borderwidths=-2]"..
		"style_type[button,button:hovered,button:pressed;bgimg_middle=8;border=false]"..
		"style_type[button;bgimg=cow_gui_button.png]"..
		"style_type[button:hovered;bgimg=cow_gui_button_hover.png]"..
		"style_type[button:pressed;bgimg=cow_gui_button_pressed.png]"..
		"style_type[button,button:hovered;textcolor=#ebecdc]"..
		"style_type[button:pressed;textcolor=#232228]")
end)


-- Periodically poll the player window information and redraw
-- some of the HUDs if the window size changed or has just
-- become available. This is because we offer some of the HUDs
-- in a small and a large version, the smaller being used for
-- smaller windows sizes.
local redrawworthy_hud_names = {
	"lives",
	"score",
	"highscore",
	"time",
}
local redraw_huds = function(player)
	local pname = player:get_player_name()
	for r=1, #redrawworthy_hud_names do
		local hud_name = redrawworthy_hud_names[r]
		local hud = get_hud(player, hud_name)
		if hud then
			cow_gui["show_"..hud_name](player)
		end
	end
	core.log("info", "[cow_gui] Redrawn HUDs for "..pname)
end
local pinfos = {}
local timer = 0
core.register_globalstep(function(dtime)
	timer = timer + dtime
	if timer < 1 then
		return
	end
	timer = 0
	local players = core.get_connected_players()
	for p=1, #players do
		local player = players[p]
		local pname = player:get_player_name()
		local pinfo = core.get_player_window_information(pname)
		if not pinfos[pname] then
			if pinfo then
				pinfos[pname] = pinfo
				redraw_huds(player)
			end
		else
			if not (pinfos[pname].size.x == pinfo.size.x and pinfos[pname].size.y == pinfo.size.y) then
				pinfos[pname] = pinfo
				redraw_huds(player)
			end
		end
	end
end)

