local S = core.get_translator("eb_graffiti")

local SEGMENT_SIZE = 32

-- Sizes
local ONE_BY_ONE = { x = 1, y = 1 }
local TWO_BY_TWO = { x = 2, y = 2 }
local THREE_BY_TWO = { x = 3, y = 2 }

local locale_graffiti_dir_list
local get_translated_textures = function(base_texture_name)
	local path = core.get_modpath("eb_graffiti").."/textures/locale/"
	local dir_list = locale_graffiti_dir_list
	if dir_list == nil then
		dir_list = core.get_dir_list(path, false)
		locale_graffiti_dir_list = dir_list
	end
	local translated = {}
	for d=1, #dir_list do
		local file = dir_list[d]
		local first_dot = string.find(file, ".", nil, true)
		if not first_dot then
			core.log("error", "[eb_graffiti] Bad file name of localized graffiti: '"..file.."'. Must use form '<basename>.<language>.png'.")
		else
			local second_dot = string.find(file, ".", first_dot+1, true)
			if not second_dot then
				core.log("error", "[eb_graffiti] Bad file name of localized graffiti: '"..file.."'. Must use form '<basename>.<language>.png'.")
			else
				local base_file_name = string.sub(file, 1, first_dot-1)
				local language = string.sub(file, first_dot+1, second_dot-1)
				if base_file_name == base_texture_name then
					table.insert(translated, { filename = file, language = language })
				end
			end
		end
	end
	return translated
end

local translated_texture_exists = function(texture_name)
	local path = core.get_modpath("eb_graffiti").."/textures/locale/"
	local dir_list = locale_graffiti_dir_list
	if dir_list == nil then
		dir_list = core.get_dir_list(path, false)
		locale_graffiti_dir_list = dir_list
	end
	for d=1, #dir_list do
		local file = dir_list[d]
		if file == texture_name then
			return true
		end
	end
	return false
end

--[[ Registers a graffiti. This will register at least 1 node, but probably
more. A graffiti may come in different sizes, given in X and Y dimensions,
where 1 unit equals 1 node length. The smallest graffiti is 1×1.
Larger graffiti consist of multiple nodes to build the whole
graffiti. Each node of this graffiti is called a "segment.

This also registers all language variants, provided a matching texture
is available in textures/locale and `translatable` is true.

The graffiti texture provided in `image` *must* have each of its dimensions
be equal to `size.x` and `size.y`, times 32 for each dimension.
For example, a 1×1 graffiti must have a texture size of 32×32.

Arguments:
* subname: Unique ID and part of the nodenames
* def: Definition table. It contains the following keys:
	* description: Short summary/description text for inventory. May be translated.
	* size: A table with keys `x` and `y` defining the number of nodes the graffiti spans
	* image: Graffiti texture base name. File stored in `textures` directory, but *without* the file name suffix
	* text: If the graffiti contains text, write a transliteration of that (English) text here. This is case-sensitive. May be translated (default: no text)
	* translatable: If true, the game picks up translations from `textures/locale` and generates new nodes for it (default: false)
})

]]
local register_graffiti = function(subname, def)
	local nb, tile
	local nb = { -0.5, -0.5, 31/64, 0.5, 0.5, 0.5 }

	local textures
	if def.translatable then
		-- Add file names of translated textures
		textures = get_translated_textures(def.image)
		-- Add file name of English texture
		table.insert(textures, {language="en", filename=def.image..".png"})
	else
		textures = {{language="en", filename=def.image..".png"}}
	end

	for t=1, #textures do
		local lang_code = textures[t].language
		local full_image = textures[t].filename
		local langsuffix
		if lang_code == "en" then
			langsuffix = ""
		else
			langsuffix = "_"..lang_code
		end

		for x=1, def.size.x do
		for y=1, def.size.y do
			local nodename, description, tile, inv_img

			local yf = def.size.y - y
			if (x == 1 and yf == 1) or (def.size.x == 1 and def.size.y == 1)  then
				nodename = "eb_graffiti:"..subname..langsuffix
			else
				nodename = "eb_graffiti:"..subname.."_"..x.."_"..y..langsuffix
			end
			if def.size.x == 1 and def.size.y == 1 then

				if def.translatable then
					--~ @1: Graffiti name, @2 language code
					description = S("Graffiti (@1, language @2)", def.description, lang_code)
				else
					--~ @1: Graffiti name
					description = S("Graffiti (@1)", def.description)
				end

				tile = full_image
				inv_img = "eb_graffiti_inventory_highlight.png^"..full_image
			else
				if def.translatable then
					--~ @1: Graffiti name, @2: language code, @3 and @4: X/Y coordinates of graffiti segment (because large graffiti are segmented into multiple parts)
					description = S("Graffiti (@1, language @2, segment @3:@4)", def.description, lang_code, x, y)
				else
					--~ @1: Graffiti name, @2 and @3: X/Y coordinates of graffiti segment (because large graffiti are segmented into multiple parts)
					description = S("Graffiti (@1, segment @2:@3)", def.description, x, y)
				end

				local x1 = SEGMENT_SIZE * (-(x-1))
				local y1 = SEGMENT_SIZE * (-(yf))
				tile = "[combine:"..SEGMENT_SIZE.."x"..SEGMENT_SIZE..":"..x1..","..y1.."="..full_image
				local ix1 = SEGMENT_SIZE * (x-1)
				local iy1 = SEGMENT_SIZE * (yf)
				inv_img = "[combine:"..(SEGMENT_SIZE*def.size.x).."x"..(SEGMENT_SIZE*def.size.y)..":"..ix1..","..iy1.."=eb_graffiti_inventory_highlight.png:0,0="..full_image
			end
			core.register_node(nodename, {
				description = description,
				paramtype = "light",
				sunlight_propagates = true,
				drawtype = "nodebox",
				paramtype2 = "facedir",
				groups = { editor_breakable = 1, graffiti = 1 },
				node_box = {
					type = "fixed",
					fixed = nb,
				},
				tiles = { "blank.png", "blank.png", "blank.png", "blank.png", "blank.png", tile },
				use_texture_alpha = "blend",
				inventory_image = inv_img,
				wield_image = tile,
				walkable = false,
				pointable = true,
				on_construct = function(pos)
					local meta = core.get_meta(pos)
					meta:set_string("infotext", def.text)
				end,
			})
		end
		end
	end
end

register_graffiti("thing_drawn", {
	--~ Graffiti description
	description = S("The Thing (drawing)"),
	text = "",
	size = TWO_BY_TWO,
	image = "eb_graffiti_thing_drawn",
})
register_graffiti("thing", {
	--~ Graffiti description
	description = S("The Thing (text)"),
	--~ Graffiti transliteration
	text = S("BEWARE THE THING!"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_thing",
	translatable = true,
})

register_graffiti("unseen", {
	--~ Graffiti description
	description = S("Unseen"),
	--~ Graffiti transliteration
	text = S("THE UNSEEN ARE LOST."),
	size = THREE_BY_TWO,
	image = "eb_graffiti_unseen",
	translatable = true,
})
register_graffiti("push", {
	--~ "Push", as in "push a button". This is a graffiti description
	description = S("Push"),
	--~ "push", as in "push a button". This is a graffiti transliteration
	text = S("PUSH\n↓"),
	size = ONE_BY_ONE,
	image = "eb_graffiti_push",
	translatable = true,
})
register_graffiti("good_eyes", {
	--~ Graffiti description
	description = S("Good Eyes"),
	--~ Graffiti transliteration
	text = S("GOOD EYES PROTECT."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_good_eyes",
	translatable = true,
})
register_graffiti("evil_eyes", {
	--~ Graffiti description
	description = S("Evil Eyes"),
	--~ Graffiti transliteration
	text = S("EVIL EYES: YOUR DEMISE."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_evil_eyes",
	translatable = true,
})
register_graffiti("open", {
	--~ Graffiti description
	description = S("Open Eyes"),
	--~ Graffiti transliteration
	text = S("OPEN THE EYES!"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_open",
	translatable = true,
})
register_graffiti("ritual", {
	--~ Graffiti description
	description = S("Ritual"),
	--~ Graffiti transliteration
	text = S("PERFORM THE RITUAL!"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_ritual",
	translatable = true,
})
register_graffiti("stand", {
	--~ Graffiti description
	description = S("Stand"),
	--~ Graffiti transliteration
	text = S("STAND HERE"),
	size = ONE_BY_ONE,
	image = "eb_graffiti_stand",
	translatable = true,
})
register_graffiti("symbol_arrow", {
	--~ Graffiti description
	description = S("Straight Arrow Symbol"),
	text = "",
	size = ONE_BY_ONE,
	image = "eb_graffiti_symbol_arrow",
})
register_graffiti("symbol_arrow_diagonal", {
	--~ Graffiti description
	description = S("Diagonal Arrow Symbol"),
	text = "",
	size = ONE_BY_ONE,
	image = "eb_graffiti_symbol_arrow_diagonal",
})

register_graffiti("symbol_eye", {
	--~ Graffiti description
	description = S("Eye Symbol"),
	text = "",
	size = ONE_BY_ONE,
	image = "eb_graffiti_symbol_eye",
})

register_graffiti("wasd", {
	--~ Graffiti description
	description = S("PC Movement Controls"),
	text =
		--~ Graffiti transliteration (PC controls)
		S("W: ↑").."\n"..
		--~ Graffiti transliteration (PC controls)
		S("A: ←").."\n"..
		--~ Graffiti transliteration (PC controls)
		S("S: ↓").."\n"..
		--~ Graffiti transliteration (PC controls)
		S("D: →"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_wasd",
	translatable = true,
})
register_graffiti("chosen", {
	--~ Graffiti description
	description = S("Chosen"),
	--~ Graffiti transliteration
	text = S("YOU’VE BEEN CHOSEN."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_chosen",
	translatable = true,
})
register_graffiti("restart", {

	description = S("Restart"),
	--~ Graffiti transliteration
	text = S("RESTART"),
	size = ONE_BY_ONE,
	image = "eb_graffiti_restart",
	translatable = true,
})
register_graffiti("thank", {
	description = S("Thank"),
	--~ Graffiti transliteration
	text = S("THANK YOU FOR PLAYING!"),
	size = THREE_BY_TWO,
	image = "eb_graffiti_thank",
	translatable = true,
})
register_graffiti("black_glass", {
	description = S("Black Glass"),
	--~ Graffiti transliteration
	text = S("THEY CAN’T SEE THRU BLACK GLASS."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_black_glass",
	translatable = true,
})
register_graffiti("patience", {
	description = S("Patience"),
	--~ Graffiti transliteration
	text = S("PATIENCE IS A VIRTUE."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_patience",
	translatable = true,
})
register_graffiti("look_back", {
	description = S("Look Back"),
	--~ Graffiti transliteration
	text = S("LOOK BACK TO MOVE FORWARDS."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_look_back",
	translatable = true,
})



for number=1, 9 do
	register_graffiti("number_"..number, {
		description = S("Number @1", tostring(number)),
		text = "",
		size = ONE_BY_ONE,
		image = "eb_graffiti_number_"..number,
	})
end
