local S = core.get_translator("eb_graffiti")

local SEGMENT_SIZE = 32

-- Sizes
local ONE_BY_ONE = { x = 1, y = 1 }
local TWO_BY_TWO = { x = 2, y = 2 }
local THREE_BY_TWO = { x = 3, y = 2 }

local registered_graffiti = {}

local available_graffiti_image_languages = { en = true }

local locale_graffiti_dir_list
local get_translated_textures = function(base_texture_name)
	local path = core.get_modpath("eb_graffiti").."/textures/locale/"
	local dir_list = locale_graffiti_dir_list
	if dir_list == nil then
		dir_list = core.get_dir_list(path, false)
		locale_graffiti_dir_list = dir_list
	end
	local translated = {}
	for d=1, #dir_list do
		local file = dir_list[d]
		local first_dot = string.find(file, ".", nil, true)
		if not first_dot then
			core.log("error", "[eb_graffiti] Bad file name of localized graffiti: '"..file.."'. Must use form '<basename>.<language>.png'.")
		else
			local second_dot = string.find(file, ".", first_dot+1, true)
			if not second_dot then
				core.log("error", "[eb_graffiti] Bad file name of localized graffiti: '"..file.."'. Must use form '<basename>.<language>.png'.")
			else
				local base_file_name = string.sub(file, 1, first_dot-1)
				local language = string.sub(file, first_dot+1, second_dot-1)
				if base_file_name == base_texture_name then
					table.insert(translated, { filename = file, language = language })
				end
			end
		end
	end
	return translated
end

local translated_texture_exists = function(texture_name)
	local path = core.get_modpath("eb_graffiti").."/textures/locale/"
	local dir_list = locale_graffiti_dir_list
	if dir_list == nil then
		dir_list = core.get_dir_list(path, false)
		locale_graffiti_dir_list = dir_list
	end
	for d=1, #dir_list do
		local file = dir_list[d]
		if file == texture_name then
			return true
		end
	end
	return false
end

--[[ Registers a graffiti. This will register at least 1 node, but probably
more. A graffiti may come in different sizes, given in X and Y dimensions,
where 1 unit equals 1 node length. The smallest graffiti is 1×1.
Larger graffiti consist of multiple nodes to build the whole
graffiti. Each node of this graffiti is called a "segment.

This also registers all language variants, provided a matching texture
is available in textures/locale and `translatable` is true.

The graffiti texture provided in `image` *must* have each of its dimensions
be equal to `size.x` and `size.y`, times 32 for each dimension.
For example, a 1×1 graffiti must have a texture size of 32×32.

Arguments:
* subname: Unique ID and part of the nodenames
* def: Definition table. It contains the following keys:
	* description: Short summary/description text for inventory. May be translated.
	* size: A table with keys `x` and `y` defining the number of nodes the graffiti spans
	* image: Graffiti texture base name. File stored in `textures` directory, but *without* the file name suffix
	* text: If the graffiti contains text, write a transliteration of that (English) text here. This is case-sensitive. May be translated (default: no text)
	* translatable: If true, the game picks up translations from `textures/locale` and generates new nodes for it (default: false)
})

]]
local register_graffiti = function(subname, def)
	local nb, tile
	local nb = { -0.5, -0.5, 31/64, 0.5, 0.5, 0.5 }

	local textures
	if def.translatable then
		-- Add file names of translated textures
		textures = get_translated_textures(def.image)
		-- Add file name of English texture
		table.insert(textures, {language="en", filename=def.image..".png"})
	else
		textures = {{language="en", filename=def.image..".png"}}
	end

	for t=1, #textures do
		local lang_code = textures[t].language
		local full_image = textures[t].filename
		local langsuffix
		if lang_code == "en" then
			langsuffix = ""
		else
			available_graffiti_image_languages[lang_code] = true
			langsuffix = "_"..lang_code
		end

		for x=1, def.size.x do
		for y=1, def.size.y do
			local nodename, description, tile, inv_img

			local yf = def.size.y - y
			if (x == 1 and yf == 1) or (def.size.x == 1 and def.size.y == 1)  then
				nodename = "eb_graffiti:"..subname..langsuffix
			else
				nodename = "eb_graffiti:"..subname.."_"..x.."_"..y..langsuffix
			end
			if def.size.x == 1 and def.size.y == 1 then

				if def.translatable then
					--~ @1: Graffiti name, @2 language code
					description = S("Graffiti (@1, language @2)", def.description, lang_code)
				else
					--~ @1: Graffiti name
					description = S("Graffiti (@1)", def.description)
				end

				tile = full_image
				inv_img = "eb_graffiti_inventory_highlight.png^"..full_image
			else
				if def.translatable then
					--~ @1: Graffiti name, @2: language code, @3 and @4: X/Y coordinates of graffiti segment (because large graffiti are segmented into multiple parts)
					description = S("Graffiti (@1, language @2, segment @3:@4)", def.description, lang_code, x, y)
				else
					--~ @1: Graffiti name, @2 and @3: X/Y coordinates of graffiti segment (because large graffiti are segmented into multiple parts)
					description = S("Graffiti (@1, segment @2:@3)", def.description, x, y)
				end

				local x1 = SEGMENT_SIZE * (-(x-1))
				local y1 = SEGMENT_SIZE * (-(yf))
				tile = "[combine:"..SEGMENT_SIZE.."x"..SEGMENT_SIZE..":"..x1..","..y1.."="..full_image
				local ix1 = SEGMENT_SIZE * (x-1)
				local iy1 = SEGMENT_SIZE * (yf)
				inv_img = "[combine:"..(SEGMENT_SIZE*def.size.x).."x"..(SEGMENT_SIZE*def.size.y)..":"..ix1..","..iy1.."=eb_graffiti_inventory_highlight.png:0,0="..full_image
			end
			core.register_node(nodename, {
				description = description,
				paramtype = "light",
				sunlight_propagates = true,
				drawtype = "nodebox",
				paramtype2 = "facedir",
				groups = { editor_breakable = 1, graffiti = 1 },
				node_box = {
					type = "fixed",
					fixed = nb,
				},
				tiles = { "blank.png", "blank.png", "blank.png", "blank.png", "blank.png", tile },
				use_texture_alpha = "blend",
				inventory_image = inv_img,
				wield_image = tile,
				walkable = false,
				pointable = true,
				on_construct = function(pos)
					local meta = core.get_meta(pos)
					meta:set_string("infotext", def.text)
				end,
			})
		end
		end
	end
	registered_graffiti[subname] = table.copy(def)
end

register_graffiti("thing_drawn", {
	--~ Graffiti description
	description = S("The Thing (drawing)"),
	text = "",
	size = TWO_BY_TWO,
	image = "eb_graffiti_thing_drawn",
})
register_graffiti("thing", {
	--~ Graffiti description
	description = S("The Thing (text)"),
	--~ Graffiti transliteration. Warns the player of a mysterious creature called "The Thing".
	text = S("BEWARE THE THING!"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_thing",
	translatable = true,
})

register_graffiti("unseen", {
	--~ Graffiti description
	description = S("Unseen"),
	--~ Graffiti transliteration. Suggests to the player that you will lose if you're not seen by eyeballs.
	text = S("THE UNSEEN ARE LOST."),
	size = THREE_BY_TWO,
	image = "eb_graffiti_unseen",
	translatable = true,
})
register_graffiti("push", {
	--~ "Push", as in "push a button". This is a graffiti description
	description = S("Push"),
	--~ Graffiti transliteration. Imperative. "push", as in "push a button". The arrow points to said button.
	text = S("PUSH\n↓"),
	size = ONE_BY_ONE,
	image = "eb_graffiti_push",
	translatable = true,
})
register_graffiti("good_eyes", {
	--~ Graffiti description
	description = S("Good Eyes"),
	--~ Graffiti transliteration. Good eyes protect the player from harm.
	text = S("GOOD EYES PROTECT."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_good_eyes",
	translatable = true,
})
register_graffiti("evil_eyes", {
	--~ Graffiti description
	description = S("Evil Eyes"),
	--~ Graffiti transliteration. Warns the player about the threat that the evil eyes pose.
	text = S("EVIL EYES: YOUR DEMISE."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_evil_eyes",
	translatable = true,
})
register_graffiti("open", {
	--~ Graffiti description
	description = S("Open Eyes"),
	--~ Graffiti transliteration. This hints to the player that some eyeballs need to be literally opened.
	text = S("OPEN THE EYES!"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_open",
	translatable = true,
})
register_graffiti("ritual", {
	--~ Graffiti description
	description = S("Ritual"),
	--~ Graffiti transliteration
	text = S("PERFORM THE RITUAL!"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_ritual",
	translatable = true,
})
register_graffiti("stand", {
	--~ Graffiti description
	description = S("Stand"),
	--~ Graffiti transliteration
	text = S("STAND HERE"),
	size = ONE_BY_ONE,
	image = "eb_graffiti_stand",
	translatable = true,
})
register_graffiti("symbol_arrow", {
	--~ Graffiti description
	description = S("Straight Arrow Symbol"),
	text = "",
	size = ONE_BY_ONE,
	image = "eb_graffiti_symbol_arrow",
})
register_graffiti("symbol_arrow_diagonal", {
	--~ Graffiti description
	description = S("Diagonal Arrow Symbol"),
	text = "",
	size = ONE_BY_ONE,
	image = "eb_graffiti_symbol_arrow_diagonal",
})

register_graffiti("symbol_eye", {
	--~ Graffiti description
	description = S("Eye Symbol"),
	text = "",
	size = ONE_BY_ONE,
	image = "eb_graffiti_symbol_eye",
})

register_graffiti("wasd", {
	--~ Graffiti description
	description = S("PC Movement Controls"),
	text =
		--~ Graffiti transliteration (PC controls)
		S("W: ↑").."\n"..
		--~ Graffiti transliteration (PC controls)
		S("A: ←").."\n"..
		--~ Graffiti transliteration (PC controls)
		S("S: ↓").."\n"..
		--~ Graffiti transliteration (PC controls)
		S("D: →"),
	size = TWO_BY_TWO,
	image = "eb_graffiti_wasd",
	translatable = true,
})
register_graffiti("chosen", {
	--~ Graffiti description
	description = S("Chosen"),
	--~ Graffiti transliteration
	text = S("YOU’VE BEEN CHOSEN."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_chosen",
	translatable = true,
})
register_graffiti("restart", {
	--~ Graffiti description
	description = S("Restart"),
	--~ Graffiti transliteration. This is a label of a portal that restarts the game.
	text = S("RESTART"),
	size = ONE_BY_ONE,
	image = "eb_graffiti_restart",
	translatable = true,
})
register_graffiti("thank", {
	--~ Graffiti description
	description = S("Thank"),
	--~ Graffiti transliteration
	text = S("THANK YOU FOR PLAYING!"),
	size = THREE_BY_TWO,
	image = "eb_graffiti_thank",
	translatable = true,
})
register_graffiti("black_glass", {
	--~ Graffiti description
	description = S("Black Glass"),
	--~ Graffiti transliteration. "They" = the eyeballs
	text = S("THEY CAN’T SEE THRU BLACK GLASS."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_black_glass",
	translatable = true,
})
register_graffiti("patience", {
	--~ Graffiti description
	description = S("Patience"),
	--~ Graffiti transliteration. A hint in the level of suspended balls. To solve the level, the player has to wait in some places.
	text = S("PATIENCE IS A VIRTUE."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_patience",
	translatable = true,
})
register_graffiti("look_back", {
	--~ Graffiti description
	description = S("Look Back"),
	--~ Graffiti transliteration. A mysterious hint in the level of suspended balls. The hidden meaning behind this is the player has to backtrack or look back to find something to proceed in the level.
	text = S("LOOK BACK TO MOVE FORWARDS."),
	size = TWO_BY_TWO,
	image = "eb_graffiti_look_back",
	translatable = true,
})



for number=1, 9 do
	register_graffiti("number_"..number, {
		--~ Graffiti description
		description = S("Number @1", tostring(number)),
		text = "",
		size = ONE_BY_ONE,
		image = "eb_graffiti_number_"..number,
	})
end



-- Hidden debug mode
if core.settings:get_bool("eb_debug", false) == true then

	-- Graffiti test: Places all available graffiti in the game

	local GRAFFITI_TEST_POS = vector.new(0, -500, 0)
	local GRAFFITI_TEST_PARAM2 = 0

	local place_single_graffiti = function(pos, graffiti_name, language)
		local gdef = registered_graffiti[graffiti_name]
		for x=1,gdef.size.x do
		for y=1,gdef.size.y do
			local nodename = "eb_graffiti:"..graffiti_name
			if x > 1 or y > 1 then
				nodename = nodename .. "_"..x.."_"..y
			end
			if language ~= "en" then
				nodename = nodename .. "_" .. language
			end
			if core.registered_nodes[nodename] then
				local gpos = vector.offset(pos, x-1, y-0, 0)
				local wpos = vector.offset(gpos, 0, 0, 1)
				core.set_node(wpos, {name = "eb_core:clean_smooth"})
				core.set_node(gpos, {name = nodename, param2 = GRAFFITI_TEST_PARAM2})
			end
		end
		end
	end

	local test_graffiti = function(pos)
		local gpos = table.copy(pos)
		local langs = {}
		for lang,_ in pairs(available_graffiti_image_languages) do
			if lang ~= "en" then
				table.insert(langs, lang)
			end
		end
		local graffiti_names = {}
		for graffiti_name,_ in pairs(registered_graffiti) do
			table.insert(graffiti_names, graffiti_name)
		end
		table.sort(langs)
		table.insert(langs, 1, "en")
		table.sort(graffiti_names)

		for g=1, #graffiti_names do
			gpos.y = pos.y
			for l=1, #langs do
				place_single_graffiti(gpos, graffiti_names[g], langs[l])
				gpos.y = gpos.y + 4
			end
			local gdef = registered_graffiti[graffiti_names[g]]
			gpos.x = gpos.x + gdef.size.x + 1
		end
	end

	-- Debug/dev command for testing all graffiti in the game
	-- to see how it renders.
	core.register_chatcommand("test_graffiti", {
		param = "",
		privs = { server = true },
		-- Intentionally NOT translated. This is an internal debug command
		description = "Place all available graffiti in the game",
		func = function(name, param)
			-- FIXME: If player is not close to the coordinates, the placement might fail.
			test_graffiti(GRAFFITI_TEST_POS)
			-- Intentionally NOT translated. This is an internal debug command
			return true, string.format("Test graffiti placed at %s.", core.pos_to_string(GRAFFITI_TEST_POS))
		end,
	})
end
