-- Bushes.
-- See README.txt for licensing and other information.

local S = minetest.get_translator("hades_bushes")

-- Min. light required for bush to grow
local MIN_BUSH_LIGHT = 9

-- The branch bush is a wooden bush that only has branches.
-- It is special because it doesn't grow.
minetest.register_node("hades_bushes:branch_bush", {
	description = S("Branch Bush"),
	drawtype = "plantlike",
	tiles = {"hades_bushes_branch.png"},
	inventory_image = "hades_bushes_branch_inv.png",
	wield_image = "hades_bushes_branch.png",
	paramtype = "light",
	paramtype2 = "meshoptions",
	place_param2 = 4,
	waving = 1,
	walkable = false,
	is_ground_content = true,
	buildable_to = true,
	floodable = true,
	groups = {snappy=3,flammable=3,attached_node=1},
	sounds = hades_sounds.node_sound_bush_defaults(),
	selection_box = {
		type = "fixed",
		fixed = {-6/16, -0.5, -6/16, 6/16, 4/16, 6/16},
	},
})

minetest.register_craft({
	output = 'hades_bushes:branch_bush',
	recipe = {
		{'hades_core:stick'},
		{'hades_core:stick'},
		{'hades_core:volcanic_sand'},
	},
})

-- Garden bushes
-- Based on the old "simple bushes" from the original Minetest Hades' simple_bushes mod

local garden_bushes = {
	{ "green", S("Garden Bush"), S("Young Garden Bush"), S("Garden Bush Seed"), false},
	{ "red", S("Garden Bush with Red Flowers"), S("Young Red-Flowering Garden Bush"), S("Red-Flowering Garden Bush Seed"), true },
	{ "orange", S("Garden Bush with Orange Flowers"), S("Young Orange-Flowering Garden Bush"), S("Orange-Flowering Garden Bush Seed"), true },
	{ "yellow", S("Garden Bush with Yellow Flowers"), S("Young Yellow-Flowering Garden Bush"), S("Yellow-Flowering Garden Bush Seed"), true },
	{ "blue", S("Garden Bush with Blue Flowers"), S("Young Blue-Flowering Garden Bush"), S("Blue-Flowering Garden Bush Seed"), true },
	{ "violet", S("Garden Bush with Violet Flowers"), S("Young Violet-Flowering Garden Bush"), S("Violet-Flowering Garden Bush Seed"), true },
	{ "white", S("Garden Bush with White Flowers"), S("Young White-Flowering Garden Bush"), S("White-Flowering Garden Bush Seed"), true },
}

-- Helper tables to make it easier to find the
-- flowerless flowering variant of garden bushes
local garden_bush_nodenames_to_flowerless = {}
local garden_bush_nodenames_from_flowerless = {}

for g=1, #garden_bushes do
	local color = garden_bushes[g][1]
	local desc = garden_bushes[g][2]
	local desc_young = garden_bushes[g][3]
	local desc_seed = garden_bushes[g][4]
	local has_flowers = garden_bushes[g][5]
	local grown_overlay_image
	if has_flowers then
		grown_overlay_image = "hades_bushes_simple_overlay.png^hades_bushes_"..color.."_overlay.png"
	else
		grown_overlay_image = "hades_bushes_simple_overlay.png"
	end
	local grown_overlay_tiles = {{ name = grown_overlay_image, color = "white" }}

	local tt_bloom = ""
	local place_node_override
	if has_flowers then
		for s=1, 2 do
			garden_bush_nodenames_to_flowerless["hades_bushes:"..color.."_"..s] = "hades_bushes:"..color.."_flowerless_"..s
			garden_bush_nodenames_from_flowerless["hades_bushes:"..color.."_flowerless_"..s] = "hades_bushes:"..color.."_"..s
		end
		-- Extend tooltip if the bush can bloom
		tt_bloom = "\n"..S("Blooms in spring and summer")

		-- In fall, place the flowerless variant instead
		place_node_override = function()
			if hades_seasons.get_season() == hades_seasons.SEASON_FALL then
				return "hades_bushes:"..color.."_flowerless"
			end
		end
	end

	hades_groundplants.register_groundplant("hades_bushes", color, {
		description = desc,
		buildable_to = false,
		colorable = true,
		set_color_on_place = true,
		place_node_override = place_node_override,
		color_stereotype = "#79f249",
		palette = "hades_bushes_palette.png",
		stages = {
			{
				description = desc_young,
				_tt_help = S("Grows on dirt in light"),
				tiles = { "hades_bushes_young_colorable.png" },
				overlay_tiles = { "hades_bushes_young_overlay.png" },
				image = "hades_bushes_young_colorable.png",
				overlay_image = "hades_bushes_young_overlay.png",
				selection_box = {
					type = "fixed",
					fixed = {-0.2, -0.5, -0.2, 0.2, 1/16, 0.2},
				},
				in_creative_inventory = false,
				drop = "hades_bushes:"..color.."_1",
			},
			{
				description = desc,
				_tt_help = S("Lives on dirt")..tt_bloom,
				tiles = { "hades_bushes_simple_colorable.png" },
				overlay_tiles = grown_overlay_tiles,
				image = "hades_bushes_simple_colorable.png",
				overlay_image = grown_overlay_image,
				selection_box = {
					type = "fixed",
					fixed = {-0.35, -0.5, -0.35, 0.35, 0.5, 0.35},
				},
				in_creative_inventory = true,
				drop = "hades_bushes:"..color.."_2",
			}
		},
		groups = {snappy=3,choppy=3,flammable=2,attached_node=1,["color_"..color]=1},
		stage_groups = { "simple_bush" },
		sounds = hades_sounds.node_sound_bush_defaults(),
	})
	minetest.register_craft({
		type = "fuel",
		recipe = "hades_bushes:"..color.."_2",
		burntime = 9,
	})
	minetest.register_craft({
		type = "fuel",
		recipe = "hades_bushes:"..color.."_1",
		burntime = 5,
	})

	if has_flowers then
		-- Flowerless variant of the garden bushes that *can* flower
		hades_groundplants.register_groundplant("hades_bushes", color.."_flowerless", {
			description = S("@1 (Flowerless)", desc),
			buildable_to = false,
			colorable = true,
			set_color_on_place = true,
			color_stereotype = "#79f249",
			palette = "hades_bushes_palette.png",
			stages = {
				{
					description = S("@1 (Flowerless)", desc_young),
					_tt_help = S("Grows on dirt in light"),
					tiles = { "hades_bushes_young_colorable.png" },
					overlay_tiles = { "hades_bushes_young_overlay.png" },
					image = "hades_bushes_young_colorable.png",
					overlay_image = "hades_bushes_young_overlay.png",
					selection_box = {
						type = "fixed",
						fixed = {-0.2, -0.5, -0.2, 0.2, 1/16, 0.2},
					},
					in_creative_inventory = false,
					drop = "hades_bushes:"..color.."_1",
				},
				{
					description = S("@1 (Flowerless)", desc),
					_tt_help = S("Lives on dirt")..tt_bloom,
					tiles = { "hades_bushes_simple_colorable.png" },
					overlay_tiles = "hades_bushes_simple_overlay.png",
					image = "hades_bushes_simple_colorable.png",
					overlay_image = "hades_bushes_simple_overlay.png",
					selection_box = {
						type = "fixed",
						fixed = {-0.35, -0.5, -0.35, 0.35, 0.5, 0.35},
					},
					in_creative_inventory = false,
					drop = "hades_bushes:"..color.."_2",
				}
			},
			groups = {snappy=3,choppy=3,flammable=2,attached_node=1,["color_"..color]=1},
			stage_groups = { "simple_bush", "simple_bush_flowerless" },
			sounds = hades_sounds.node_sound_bush_defaults(),
		})
		minetest.register_craft({
			type = "fuel",
			recipe = "hades_bushes:"..color.."_flowerless_2",
			burntime = 9,
		})
		minetest.register_craft({
			type = "fuel",
			recipe = "hades_bushes:"..color.."_flowerless_1",
			burntime = 5,
		})
	end

	hades_seeds.register_seed("hades_bushes:seed_"..color, {
		description = desc_seed,
		_tt_help = S("Grows on dirt in light").."\n"..S("Seasons: Spring, Summer"),
		image = "hades_bushes_seed_"..color..".png",
		surface_check = function(node)
			return minetest.get_item_group(node.name, "dirt") ~= 0
		end,
		extra_groups = { seed_simple_bush = 1 },
		_hades_grows_to = "hades_bushes:"..color.."_1",
	})

	minetest.register_craft({
		output = "hades_bushes:seed_"..color.." 2",
		recipe = {{"hades_bushes:"..color.."_2"}},
	})
	if color ~= "green" then
		minetest.register_craft({
			output = 'hades_bushes:seed_'..color,
			recipe = {
				{'hades_flowers:'..color},
				{'hades_bushes:green_2'},
				{'hades_core:dirt'},
			}
		})
	end
end

minetest.register_craft({
	output = 'hades_bushes:seed_green',
	recipe = {
		{'group:leaves'},
		{'group:stick'},
		{'hades_core:dirt'},
	}
})

minetest.register_node("hades_bushes:dead", {
	description = S("Dead Garden Bush"),
	drawtype = "plantlike",
	tiles = { "hades_bushes_dead.png" },
	use_texture_alpha = "clip",
	inventory_image = "hades_bushes_dead.png",
	wield_image = "hades_bushes_dead.png",
	waving = 1,
	sunlight_propagates = true,
	paramtype = "light",
	walkable = false,
	buildable_to = true,
	floodable = true,
	groups = {snappy=3, flammable=2, attached_node=1},
	sounds = hades_sounds.node_sound_bush_defaults(),
	drop = "",
	selection_box = {
		type = "fixed",
		fixed = {-0.25, -0.5, -0.25, 0.25, 1/16, 0.25},
	},
})

minetest.register_craft({
	type = "fuel",
	recipe = "hades_bushes:dead",
	burntime = 4,
})
minetest.register_craft({
	type = "fuel",
	recipe = "hades_bushes:branch_bush",
	burntime = 7,
})


local function garden_bush_flowerless_convert(nodename)
	if minetest.get_item_group(nodename, "simple_bush_flowerless") ~= 0 then
		return garden_bush_nodenames_from_flowerless[nodename]
	else
		return garden_bush_nodenames_to_flowerless[nodename]
	end
end

minetest.register_abm({
	label = "Grow garden bushes and garden bush seeds",
	nodenames = {"group:simple_bush", "group:seed_simple_bush"},
	interval = 90,
	chance = 15,
	action = function(pos, node)
		local split = node.name:split(":")
		local mname = split[1]
		local pname = split[2]
		pname = pname:gsub("seed_", "")
		pname = pname:gsub("_%d+", "")

		local soil_node = minetest.get_node({x = pos.x, y = pos.y - 1, z = pos.z})
		local is_seed = minetest.get_item_group(node.name, "seed_simple_bush") ~= 0

		-- Die if not on dirt, plant pot or soil
		if minetest.get_item_group(soil_node.name, "dirt") == 0 and
				minetest.get_item_group(soil_node.name, "soil") == 0 and
				soil_node.name ~= "hades_furniture:plant_pot" then
			if is_seed then
				minetest.remove_node(pos)
			else
				minetest.set_node(pos, {name="hades_bushes:dead"})
			end
			return
		end

		local plant_height = minetest.get_item_group(node.name, "simple_bush")

		-- return if already full grown
		if plant_height == 2 then
			return
		end

		local node_def = minetest.registered_items[node.name] or nil

		-- grow seed
		if is_seed then
			if hades_seasons.get_season() == hades_seasons.SEASON_FALL then
				-- Won't grow in fall
				return
			end
			minetest.set_node(pos, {name = mname .. ":" .. pname .. "_1"})
			return
		end

		-- check light
		local ll = minetest.get_node_light(pos)
		if not ll or ll < MIN_BUSH_LIGHT then
			return
		end

		-- grow bush
		minetest.set_node(pos, {name = mname .. ":" .. pname .. "_" .. plant_height + 1, param2 = node.param2})
	end,
})

minetest.register_abm({
	label = "Update seasonal garden bush nodes, flowers and color",
	nodenames = {"group:simple_bush"},
	interval = 5,
	chance = 20,
	action = function(pos, node)
		local season = hades_seasons.get_season()
		if not season then
			return
		end
		local old_nodename = node.name
		local new_nodename
		local old_param2 = node.param2
		local new_param2

		if season == hades_seasons.SEASON_FALL then
			-- When transitioning to fall, first the flowers disappear,
			-- then in the next step the bush leaf color is updated to
			-- the fall color.
			local convert = garden_bush_flowerless_convert(node.name)
			if convert and minetest.get_item_group(node.name, "simple_bush_flowerless") == 0 then
				new_nodename = convert
				new_param2 = old_param2
			else
				new_param2 = hades_seasons.get_seasonal_palette_color_param2(old_param2)
			end
		else
			-- When transitioning to spring/summer, first the leaf color
			-- turns to the spring/summer color, then in the next step
			-- the flowers appear.
			new_param2 = hades_seasons.get_seasonal_palette_color_param2(old_param2)
			if new_param2 ~= old_param2 then
				new_nodename = old_nodename
			elseif minetest.get_item_group(node.name, "simple_bush_flowerless") ~= 0 then
				new_nodename = garden_bush_flowerless_convert(node.name)
			end
		end
		if not new_nodename then
			new_nodename = old_nodename
		end

		if new_param2 ~= old_param2 or new_nodename ~= old_nodename then
			minetest.set_node(pos, {name = new_nodename, param2 = new_param2})
		end
	end
})

minetest.register_alias("hades_bushes:white", "hades_bushes:white_2")
minetest.register_alias("hades_bushes:red", "hades_bushes:red_2")
minetest.register_alias("hades_bushes:orange", "hades_bushes:orange_2")
minetest.register_alias("hades_bushes:yellow", "hades_bushes:yellow_2")
minetest.register_alias("hades_bushes:green", "hades_bushes:green_2")
minetest.register_alias("hades_bushes:blue", "hades_bushes:blue_2")
minetest.register_alias("hades_bushes:violet", "hades_bushes:violet_2")
