--[[

Bags for Hades Revisited

Copyright (c) 2012 cornernote, Brett O'Donnell <cornernote@gmail.com>
Original Source Code: https://github.com/cornernote/minetest-bags

]]--

local S = minetest.get_translator("hades_bags")
local F = minetest.formspec_escape

local BAGS_COUNT = 4

local bags_page = {}

local get_formspec = function(player, page)
	if page=="bags" then
		local name = player:get_player_name()
		local form = "container[1.75,2]"
		for b=1, BAGS_COUNT do
			local bx = (b-1) * 2.3
			local lx = bx + 0.5
			form = form .."list[detached:"..name.."_bags;bags;"..lx..",0;1,1;"..(b-1).."]"
			.."button["..bx..",1.2;2,0.6;bag"..b..";"..F(S("Bag @1", b)).."]"
		end
		form = form .. "container_end[]"
			.."listring[current_player;main]"
			.."listring[detached:"..name.."_bags;bags]"
		return form
	end
	for b=1,4 do
		if page=="bag"..b then
			local image = player:get_inventory():get_stack("bags", b):get_definition().inventory_image
			--~ main bags page button
			return "button[0.375,0.375;2,0.6;main;"..F(S("Main")).."]"
				.."label[3,0.675;"..F(S("Bag @1", b)).."]"
				.."image[7,0.375;1,1;"..image.."]"
				.."list[current_player;bag"..b.."contents;1.6,1.9;8,3;]"
				.."listring[]"
		end
	end
end

local go_to_bags_page = function(player, page)
	local player_name = player:get_player_name()
	local inv = player:get_inventory()
	assert(inv)

	if page == "main" then
		bags_page[player_name] = "bags"
	else
		for b=1, BAGS_COUNT do
			if page == "bag"..b and not inv:get_stack("bags", b):is_empty() then
				bags_page[player_name] = "bag"..b
				break
			end
		end
	end
	local context = sfinv.get_or_create_context(player)
	sfinv.set_player_inventory_formspec(player, context)
end

sfinv.register_page("hades_bags:bags", {
	title = S("Bags"),
	is_in_nav = function(self, player, context)
		return true
	end,
	get = function(self, player, context)
		local player_name = player:get_player_name()
		local inv = player:get_inventory()
		local page
		if not bags_page[player_name] then
			page = "bags"
		else
			page = bags_page[player_name]
		end
		return sfinv.make_formspec(player, context, get_formspec(player, page), true)
	end,
	on_player_receive_fields = function(self, player, context, fields)
		local page
		if fields.main then
			page = "main"
		else
			for b=1, BAGS_COUNT do
				if fields["bag"..b] then
					page = "bag"..b
					break
				end
			end
		end
		if not page then
			return
		end
		go_to_bags_page(player, page)
	end,
})

-- register_on_joinplayer
minetest.register_on_joinplayer(function(player)
	local player_inv = player:get_inventory()
	local bags_inv = minetest.create_detached_inventory(player:get_player_name().."_bags",{
		on_put = function(inv, listname, index, stack, player)
			player:get_inventory():set_stack(listname, index, stack)

			local bagname = "bag"..index
			player:get_inventory():set_size(bagname.."contents", stack:get_definition().groups.bagslots)
		end,
		on_take = function(inv, listname, index, stack, player)
			go_to_bags_page(player, "main")
			player:get_inventory():set_stack(listname, index, nil)
		end,
		allow_put = function(inv, listname, index, stack, player)
			if stack:get_definition().groups.bag == 1 and inv:get_stack(listname, index):is_empty() then
				return 1
			else
				return 0
			end
		end,
		allow_take = function(inv, listname, index, stack, player)
			local bagname = "bag"..index
			local pinv = player:get_inventory()
			if bagname and pinv:is_empty(bagname.."contents") then
				return stack:get_count()
			else
				return 0
			end
		end,
		allow_move = function(inv, from_list, from_index, to_list, to_index, count, player)
			return 0
		end,
	}, player:get_player_name())
	player_inv:set_size("bags", BAGS_COUNT)
	bags_inv:set_size("bags", BAGS_COUNT)
	bags_inv:set_list("bags", player_inv:get_list("bags"))
end)

minetest.register_allow_player_inventory_action(function(player, action, inventory, inventory_info)
	-- Disallow putting or moving bags into bag contents
	if action == "put" then
		if minetest.get_item_group(inventory_info.stack:get_name(), "bag") == 1 then
			for b=1, BAGS_COUNT do
				if inventory_info.listname == "bag"..b.."contents" then
					return 0
				end
			end
		end
	elseif action == "move" then
		local stack = inventory:get_stack(inventory_info.from_list, inventory_info.from_index)
		if minetest.get_item_group(stack:get_name(), "bag") == 1 then
			for b=1, BAGS_COUNT do
				if inventory_info.to_list == "bag"..b.."contents" then
					return 0
				end
			end
		end
	end
	return nil
end)

minetest.register_on_leaveplayer(function(player)
	bags_page[player:get_player_name()] = nil
end)

-- register bags
minetest.register_craftitem("hades_bags:small", {
	description = S("Small Bag"),
	stack_max = 1,
	_tt_help = S("+8 inventory slots"),
	inventory_image = "bags_small.png",
	groups = {bag=1,bagslots=8, disable_repair=1},
})
minetest.register_craftitem("hades_bags:medium", {
	description = S("Medium Bag"),
	stack_max = 1,
	_tt_help = S("+16 inventory slots"),
	inventory_image = "bags_medium.png",
	groups = {bag=1,bagslots=16, disable_repair=1},
})
minetest.register_craftitem("hades_bags:large", {
	description = S("Large Bag"),
	stack_max = 1,
	_tt_help = S("+24 inventory slots"),
	inventory_image = "bags_large.png",
	groups = {bag=1,bagslots=24, disable_repair=1},
})

-- register bag crafts
minetest.register_craft({
	output = "hades_bags:small",
	recipe = {
        {"", "hades_farming:string", ""},
        {"group:cloth", "group:cloth", "group:cloth"},
        {"group:cloth", "group:cloth", "group:cloth"},
    },
})
minetest.register_craft({
	output = "hades_bags:medium",
	recipe = {
        {"", "group:stick", ""},
        {"hades_bags:small", "hades_farming:string", "hades_bags:small"},
        {"hades_bags:small", "hades_farming:string", "hades_bags:small"},
    },
})
minetest.register_craft({
	output = "hades_bags:large",
	recipe = {
        {"", "group:stick", ""},
        {"hades_bags:medium", "hades_farming:string", "hades_bags:medium"},
        {"hades_bags:medium", "hades_farming:string", "hades_bags:medium"},
    },
})

-- Legacy aliases
minetest.register_alias("bags:small", "hades_bags:small")
minetest.register_alias("bags:medium", "hades_bags:medium")
minetest.register_alias("bags:large", "hades_bags:large")

if minetest.get_modpath("hades_ghostinv") then
	for b=1, BAGS_COUNT do
		hades_ghostinv.declare_inventory("bag"..b.."contents")
	end
	hades_ghostinv.declare_inventory("bags", "bags")
end
