local S = minetest.get_translator("hades_magic_wand")
local F = minetest.formspec_escape

minetest.register_craftitem("hades_magic_wand:magic_wand", {
	description = S("Magic Wand"),
	_tt_help = S("Changes block variant"),
	inventory_image = "default_stick.png^[colorize:#940EFE:127",
	wield_image = "default_stick.png^[colorize:#940EFE:127",
	groups = { tool = 1, disable_repair = 1 },
	on_use = function(itemstack, user, pointed_thing)
		if pointed_thing.type == "node" then
			local pos = pointed_thing.under
			local name = user:get_player_name()
			if minetest.is_protected(pos, name) and not minetest.check_player_privs(name, "protection_bypass") then
				minetest.record_protection_violation(pos, name)
				return itemstack
			end
			local node = minetest.get_node(pointed_thing.under)
			local def = minetest.registered_items[node.name]
			if def then
				if def._hades_magic_next then
					node.name = def._hades_magic_next
					minetest.swap_node(pos, node)
				elseif def._hades_grows_to then
					node.name = def._hades_grows_to
					minetest.swap_node(pos, node)
				end
			end
		end
		return itemstack
	end,
})

minetest.register_craftitem("hades_magic_wand:magic_shaper", {
	description = S("Magic Shaper"),
	_tt_help = S("Changes a block shape"),
	inventory_image = "default_stick.png^[colorize:#0F0EFF:127",
	wield_image = "default_stick.png^[colorize:#0F0EFF:127",
	groups = { tool = 1, disable_repair = 1 },
	on_use = function(itemstack, user, pointed_thing)
		if pointed_thing.type == "node" then
			local pos = pointed_thing.under
			local name = user:get_player_name()
			if minetest.is_protected(pos, name) and not minetest.check_player_privs(name, "protection_bypass") then
				minetest.record_protection_violation(pos, name)
				return itemstack
			end
			local node = minetest.get_node(pointed_thing.under)
			local def = minetest.registered_items[node.name]
			if def and def._hades_shaper_next then
				node.name = def._hades_shaper_next
				minetest.swap_node(pos, node)
			end
		end
		return itemstack
	end,
})

local owner_edits = {}

--[[ Admin tool to change owner of owned blocks.
Supports all nodes that:
* have the `owned=1` group
* store the name of the owner (a player) in the string "owner" in node meta
* (optional) define _hades_on_change_owner(pos, owner) which
  will be called on owner change with pos being the node position
  and owner being the name of the new owner (can also be the empty string)
]]
minetest.register_craftitem("hades_magic_wand:owner_changer", {
	description = S("Owner Changer"),
	_tt_help = S("Changes the owner of a block"),
	inventory_image = "default_stick.png^[colorize:#00FF00:127",
	wield_image = "default_stick.png^[colorize:#00FF00:127",
	groups = { tool = 1, disable_repair = 1 },
	on_use = function(itemstack, user, pointed_thing)
		if pointed_thing.type == "node" then
			local pos = pointed_thing.under
			if not user or not user:is_player() then
				return itemstack
			end
			local name = user:get_player_name()
			if not minetest.check_player_privs(name, "server") then
				minetest.chat_send_player(name, S("You need the “server” privilege to use this tool."))
				return itemstack
			end
			local node = minetest.get_node(pointed_thing.under)
			local def = minetest.registered_items[node.name]
			if minetest.get_item_group(node.name, "owned") == 1 then
				local meta = minetest.get_meta(pointed_thing.under)
				local owner = meta:get_string("owner")
				local desc = ItemStack(node.name):get_short_description()
				local form = "formspec_version[6]size[8,5]" ..
					"label[0.35,0.5;"..F(desc.."\n"..minetest.pos_to_string(pos)).."]"..
					"field[0.5,2;7,0.8;owner;" .. F(S("Owner")) .. ";"..F(owner).."]" ..
					"button_exit[2.5,3.5;3,1;okay;" .. F(S("Okay")) .. "]"
				owner_edits[name] = pointed_thing.under
				minetest.show_formspec(name, "hades_magic_wand:owner_changer", form)
			else
				minetest.chat_send_player(name, S("This block cannot be owned."))
				return itemstack
			end
		end
		return itemstack
	end,
})

minetest.register_on_player_receive_fields(function(sender, formname, fields)
	if formname ~= "hades_magic_wand:owner_changer" then
		return
	end

	local name = sender:get_player_name()
	if not fields.owner then
		owner_edits[name] = nil
		return
	end
	if fields.cancel then
		owner_edits[name] = nil
		return
	end
	if not fields.okay and not fields.key_enter then
		owner_edits[name] = nil
		return
	end
	local privs = minetest.get_player_privs(name)
	if not privs.server then
		minetest.chat_send_player(name, S("You need the “server” privilege to change the owner!"))
		owner_edits[name] = nil
		return
	end

	local pos = owner_edits[name]
	if not pos then
		return
	end
	local meta = minetest.get_meta(pos)
	local owner = fields.owner
	-- Trim owner name to 255 chars for safety reasons
	owner = string.sub(owner, 1, 255)
	-- Don’t do anything if owner is the same
	if owner == meta:get_string("owner") then
		return
	end
	meta:set_string("owner", owner)
	local node = minetest.get_node(pos)
	local def = minetest.registered_nodes[node.name]
	if def and def._hades_on_change_owner then
		def._hades_on_change_owner(pos, owner)
	end
	if owner ~= "" then
		minetest.chat_send_player(name, S("Owner of block at @1 changed to @2!", minetest.pos_to_string(pos), owner))
	else
		minetest.chat_send_player(name, S("Owner of block at @1 removed!", minetest.pos_to_string(pos)))
	end
	owner_edits[name] = nil
end)

minetest.register_on_leaveplayer(function(player)
	owner_edits[player:get_player_name()] = nil
end)
