local S = minetest.get_translator("hades_hunger")

hades_hunger = {}

-- Check eatable group for validity.
-- The eatable group of an itemstack MUST be equal to the hp_change argument of minetest.item_eat when that function was called.
-- Exception: The rating of 0 only triggers a warning when it's unequal to hp_change
hades_hunger.check_eatable_group = function(itemstack, hp_change)
	local eatable = minetest.get_item_group(itemstack:get_name(), "eatable")
	if eatable ~= hp_change then
		if eatable == 0 then
			minetest.log("warning", "[hades_hunger] Missing 'eatable' group for food item '"..itemstack:get_name().."'")
		else
			minetest.log("error", "[hades_hunger] 'eatable' group of '"..itemstack:get_name().."' doesn't equal 'hp_change' argument in minetest.item_eat! (eatable="..eatable.."; hp_change="..tostring(hp_change))
		end
	end
end

if minetest.settings:get_bool("enable_damage") then

hades_hunger.food = {}

-- HUD statbar values
hades_hunger.hunger = {}
hades_hunger.hunger_out = {}
hades_hunger.exhaustion = {} -- Exhaustion is for internal use only!

-- HUD item ids
local hunger_hud = {}

hades_hunger.HUD_TICK = 0.1
hades_hunger.HEAL_DMG_TICK = 4 -- time in seconds after which the player health is updated

--Some hunger settings

hades_hunger.HUNGER_TICK = 800 -- time in seconds after that 1 hunger point is taken

hades_hunger.EXHAUST_DIG = 60  -- exhaustion increased by this value after digged node
hades_hunger.EXHAUST_PLACE = 20 -- exhaustion increased by this value after placed
hades_hunger.EXHAUST_MOVE = 6 -- exhaustion increased by this value if player movement detected
hades_hunger.EXHAUST_MOVE_SNEAK = 4 -- exhaustion increased by this value if player sneak movement detected
hades_hunger.EXHAUST_LVL = 3200 -- at what exhaustion player satiation gets lowerd

hades_hunger.SAT_MAX = 20 -- maximum satiation points
hades_hunger.SAT_INIT = 20 -- initial satiation points
hades_hunger.SAT_HEAL = 15 -- required satiation points to start healing


--load custom settings
local set = io.open(minetest.get_modpath("hades_hunger").."/hades_hunger.conf", "r")
if set then 
	dofile(minetest.get_modpath("hades_hunger").."/hades_hunger.conf")
	set:close()
end

local function custom_hud(player)
	hb.init_hudbar(player, "satiation", hades_hunger.get_hunger_raw(player))
end

dofile(minetest.get_modpath("hades_hunger").."/hunger.lua")

-- register satiation hudbar
hb.register_hudbar("satiation", 0xFFFFFF, S("Satiation"), { icon = "hades_hunger_icon.png", bgicon = "hades_hunger_bgicon.png",  bar = "hades_hunger_bar.png" }, hades_hunger.SAT_INIT, hades_hunger.SAT_MAX, false, nil, { format_value = "%.1f", format_max_value = "%d" })

-- update hud elemtens if value has changed
local function update_hud(player)
	local name = player:get_player_name()
 --hunger
	local h_out = tonumber(hades_hunger.hunger_out[name])
	local h = tonumber(hades_hunger.hunger[name])
	if h_out ~= h then
		hades_hunger.hunger_out[name] = h
		hb.change_hudbar(player, "satiation", h)
	end
end

hades_hunger.get_hunger_raw = function(player)
	local inv = player:get_inventory()
	if not inv then return nil end
	local hgp = inv:get_stack("hunger", 1):get_count()
	if hgp == 0 then
		hgp = 21
		inv:set_stack("hunger", 1, ItemStack({name=":", count=hgp}))
	else
		hgp = hgp
	end
	return hgp-1
end

hades_hunger.set_hunger_raw = function(player)
	local inv = player:get_inventory()
	local name = player:get_player_name()
	local value = hades_hunger.hunger[name]
	if not inv  or not value then return nil end
	if value > hades_hunger.SAT_MAX then value = hades_hunger.SAT_MAX end
	if value < 0 then value = 0 end
	
	inv:set_stack("hunger", 1, ItemStack({name=":", count=value+1}))

	return true
end

minetest.register_on_joinplayer(function(player)
	local name = player:get_player_name()
	local inv = player:get_inventory()
	inv:set_size("hunger",1)
	hades_hunger.hunger[name] = hades_hunger.get_hunger_raw(player)
	hades_hunger.hunger_out[name] = hades_hunger.hunger[name]
	hades_hunger.exhaustion[name] = 0
	custom_hud(player)
	hades_hunger.set_hunger_raw(player)
end)

minetest.register_on_respawnplayer(function(player)
	-- reset hunger (and save)
	local name = player:get_player_name()
	hades_hunger.hunger[name] = hades_hunger.SAT_INIT
	hades_hunger.set_hunger_raw(player)
	hades_hunger.exhaustion[name] = 0
end)

local main_timer = 0
local timer = 0
local timer2 = 0
minetest.register_globalstep(function(dtime)
	main_timer = main_timer + dtime
	timer = timer + dtime
	timer2 = timer2 + dtime
	if main_timer > hades_hunger.HUD_TICK or timer >= hades_hunger.HEAL_DMG_TICK or timer2 > hades_hunger.HUNGER_TICK then
		if main_timer > hades_hunger.HUD_TICK then main_timer = 0 end
		for _,player in ipairs(minetest.get_connected_players()) do
		local name = player:get_player_name()

		local h = tonumber(hades_hunger.hunger[name])
		local hp = player:get_hp()
		if timer >= hades_hunger.HEAL_DMG_TICK then
			-- heal player by 1 hp if not dead and satiation is > hades_hunger.SAT_HEAL
			if h > hades_hunger.SAT_HEAL and hp > 0 and player:get_breath() > 0 then
				player:set_hp(hp+1, {type="set_hp", from="mod", _custom_reason="hunger_regenerate"})
				-- or damage player by 1 hp if satiation is <= 0
				elseif h <= 0 then
					if hp-1 >= 0 then
						hades_death_messages.player_damage(player, S("Starvation"))
						player:set_hp(hp-1, {type="set_hp", from="mod", _custom_reason="hunger"})
					end
				end
			end
			-- lower satiation by 1 point after xx seconds
			if timer2 > hades_hunger.HUNGER_TICK then
				if h > 0 then
					h = h-1
					hades_hunger.hunger[name] = h
					hades_hunger.set_hunger_raw(player)
				end
			end

			-- update all hud elements
			update_hud(player)
			
			local controls = player:get_player_control()

			-- Determine if the player is walking or jumping
			-- and increase exhaustion accordingly.
			-- Note: This only looks at the controls, not actual movement.
			-- So if the player keeps these controls pressed, they will
			-- get exhausted regardless of result.
			local exhaust = 0
			if controls.jump then
				exhaust = hades_hunger.EXHAUST_MOVE
			elseif controls.up or controls.down or controls.left or controls.right then
				if controls.sneak then
					exhaust = hades_hunger.EXHAUST_MOVE_SNEAK
				else
					exhaust = hades_hunger.EXHAUST_MOVE
				end
			end
			if exhaust > 0 then
				hades_hunger.exhaust(player, exhaust)
			end
		end
	end
	if timer > 4 then timer = 0 end
	if timer2 > hades_hunger.HUNGER_TICK then timer2 = 0 end
end)

minetest.register_chatcommand("satiation", {
	privs = {["server"]=true},
	params = S("[<player>] <satiation>"),
	description = S("Set satiation of player or yourself"),
	func = function(name, param)
		if minetest.settings:get_bool("enable_damage") == false then
			return false, S("Not possible, damage is disabled.")
		end
		local targetname, satiation = string.match(param, "(%S+) (%S+)")
		if not targetname then
			satiation = param
		end
		if not targetname then
			targetname = name
		end
		local target = minetest.get_player_by_name(targetname)
		if target == nil then
			return false, S("Player @1 does not exist.", targetname)
		end
		local current_satiation = hades_hunger.hunger[targetname]
		satiation = minetest.parse_relative_number(satiation, current_satiation)
		if not satiation then
			return false, S("Invalid satiation!")
		end
		if satiation > hades_hunger.SAT_MAX then
			satiation = hades_hunger.SAT_MAX
		elseif satiation < 0 then
			satiation = 0
		end
		hades_hunger.hunger[targetname] = satiation
		hades_hunger.set_hunger_raw(target)
		return true
	end,
})

else

-- Eating function if damage is disabled
minetest.do_item_eat = function(hp_change, replace_with_item, itemstack, user, pointed_thing)
	-- Check if the item's 'eatable' group is valid
	hades_hunger.check_eatable_group(itemstack, hp_change)

	local def = itemstack:get_definition()
	local sound
	if def and def.sound then
		sound = def.sound
	end

	if not sound then
		local item = itemstack:get_name()
		if minetest.get_item_group(item, "food") == 3 then
			sound = "survival_thirst_drink"
		elseif not sound then
			sound = "hbhunger_eat_generic"
		end
	end
	minetest.sound_play(
		{name = sound or "hbhunger_eat_generic",
		gain = 1},
		{object=user,
		max_hear_distance = 16,
		pitch = 1 + math.random(-10, 10)*0.005,},
		true
	)
	if hades_simulation.is_in_simulation(user) then
		return itemstack
	end
	local name = user and user:get_player_name() or ""
	if minetest.is_creative_enabled(name) then
		return itemstack
	end

	-- Remove eaten item
	local take = itemstack:take_item()
	if not take then
		return itemstack
	end
	if itemstack:get_count() == 0 then
		itemstack:add_item(replace_with_item)
	else
		local inv = user:get_inventory()
		if inv:room_for_item("main", replace_with_item) then
			inv:add_item("main", replace_with_item)
		else
			minetest.add_item(user:get_pos(), replace_with_item)
		end
	end
	return itemstack

end

end
