--
-- Editable books
--

--[[ Get translation function. The 'book' name is used
here instead of 'rp_book' due to legacy reasons,
so that old written books still have their
correct translation after the renaming orgy
after 1.5.3. If we switch to 'rp_book',
then the item tooltip of writen books would
lose their translation.
This could be solved if Minetest had a function like
minetest.register_lbm, but for items.
But the 'book' identifier works just fine for now,
it's just a minor deviation from convention. ]]
local S = minetest.get_translator("book")
local F = minetest.formspec_escape

local BOOK_MAX_TITLE_LENGTH = 64
local BOOK_MAX_TEXT_LENGTH = 4500

local on_use = function(itemstack, player, pointed_thing)
   local name = player:get_player_name()
   local data = itemstack:get_meta()

   local title = ""
   local text = ""

   if data then
      text = data:get_string("book:text")
      title = data:get_string("book:title")
   end

   local form = rp_formspec.get_page("rp_formspec:default")
   form = form .. "field[0.5,1.25;8,0;title;"..F(S("Title:"))..";"..F(title).."]"
   form = form .. "textarea[0.5,1.75;8,6.75;text;"..F(S("Contents:"))..";"..F(text).."]"
   form = form .. rp_formspec.button_exit(2.75, 7.75, 3, 1, "write", S("Write"))

   minetest.show_formspec(name, "rp_book:book", form)
end

-- Unwritten book (stackable)
minetest.register_craftitem(
   ":rp_default:book_empty",
   {
      description = S("Book"),
      _tt_help = S("Write down some notes"),
      inventory_image = "rp_book_book_empty.png",
      groups = { book = 2, tool = 1 },

      on_use = function(itemstack, player, pointed_thing)
         on_use(itemstack, player, pointed_thing)
      end,
})

-- Writable book (not stackable)
minetest.register_craftitem(
   ":rp_default:book",
   {
      description = S("Unnamed Book"),
      inventory_image = "default_book.png",
      stack_max = 1,
      groups = { book = 2, tool = 1, not_in_creative_inventory = 1 },

      on_use = function(itemstack, player, pointed_thing)
         on_use(itemstack, player, pointed_thing)
      end,
})

minetest.register_on_player_receive_fields(
   function(player, form_name, fields)
      if form_name ~= "rp_book:book" or (not fields.write and not fields.key_enter) then
         return
      end

      local wieldstack = player:get_wielded_item()
      local iname = wieldstack:get_name()
      if minetest.get_item_group(iname, "book") ~= 2 then
         return
      end

      local title = fields.title
      local text = fields.text

      -- Limit title and text length
      if string.len(title) > BOOK_MAX_TITLE_LENGTH then
          title = string.sub(title, 1, BOOK_MAX_TITLE_LENGTH)
      end
      if string.len(text) > BOOK_MAX_TEXT_LENGTH then
          text= string.sub(text, 1, BOOK_MAX_TEXT_LENGTH)
      end

      local function set_book_meta(item, title, text)
         local meta = item:get_meta()
	 if title ~= "" then
            meta:set_string("description", S("Book: “@1”", minetest.colorize("#ffff00", title))) -- Set the item description
         else
            meta:set_string("description", "")
         end
         meta:set_string("book:title", title)
         meta:set_string("book:text", text)
      end

      -- Revert book back to stackable book if empty contents
      if title == "" and text == "" then
         if wieldstack:get_name() == "rp_default:book" then
            wieldstack:set_name("rp_default:book_empty")
            set_book_meta(wieldstack, title, text)
            player:set_wielded_item(wieldstack)
            return
         end
      end

      -- Contents written: Update the player inventory
      if wieldstack:get_name() ~= "rp_default:book" then
         -- 1 book: Replace with written book
	 if wieldstack:get_count() == 1 then
            wieldstack:set_name("rp_default:book")
	    set_book_meta(wieldstack, title, text)
            player:set_wielded_item(wieldstack)
         -- Book stack
         else
	    local newstack = wieldstack:take_item()
	    newstack:set_count(1)
	    newstack:set_name("rp_default:book")
	    set_book_meta(newstack, title, text)
            local inv = player:get_inventory()
	    if inv:room_for_item("main", newstack) then
               -- If space available: Add written book in inventory
               inv:add_item("main", newstack)
               player:set_wielded_item(wieldstack)
	    else
               -- Inventory overflow: Drop the written book
               minetest.add_item(player:get_pos(), newstack)
               player:set_wielded_item(wieldstack)
	    end
	 end
      -- Update an existing written book
      else
	 set_book_meta(wieldstack, title, text)
         player:set_wielded_item(wieldstack)
      end
end)

minetest.register_craft(
   {
      type = "fuel",
      recipe = "rp_default:book_empty",
      burntime = 2,
})

minetest.register_craft(
   {
      type = "fuel",
      recipe = "rp_default:book",
      burntime = 2,
})

crafting.register_craft(
   {
      output = "rp_default:book_empty",
      items = {
         "rp_default:paper 3",
         "rp_default:stick",
         "rp_default:fiber",
      }
})

