-- This work is available under CC0; see the file LICENSE for details.

local core = core
local settings = core.settings

local name  -- Table index instead of entire ObjectRef.
local charge, operand, hud = {}, {}, {}

local hud_enabled = settings:get_bool("ejection.hud", true)
local whoosh = not settings:get_bool("ejection.nowhoosh", false)
local recharge = not settings:get_bool("ejection.norecharge", false)
local strength = settings:get("ejection.strength") or 256

local function update_name(player) name = player:get_player_name() end

local change
if hud_enabled then
	function change(player, key, val)
		player:hud_change(hud[name], key, val)
	end
else
	function change() end
end

core.register_on_joinplayer(function(player)
	update_name(player)
	operand[name] = 8
	charge[name] = 0
	if hud_enabled then
		-- The "Deprecated usage of statbar without size!" warning
		-- is unavoidable.
		hud[name] = player:hud_add({
			type = "statbar",
			name = "Ejection charge and cooldown scale",
			position = {x=0.5},
			offset = {x=-128},
			text = "ejection_chrg.png",
			text2 = "ejection_bg.png"
		})
	end
end)

core.register_globalstep(function()
	for _, player in ipairs(core.get_connected_players()) do
		update_name(player)

		if charge[name] < 0 then
			-- Negative charge means cooldown.
			charge[name] = charge[name] + 1
			if charge[name] < 0 then
				charge[name] = charge[name] + 1
			end
			if charge[name] >= 0 then
				if recharge then
					core.sound_play(
						"ejection_recharge",
						{
							object=player,
							max_hear_distance=512
						},
						true
					)
				end
				change(player, "item", 0)
			end
			change(player, "number", -charge[name])
		elseif player:get_player_control().aux1 then
			change(player, "item", 256)
			change(player, "text", "ejection_chrg.png")
			if charge[name] >= strength then
				operand[name] = -1
			elseif charge[name] <= 0 then
				operand[name] = 8
			end
			charge[name] = charge[name] + operand[name]
			change(player, "number", charge[name])
		elseif charge[name] > 0 then
			if whoosh then
				core.sound_play(
					"ejection_whoosh",
					{object=player, max_hear_distance=2048},
					true
				)
			end
			local vel = player:get_look_dir()
			vel = {
				x = vel.x * charge[name] / 8,
				y = vel.y * charge[name] / 8 + 4,
				z = vel.z * charge[name] / 8,
			}
			player:add_velocity(vel)
			charge[name] = -charge[name]
			change(player, "text", "ejection_cd.png")
		end
	end
end)
