-- This work is available under CC0; see the file LICENSE for details.

local core, tonumber = core, tonumber
local settings = core.settings

-- Common variables to share between functions instead of function arguments.
local name  -- Table index instead of entire ObjectRef.
local player

local charge, addend, hud = {}, {}, {}

local hud_enabled = settings:get_bool("ejection.hud", true)
local whoosh      = not settings:get_bool("ejection.nowhoosh", false)
local recharge    = not settings:get_bool("ejection.norecharge", false)
local strength    = tonumber(settings:get("ejection.strength")) or 256
local ctl = settings:get("ejection.control") or "aux1"

local function update_name(obj) name = obj:get_player_name() end

local change
if hud_enabled then
	function change(key, val)
		player:hud_change(hud[name], key, val)
	end
else
	function change() end
end

local function cooldown()
	charge[name] = charge[name] + 1
	if charge[name] < 0 then
		charge[name] = charge[name] + 1
	end
	if charge[name] >= 0 then
		if recharge then
			core.sound_play(
				"ejection_recharge",
				{object=player, max_hear_distance=512},
				true
			)
		end
		change("item", 0)
	end
	change("number", -charge[name])
end

local function charging()
	change("item", 256)
	change("text", "ejection_chrg.png")
	if charge[name] >= 256 then
		addend[name] = -1
	elseif charge[name] <= 0 then
		addend[name] = 8
	end
	charge[name] = charge[name] + addend[name]
	change("number", charge[name])
end

local function eject()
	if whoosh then
		core.sound_play(
			"ejection_whoosh",
			{object=player, max_hear_distance=2048},
			true
		)
	end
	local vel, factr = player:get_look_dir(), charge[name] * strength / 2048
	vel = {x=vel.x*factr, y=vel.y*factr+4, z=vel.z*factr}
	player:add_velocity(vel)
	charge[name] = -charge[name]
	change("text", "ejection_cd.png")
end

core.register_on_joinplayer(function(obj)
	update_name(obj)
	addend[name], charge[name] = 8, 0
	if hud_enabled then
		-- The "Deprecated usage of statbar without size!" warning
		-- is unavoidable.
		hud[name] = obj:hud_add({
			type = "statbar",
			name = "Ejection charge and cooldown scale",
			position = {x=0.5},
			offset = {x=-256},
			text = "ejection_chrg.png",
			text2 = "ejection_bg.png"
		})
	end
end)

core.register_globalstep(function()
	for _, obj in ipairs(core.get_connected_players()) do
		update_name(obj)
		player = obj

		-- Negative charge means cooldown.
		if     charge[name] < 0              then cooldown()
		elseif obj:get_player_control()[ctl] then charging()
		elseif charge[name] > 0              then eject()
		end
	end
end)

if ctl == "aux1" then
	core.dynamic_add_media({
		filename = "aux1_btn.png",
		filepath = core.get_modpath("ejection") ..
			"/textures/aux1.png"
	})
end
