-- This work is available under CC0; see the file LICENSE for details.

local core = core

local charge, addend, hud, armor, immunity = {}, {}, {}, {}, {}

local settings = core.settings

local hud_enabled = settings:get_bool("ejection.hud", true)
local whoosh      = not settings:get_bool("ejection.nowhoosh", false)
local recharge    = not settings:get_bool("ejection.norecharge", false)
local damage      = not settings:get_bool("ejection.immunity", false)
local strength    = tonumber(settings:get("ejection.strength")) or 256
local control     = settings:get("ejection.control") or "aux1"

local change, mk_hud, grant_immunity, deny_immunity

if hud_enabled then
	function change(key, val, player)
		player:hud_change(hud[player], key, val)
	end

	function mk_hud(player)
		-- The "Deprecated usage of statbar without size!" warning
		-- is unavoidable.
		hud[player] = player:hud_add({
			type = "statbar",
			size = 16,
			name = "Ejection charge and cooldown scale",
			position = {x=0.5, y=0},
			offset = {x=-256, y=0},
			text = "ejection_chrg.png",
			text2 = "ejection_bg.png"
		})
	end
else
	function change() end
	function mk_hud() end
end

if whoosh then
	function whoosh(player)
		core.sound_play(
			"ejection_whoosh",
			{object=player, max_hear_distance=2048},
			true
		)
	end
else
	function whoosh() end
end

if recharge then
	function recharge(player)
		core.sound_play(
			"ejection_recharge",
			{object=player, max_hear_distance=512},
			true
		)
	end
else
	function recharge() end
end

if damage then
	function grant_immunity() end
	function deny_immunity() end
else
	function grant_immunity(player)
		if not immunity[player] then
			armor[player] =
				player:get_armor_groups().fall_damage_add_percent or 0
			player:set_armor_groups({fall_damage_add_percent=-100})
		end
		immunity[player] = true
	end

	function deny_immunity(player)
		if immunity[player] and charge[player] == 0 then
			player:set_armor_groups({fall_damage_add_percent=armor[player]})
			immunity[player] = false
		end
	end
end

local function cooldown(player)
	charge[player] = charge[player] + 1
	if charge[player] < 0 then
		charge[player] = charge[player] + 1
	end

	if charge[player] >= 0 then
		core.after(
			2 / player:get_physics_override().gravity,
			deny_immunity,
			player
		)
		change("item", 0, player)
		recharge(player)
	end
	change("number", -charge[player], player)
end

local function charging(player)
	change("item", 256, player)
	change("text", "ejection_chrg.png", player)
	if charge[player] >= 256 then
		addend[player] = -1
	elseif charge[player] <= 0 then
		addend[player] = 8
	end
	charge[player] = charge[player] + addend[player]
	change("number", charge[player], player)
end

local function eject(player)
	grant_immunity(player)
	whoosh(player)

	local vel = player:get_look_dir()
	local factor = charge[player] * strength / 2048
	vel = {x=vel.x*factor, y=vel.y*factor+4, z=vel.z*factor}
	player:add_velocity(vel)

	charge[player] = -charge[player]
	change("text", "ejection_cd.png", player)
end

core.register_on_joinplayer(function(player)
	addend[player], charge[player] = 8, 0
	mk_hud(player)
end)

core.register_globalstep(function()
	for _, player in ipairs(core.get_connected_players()) do
		if charge[player] < 0 then
			cooldown(player)
		elseif player:get_player_control()[control] then
			charging(player)
		elseif charge[player] > 0 then
			eject(player)
		end
	end
end)

if control == "aux1" then
	core.dynamic_add_media({
		filename = "aux1_btn.png",
		filepath = core.get_modpath("ejection") ..
			"/textures/aux1.png"
	})
end
