local CAPACITIES = {
    ["splink:splink"]=1000,
    ["splink:big_splink"]=7500,
    ["splink:mega_splink"]=40000,
    ["splink:insane_splink"]=500000,
    ["splink:endless_splink"]=1e13,
}

function clamp(x, min, max)
    if x < min then return min end
    if x > max then return max end
    return x
end

local random = math.random
math.randomseed(os.time())
local function new_uuid()
    local template ='xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'
    return string.gsub(template, '[xy]', function (c)
        local v = (c == 'x') and random(0, 0xf) or random(8, 0xb)
        return string.format('%x', v)
    end)
end


local SUFFIXIES = {"K", "M", "B", "T", "q", "Q", "s", "S", "O", "N"}

function abberivate(number)
    if number < 0 then return "-" .. abberivate(-number) end
    if number < 10000 then return math.floor(number) end

    local powerOf1000 = math.floor(math.log(number) / math.log(1000))
    local mantissa = number / (1000 ^ powerOf1000)
    return sigfig(mantissa, 3) .. SUFFIXIES[powerOf1000]
end

function format_int(number)
    local i, j, minus, int, fraction = tostring(number):find('([-]?)(%d+)([.]?%d*)')
    int = int:reverse():gsub("(%d%d%d)", "%1,")
    return minus .. int:reverse():gsub("^,", "") .. fraction
  end
  
function sigfig(num, sigfig)
    local exponent = math.floor(math.log10(math.abs(num)))
    local multiplier = 10 ^ (sigfig - exponent - 1)
    return math.floor(num * multiplier) / multiplier
end

local function change_splink_count(splink, amount)
    local meta = splink:get_meta()
    local remaining = tonumber(meta:get_string("splink:count"))
    if remaining == nil then
        remaining = clamp(amount, 0, 1e13)
    else
        remaining = clamp(remaining + math.floor(amount), 0, 1e13)
    end
    meta:set_string("splink:count", tostring(remaining))
    meta:set_string("count_meta", abberivate(remaining))
    update_description(splink)
    return splink
end

function update_description(splink)
    local additional_description = ""

    local meta = splink:get_meta()
    local remaining = tonumber(meta:get_string("splink:count"))
    if remaining > 0 then 
        additional_description = additional_description .. "\n\n"
        additional_description = additional_description .. "Storing " .. meta:get_string("splink:type")
        if remaining >= 10000 then additional_description = additional_description .. "\n" .. format_int(remaining) ..  " items" end
    end
    meta:set_string("description", splink:get_short_description() .. additional_description)
end

minetest.register_on_joinplayer(function(player)
	local inv = player:get_inventory()
	inv:set_size("splink", 2) -- 9*9
end)

local function get_splink_hud(uuid)
    local formspec = {
        "formspec_version[8]",
        "size[12,7]",
        "list[detached:splink_" .. uuid .. ";main;5,0.5;2,1;]",
        "list[current_player;main;1,2;8,4;]",
        "listring[]",
    }

    return table.concat(formspec, "")
end

function get_splink_inv(splink)
    local meta = splink:get_meta()
    local uuid = meta:get_string("uuid")
    if uuid == "" then
        uuid = new_uuid()
        meta:set_string("uuid", uuid)
    end

    local inventory_name = "splink_" .. uuid
    local inv = core.get_inventory({
        type="detached", name=inventory_name})
    
    if inv then return inv
    else 
        inv = core.create_detached_inventory(inventory_name, {
            allow_move = function(inv, from_list, from_index, to_list, to_index, count, player)
                return 0 -- allow moving
            end,
        
            allow_put = function(inv, listname, index, stack, player)
                if stack:get_name() == "splink:splink" then return 0 end
                if stack:get_stack_max() == 1 then return 0 end
                local splink = player:get_wielded_item()
                local meta = splink:get_meta()
                if meta:get_string("splink:count") ~= "0" and meta:get_string("splink:type") ~= stack:get_name() then return 0 end

                local count = tonumber(meta:get_string("splink:count"))
                local amount_added = math.min(stack:get_count(), CAPACITIES[splink:get_name()] - count)

                return amount_added -- allow putting
            end,
        
            allow_take = function(inv, listname, index, stack, player)
                return stack:get_count()
            end,
        
            on_put = function(inv, listname, index, stack, player)
                local splink = player:get_wielded_item()
                local meta = splink:get_meta()
                local count = tonumber(meta:get_string("splink:count"))
                if count == 0 then
                    meta:set_string("splink:type", stack:get_name())
                end

                splink = change_splink_count(splink, stack:get_count())
                update_splink_inv(splink)
                player:set_wielded_item(splink)
            end,

            on_take = function(inv, listname, index, stack, player)
                local splink = player:get_wielded_item()
                splink = change_splink_count(splink, -stack:get_count())
                update_splink_inv(splink)
                player:set_wielded_item(splink)

            end,
        }) 
        inv:set_size("main", 2)
        return inv
    end
end

function update_splink_inv(splink)
    local meta = splink:get_meta()
    local stack_type = meta:get_string("splink:type") 
    
    local inv = get_splink_inv(splink)
    if stack_type == "" then return end

    local remaining = tonumber(meta:get_string("splink:count"))
    local item = ItemStack(meta:get_string("splink:type"))
    item:set_count(math.min(remaining, item:get_stack_max()))
    inv:set_stack("main", 1, item)
    inv:set_stack("main", 2, nil)

end

function is_entity_at_position(pos, range_check)
    if range_check == nil then range_check = 1 end
    for obj in core.objects_inside_radius(pos, range_check) do
        return true
    end
    return false
end

function safe_to_place_at(pointed_thing)
    local block_face = vector.subtract(pointed_thing.above, pointed_thing.under)
    
    local check_pos_1 = pointed_thing.above

    if  block_face.y ~= 1 then
        check_pos_2 = vector.add(pointed_thing.above, {x=0, y=-1, z=0})
    else
        check_pos_1 = pointed_thing.under
        return is_entity_at_position(check_pos_1)
    end
    return is_entity_at_position(check_pos_1) or is_entity_at_position(check_pos_2)
end

local function name_to_id(name)
    name = string.lower(name)
    name = string.gsub(name, " ", "_")
    return name
end

local function add_splink_tier(name, capacicty)
    local id = name_to_id(name)
    core.register_craftitem("splink:" .. id, {
        description = name,
        inventory_image = "splink_" .. id .. ".png",
        stack_max = 1,
        _capacity = capacicty,
        on_place = function(itemstack, placer, pointed_thing)
            local meta = itemstack:get_meta()
            local remaining = meta:get_string("splink:count")
            local block_type = meta:get_string("splink:type")

            if not core.registered_nodes[block_type] then return itemstack end
            if remaining ~= "0" then
                local block = core.get_node(pointed_thing.above)
                if safe_to_place_at(pointed_thing) then return itemstack end
                if block.name ~= "air" then return itemstack end
                core.set_node(pointed_thing.above, {name = meta:get_string("splink:type")})
    
                itemstack = change_splink_count(itemstack, -1)
            end
            return itemstack
        end,
        on_use = function(itemstack, user)
            local meta = itemstack:get_meta()
            -- local remaining = tonumber(meta:get_string("splink:count"))
            -- if remaining == 1e13 then
            --     itemstack = change_splink_count(itemstack, -1e13 + 50)
            -- elseif remaining == nil then
            --     itemstack = change_splink_count(itemstack, 1)
            -- else            
            --     itemstack = change_splink_count(itemstack, (math.floor(remaining / 2)) + 1)
            -- end
            change_splink_count(itemstack, 0)
            update_splink_inv(itemstack)
            core.show_formspec(user:get_player_name(), "splink:splink_hud", get_splink_hud(meta:get_string("uuid")))
            return itemstack
        end,
    })    
end

add_splink_tier("Splink")
add_splink_tier("Big Splink")
add_splink_tier("Mega Splink")
add_splink_tier("Insane Splink")
add_splink_tier("Endless Splink")